/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 */
'use strict';

function _slicedToArray(arr, i) {
  return (
    _arrayWithHoles(arr) ||
    _iterableToArrayLimit(arr, i) ||
    _unsupportedIterableToArray(arr, i) ||
    _nonIterableRest()
  );
}

function _nonIterableRest() {
  throw new TypeError(
    'Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.',
  );
}

function _unsupportedIterableToArray(o, minLen) {
  if (!o) return;
  if (typeof o === 'string') return _arrayLikeToArray(o, minLen);
  var n = Object.prototype.toString.call(o).slice(8, -1);
  if (n === 'Object' && o.constructor) n = o.constructor.name;
  if (n === 'Map' || n === 'Set') return Array.from(o);
  if (n === 'Arguments' || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n))
    return _arrayLikeToArray(o, minLen);
}

function _arrayLikeToArray(arr, len) {
  if (len == null || len > arr.length) len = arr.length;
  for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i];
  return arr2;
}

function _iterableToArrayLimit(arr, i) {
  var _i =
    arr &&
    ((typeof Symbol !== 'undefined' && arr[Symbol.iterator]) ||
      arr['@@iterator']);
  if (_i == null) return;
  var _arr = [];
  var _n = true;
  var _d = false;
  var _s, _e;
  try {
    for (_i = _i.call(arr); !(_n = (_s = _i.next()).done); _n = true) {
      _arr.push(_s.value);
      if (i && _arr.length === i) break;
    }
  } catch (err) {
    _d = true;
    _e = err;
  } finally {
    try {
      if (!_n && _i['return'] != null) _i['return']();
    } finally {
      if (_d) throw _e;
    }
  }
  return _arr;
}

function _arrayWithHoles(arr) {
  if (Array.isArray(arr)) return arr;
}

const _require = require('./Utils'),
  createAliasResolver = _require.createAliasResolver,
  getModules = _require.getModules;

const _require2 = require('../../parsers/flow/modules/utils'),
  unwrapNullable = _require2.unwrapNullable;

const HostFunctionTemplate = ({
  hasteModuleName,
  propertyName,
  jniSignature,
  jsReturnType,
}) => {
  return `static facebook::jsi::Value __hostFunction_${hasteModuleName}SpecJSI_${propertyName}(facebook::jsi::Runtime& rt, TurboModule &turboModule, const facebook::jsi::Value* args, size_t count) {
  return static_cast<JavaTurboModule &>(turboModule).invokeJavaMethod(rt, ${jsReturnType}, "${propertyName}", "${jniSignature}", args, count);
}`;
};

const ModuleClassConstructorTemplate = ({hasteModuleName, methods}) => {
  return `
${hasteModuleName}SpecJSI::${hasteModuleName}SpecJSI(const JavaTurboModule::InitParams &params)
  : JavaTurboModule(params) {
${methods
  .map(({propertyName, argCount}) => {
    return `  methodMap_["${propertyName}"] = MethodMetadata {${argCount}, __hostFunction_${hasteModuleName}SpecJSI_${propertyName}};`;
  })
  .join('\n')}
}`.trim();
};

const ModuleLookupTemplate = ({moduleName, hasteModuleName}) => {
  return `  if (moduleName == "${moduleName}") {
    return std::make_shared<${hasteModuleName}SpecJSI>(params);
  }`;
};

const FileTemplate = ({libraryName, include, modules, moduleLookups}) => {
  return `
/**
 * ${'C'}opyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * ${'@'}generated by codegen project: GenerateModuleJniCpp.js
 */

#include ${include}

namespace facebook {
namespace react {

${modules}

std::shared_ptr<TurboModule> ${libraryName}_ModuleProvider(const std::string moduleName, const JavaTurboModule::InitParams &params) {
${moduleLookups.map(ModuleLookupTemplate).join('\n')}
  return nullptr;
}

} // namespace react
} // namespace facebook
`;
};

function translateReturnTypeToKind(nullableTypeAnnotation, resolveAlias) {
  const _unwrapNullable = unwrapNullable(nullableTypeAnnotation),
    _unwrapNullable2 = _slicedToArray(_unwrapNullable, 1),
    typeAnnotation = _unwrapNullable2[0];

  let realTypeAnnotation = typeAnnotation;

  if (realTypeAnnotation.type === 'TypeAliasTypeAnnotation') {
    realTypeAnnotation = resolveAlias(realTypeAnnotation.name);
  }

  switch (realTypeAnnotation.type) {
    case 'ReservedTypeAnnotation':
      switch (realTypeAnnotation.name) {
        case 'RootTag':
          return 'NumberKind';

        default:
          realTypeAnnotation.name;
          throw new Error(
            `Invalid ReservedFunctionValueTypeName name, got ${realTypeAnnotation.name}`,
          );
      }

    case 'VoidTypeAnnotation':
      return 'VoidKind';

    case 'StringTypeAnnotation':
      return 'StringKind';

    case 'BooleanTypeAnnotation':
      return 'BooleanKind';

    case 'NumberTypeAnnotation':
      return 'NumberKind';

    case 'DoubleTypeAnnotation':
      return 'NumberKind';

    case 'FloatTypeAnnotation':
      return 'NumberKind';

    case 'Int32TypeAnnotation':
      return 'NumberKind';

    case 'PromiseTypeAnnotation':
      return 'PromiseKind';

    case 'GenericObjectTypeAnnotation':
      return 'ObjectKind';

    case 'ObjectTypeAnnotation':
      return 'ObjectKind';

    case 'ArrayTypeAnnotation':
      return 'ArrayKind';

    default:
      realTypeAnnotation.type;
      throw new Error(
        `Unknown prop type for returning value, found: ${realTypeAnnotation.type}"`,
      );
  }
}

function translateParamTypeToJniType(param, resolveAlias) {
  const optional = param.optional,
    nullableTypeAnnotation = param.typeAnnotation;

  const _unwrapNullable3 = unwrapNullable(nullableTypeAnnotation),
    _unwrapNullable4 = _slicedToArray(_unwrapNullable3, 2),
    typeAnnotation = _unwrapNullable4[0],
    nullable = _unwrapNullable4[1];

  const isRequired = !optional && !nullable;
  let realTypeAnnotation = typeAnnotation;

  if (realTypeAnnotation.type === 'TypeAliasTypeAnnotation') {
    realTypeAnnotation = resolveAlias(realTypeAnnotation.name);
  }

  switch (realTypeAnnotation.type) {
    case 'ReservedTypeAnnotation':
      switch (realTypeAnnotation.name) {
        case 'RootTag':
          return !isRequired ? 'Ljava/lang/Double;' : 'D';

        default:
          realTypeAnnotation.name;
          throw new Error(
            `Invalid ReservedFunctionValueTypeName name, got ${realTypeAnnotation.name}`,
          );
      }

    case 'StringTypeAnnotation':
      return 'Ljava/lang/String;';

    case 'BooleanTypeAnnotation':
      return !isRequired ? 'Ljava/lang/Boolean;' : 'Z';

    case 'NumberTypeAnnotation':
      return !isRequired ? 'Ljava/lang/Double;' : 'D';

    case 'DoubleTypeAnnotation':
      return !isRequired ? 'Ljava/lang/Double;' : 'D';

    case 'FloatTypeAnnotation':
      return !isRequired ? 'Ljava/lang/Double;' : 'D';

    case 'Int32TypeAnnotation':
      return !isRequired ? 'Ljava/lang/Double;' : 'D';

    case 'GenericObjectTypeAnnotation':
      return 'Lcom/facebook/react/bridge/ReadableMap;';

    case 'ObjectTypeAnnotation':
      return 'Lcom/facebook/react/bridge/ReadableMap;';

    case 'ArrayTypeAnnotation':
      return 'Lcom/facebook/react/bridge/ReadableArray;';

    case 'FunctionTypeAnnotation':
      return 'Lcom/facebook/react/bridge/Callback;';

    default:
      realTypeAnnotation.type;
      throw new Error(
        `Unknown prop type for method arg, found: ${realTypeAnnotation.type}"`,
      );
  }
}

function translateReturnTypeToJniType(nullableTypeAnnotation, resolveAlias) {
  const _unwrapNullable5 = unwrapNullable(nullableTypeAnnotation),
    _unwrapNullable6 = _slicedToArray(_unwrapNullable5, 2),
    typeAnnotation = _unwrapNullable6[0],
    nullable = _unwrapNullable6[1];

  let realTypeAnnotation = typeAnnotation;

  if (realTypeAnnotation.type === 'TypeAliasTypeAnnotation') {
    realTypeAnnotation = resolveAlias(realTypeAnnotation.name);
  }

  switch (realTypeAnnotation.type) {
    case 'ReservedTypeAnnotation':
      switch (realTypeAnnotation.name) {
        case 'RootTag':
          return nullable ? 'Ljava/lang/Double;' : 'D';

        default:
          realTypeAnnotation.name;
          throw new Error(
            `Invalid ReservedFunctionValueTypeName name, got ${realTypeAnnotation.name}`,
          );
      }

    case 'VoidTypeAnnotation':
      return 'V';

    case 'StringTypeAnnotation':
      return 'Ljava/lang/String;';

    case 'BooleanTypeAnnotation':
      return nullable ? 'Ljava/lang/Boolean;' : 'Z';

    case 'NumberTypeAnnotation':
      return nullable ? 'Ljava/lang/Double;' : 'D';

    case 'DoubleTypeAnnotation':
      return nullable ? 'Ljava/lang/Double;' : 'D';

    case 'FloatTypeAnnotation':
      return nullable ? 'Ljava/lang/Double;' : 'D';

    case 'Int32TypeAnnotation':
      return nullable ? 'Ljava/lang/Double;' : 'D';

    case 'PromiseTypeAnnotation':
      return 'Lcom/facebook/react/bridge/Promise;';

    case 'GenericObjectTypeAnnotation':
      return 'Lcom/facebook/react/bridge/WritableMap;';

    case 'ObjectTypeAnnotation':
      return 'Lcom/facebook/react/bridge/WritableMap;';

    case 'ArrayTypeAnnotation':
      return 'Lcom/facebook/react/bridge/WritableArray;';

    default:
      realTypeAnnotation.type;
      throw new Error(
        `Unknown prop type for method return type, found: ${realTypeAnnotation.type}"`,
      );
  }
}

function translateMethodTypeToJniSignature(property, resolveAlias) {
  const name = property.name,
    typeAnnotation = property.typeAnnotation;

  let _unwrapNullable7 = unwrapNullable(typeAnnotation),
    _unwrapNullable8 = _slicedToArray(_unwrapNullable7, 1),
    _unwrapNullable8$ = _unwrapNullable8[0],
    returnTypeAnnotation = _unwrapNullable8$.returnTypeAnnotation,
    params = _unwrapNullable8$.params;

  params = [...params];
  let processedReturnTypeAnnotation = returnTypeAnnotation;
  const isPromiseReturn = returnTypeAnnotation.type === 'PromiseTypeAnnotation';

  if (isPromiseReturn) {
    processedReturnTypeAnnotation = {
      type: 'VoidTypeAnnotation',
    };
  }

  const argsSignatureParts = params.map(t => {
    return translateParamTypeToJniType(t, resolveAlias);
  });

  if (isPromiseReturn) {
    // Additional promise arg for this case.
    argsSignatureParts.push(
      translateReturnTypeToJniType(returnTypeAnnotation, resolveAlias),
    );
  }

  const argsSignature = argsSignatureParts.join('');
  const returnSignature =
    name === 'getConstants'
      ? 'Ljava/util/Map;'
      : translateReturnTypeToJniType(
          processedReturnTypeAnnotation,
          resolveAlias,
        );
  return `(${argsSignature})${returnSignature}`;
}

function translateMethodForImplementation(
  hasteModuleName,
  property,
  resolveAlias,
) {
  const _unwrapNullable9 = unwrapNullable(property.typeAnnotation),
    _unwrapNullable10 = _slicedToArray(_unwrapNullable9, 1),
    propertyTypeAnnotation = _unwrapNullable10[0];

  const returnTypeAnnotation = propertyTypeAnnotation.returnTypeAnnotation;

  if (
    property.name === 'getConstants' &&
    returnTypeAnnotation.type === 'ObjectTypeAnnotation' &&
    returnTypeAnnotation.properties.length === 0
  ) {
    return '';
  }

  return HostFunctionTemplate({
    hasteModuleName,
    propertyName: property.name,
    jniSignature: translateMethodTypeToJniSignature(property, resolveAlias),
    jsReturnType: translateReturnTypeToKind(returnTypeAnnotation, resolveAlias),
  });
}

module.exports = {
  generate(libraryName, schema, packageName, assumeNonnull = false) {
    const nativeModules = getModules(schema);
    const modules = Object.keys(nativeModules)
      .filter(hasteModuleName => {
        const module = nativeModules[hasteModuleName];
        return !(
          module.excludedPlatforms != null &&
          module.excludedPlatforms.includes('android')
        );
      })
      .sort()
      .map(hasteModuleName => {
        const _nativeModules$hasteM = nativeModules[hasteModuleName],
          aliases = _nativeModules$hasteM.aliases,
          properties = _nativeModules$hasteM.spec.properties;
        const resolveAlias = createAliasResolver(aliases);
        const translatedMethods = properties
          .map(property =>
            translateMethodForImplementation(
              hasteModuleName,
              property,
              resolveAlias,
            ),
          )
          .join('\n\n');
        return (
          translatedMethods +
          '\n\n' +
          ModuleClassConstructorTemplate({
            hasteModuleName,
            methods: properties
              .map(({name: propertyName, typeAnnotation}) => {
                const _unwrapNullable11 = unwrapNullable(typeAnnotation),
                  _unwrapNullable12 = _slicedToArray(_unwrapNullable11, 1),
                  _unwrapNullable12$ = _unwrapNullable12[0],
                  returnTypeAnnotation =
                    _unwrapNullable12$.returnTypeAnnotation,
                  params = _unwrapNullable12$.params;

                if (
                  propertyName === 'getConstants' &&
                  returnTypeAnnotation.type === 'ObjectTypeAnnotation' &&
                  returnTypeAnnotation.properties &&
                  returnTypeAnnotation.properties.length === 0
                ) {
                  return null;
                }

                return {
                  propertyName,
                  argCount: params.length,
                };
              })
              .filter(Boolean),
          })
        );
      })
      .join('\n'); // $FlowFixMe[missing-type-arg]

    const moduleLookups = Object.keys(nativeModules)
      .filter(hasteModuleName => {
        const module = nativeModules[hasteModuleName];
        return !(
          module.excludedPlatforms != null &&
          module.excludedPlatforms.includes('android')
        );
      })
      .sort((a, b) => {
        const moduleA = nativeModules[a];
        const moduleB = nativeModules[b];
        const nameA = moduleA.moduleNames[0];
        const nameB = moduleB.moduleNames[0];

        if (nameA < nameB) {
          return -1;
        } else if (nameA > nameB) {
          return 1;
        }

        return 0;
      })
      .flatMap(hasteModuleName => {
        const moduleNames = nativeModules[hasteModuleName].moduleNames;
        return moduleNames.map(moduleName => ({
          moduleName,
          hasteModuleName,
        }));
      });
    const fileName = `${libraryName}-generated.cpp`;
    const replacedTemplate = FileTemplate({
      modules: modules,
      libraryName: libraryName.replace(/-/g, '_'),
      moduleLookups,
      include: `"${libraryName}.h"`,
    });
    return new Map([[`jni/${fileName}`, replacedTemplate]]);
  },
};
