/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 */
'use strict';

function upperCaseFirst(inString) {
  if (inString.length === 0) {
    return inString;
  }

  return inString[0].toUpperCase() + inString.slice(1);
}

function toSafeCppString(input) {
  return input
    .split('-')
    .map(upperCaseFirst)
    .join('');
}

function toIntEnumValueName(propName, value) {
  return `${toSafeCppString(propName)}${value}`;
}

function getCppTypeForAnnotation(type) {
  switch (type) {
    case 'BooleanTypeAnnotation':
      return 'bool';

    case 'StringTypeAnnotation':
      return 'std::string';

    case 'Int32TypeAnnotation':
      return 'int';

    case 'DoubleTypeAnnotation':
      return 'double';

    case 'FloatTypeAnnotation':
      return 'Float';

    default:
      type;
      throw new Error(`Received invalid typeAnnotation ${type}`);
  }
}

function getImports(properties) {
  const imports = new Set();

  function addImportsForNativeName(name) {
    switch (name) {
      case 'ColorPrimitive':
        return;

      case 'PointPrimitive':
        return;

      case 'EdgeInsetsPrimitive':
        return;

      case 'ImageSourcePrimitive':
        imports.add('#include <react/renderer/components/image/conversions.h>');
        return;

      default:
        name;
        throw new Error(`Invalid name, got ${name}`);
    }
  }

  properties.forEach(prop => {
    const typeAnnotation = prop.typeAnnotation;

    if (typeAnnotation.type === 'ReservedPropTypeAnnotation') {
      addImportsForNativeName(typeAnnotation.name);
    }

    if (
      typeAnnotation.type === 'ArrayTypeAnnotation' &&
      typeAnnotation.elementType.type === 'ReservedPropTypeAnnotation'
    ) {
      addImportsForNativeName(typeAnnotation.elementType.name);
    }

    if (typeAnnotation.type === 'ObjectTypeAnnotation') {
      const objectImports = getImports(typeAnnotation.properties); // $FlowFixMe[method-unbinding] added when improving typing for this parameters

      objectImports.forEach(imports.add, imports);
    }
  });
  return imports;
}

function generateEventStructName(parts = []) {
  const additional = parts.map(toSafeCppString).join('');
  return `${additional}`;
}

function generateStructName(componentName, parts = []) {
  const additional = parts.map(toSafeCppString).join('');
  return `${componentName}${additional}Struct`;
}

function getEnumName(componentName, propName) {
  const uppercasedPropName = toSafeCppString(propName);
  return `${componentName}${uppercasedPropName}`;
}

function getEnumMaskName(enumName) {
  return `${enumName}Mask`;
}

function convertDefaultTypeToString(componentName, prop) {
  const typeAnnotation = prop.typeAnnotation;

  switch (typeAnnotation.type) {
    case 'BooleanTypeAnnotation':
      if (typeAnnotation.default == null) {
        return '';
      }

      return String(typeAnnotation.default);

    case 'StringTypeAnnotation':
      if (typeAnnotation.default == null) {
        return '';
      }

      return `"${typeAnnotation.default}"`;

    case 'Int32TypeAnnotation':
      return String(typeAnnotation.default);

    case 'DoubleTypeAnnotation':
      const defaultDoubleVal = typeAnnotation.default;
      return parseInt(defaultDoubleVal, 10) === defaultDoubleVal
        ? typeAnnotation.default.toFixed(1)
        : String(typeAnnotation.default);

    case 'FloatTypeAnnotation':
      const defaultFloatVal = typeAnnotation.default;

      if (defaultFloatVal == null) {
        return '';
      }

      return parseInt(defaultFloatVal, 10) === defaultFloatVal
        ? defaultFloatVal.toFixed(1)
        : String(typeAnnotation.default);

    case 'ReservedPropTypeAnnotation':
      switch (typeAnnotation.name) {
        case 'ColorPrimitive':
          return '';

        case 'ImageSourcePrimitive':
          return '';

        case 'PointPrimitive':
          return '';

        case 'EdgeInsetsPrimitive':
          return '';

        default:
          typeAnnotation.name;
          throw new Error(
            `Unsupported type annotation: ${typeAnnotation.name}`,
          );
      }

    case 'ArrayTypeAnnotation': {
      const elementType = typeAnnotation.elementType;

      switch (elementType.type) {
        case 'StringEnumTypeAnnotation':
          if (elementType.default == null) {
            throw new Error(
              'A default is required for array StringEnumTypeAnnotation',
            );
          }

          const enumName = getEnumName(componentName, prop.name);
          const enumMaskName = getEnumMaskName(enumName);
          const defaultValue = `${enumName}::${toSafeCppString(
            elementType.default,
          )}`;
          return `static_cast<${enumMaskName}>(${defaultValue})`;

        default:
          return '';
      }
    }

    case 'ObjectTypeAnnotation': {
      return '';
    }

    case 'StringEnumTypeAnnotation':
      return `${getEnumName(componentName, prop.name)}::${toSafeCppString(
        typeAnnotation.default,
      )}`;

    case 'Int32EnumTypeAnnotation':
      return `${getEnumName(componentName, prop.name)}::${toIntEnumValueName(
        prop.name,
        typeAnnotation.default,
      )}`;

    default:
      typeAnnotation;
      throw new Error(`Unsupported type annotation: ${typeAnnotation.type}`);
  }
}

module.exports = {
  convertDefaultTypeToString,
  getCppTypeForAnnotation,
  getEnumName,
  getEnumMaskName,
  getImports,
  toSafeCppString,
  toIntEnumValueName,
  generateStructName,
  generateEventStructName,
};
