package com.starapps.tools.tnefextractor;

import androidx.appcompat.app.AppCompatActivity;
import androidx.documentfile.provider.DocumentFile;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ContentResolver;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.webkit.MimeTypeMap;
import android.widget.Toast;

import net.freeutils.tnef.Attachment;
import net.freeutils.tnef.Message;
import net.freeutils.tnef.TNEFInputStream;
import net.freeutils.tnef.TNEFUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

public class TNEFActivity extends AppCompatActivity {
	private static final int REQUEST_CODE_BROWSE_FOLDER = 1;
	private ArrayList<CharSequence> attachmentNames = new ArrayList<>();
	private ArrayList<Uri> attachmentFiles = new ArrayList<>();

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
			Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT_TREE);

			startActivityForResult(intent, REQUEST_CODE_BROWSE_FOLDER);
		} else {
			File folder = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS);

			if(folder != null) {
				extractTo(DocumentFile.fromFile(folder));
			}
		}
	}

	@Override
	public void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);

		if(resultCode != Activity.RESULT_OK || data == null || data.getData() == null) {
			finish();

			return;
		}

		if(requestCode == REQUEST_CODE_BROWSE_FOLDER) {
			extractTo(DocumentFile.fromTreeUri(getApplicationContext(), data.getData()));
		}
	}

	private void extractTo(DocumentFile folderFile) {
		Uri file = getFile();

		if(file != null) {
			InputStream inputStream = null;
			TNEFInputStream tnefInputStream = null;
			Message message = null;

			try {
				ContentResolver contentResolver = getApplicationContext().getContentResolver();
				inputStream = contentResolver.openInputStream(file);
				tnefInputStream = new TNEFInputStream(inputStream);
				message = new Message(tnefInputStream);

				extractAttachmentsTo(message, folderFile);

				AlertDialog.Builder builder = new AlertDialog.Builder(this);
				builder.setTitle(R.string.extraction_dialog_title)
						.setCancelable(true)
						.setOnCancelListener(new DialogInterface.OnCancelListener() {
							public void onCancel(DialogInterface dialogInterface) {
								finish();
							}
						})
						.setItems(attachmentNames.toArray(new CharSequence[0]), new DialogInterface.OnClickListener() {
							public void onClick(DialogInterface dialog, int which) {
								Intent intent = new Intent(Intent.ACTION_VIEW);
								intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
								intent.setDataAndType(attachmentFiles.get(which), getMimeType(attachmentNames.get(which).toString()));

								startActivity(intent);
								finish();
							}
						}).show();
			} catch(IOException e) {
				Toast.makeText(getApplicationContext(), getString(R.string.extraction_failed_all), Toast.LENGTH_LONG).show();
				finish();
			} finally {
				try {
					TNEFUtils.closeAll(message, tnefInputStream, inputStream);
				} catch(IOException ignored) {
				}
			}
		} else {
			finish();
		}
	}

	private void extractAttachmentsTo(Message message, DocumentFile folder) {
		ContentResolver contentResolver = getApplicationContext().getContentResolver();

		for(Attachment attachment : message.getAttachments()) {
			if(attachment.getNestedMessage() != null) {
				extractAttachmentsTo(attachment.getNestedMessage(), folder);
			} else {
				String filename = attachment.getFilename();

				if(filename != null) {
					filename = filename.replaceAll("(.*[/\\\\])*", "");

					try {
						DocumentFile file = folder.createFile(getMimeType(filename), filename);

						if(file != null) {
							attachment.writeTo(contentResolver.openOutputStream(file.getUri()));
							attachmentNames.add(filename);
							attachmentFiles.add(file.getUri());
						}
					} catch(IOException e) {
						Toast.makeText(getApplicationContext(), getString(R.string.extraction_failed_single, filename), Toast.LENGTH_LONG).show();
					}
				}
			}
		}
	}

	private Uri getFile() {
		String action = getIntent().getAction();

		if(action != null) {
			if(action.equals(Intent.ACTION_VIEW)) {
				return getIntent().getData();
			} else if(action.equals(Intent.ACTION_SEND)) {
				return getIntent().getParcelableExtra(Intent.EXTRA_STREAM);
			}
		}

		return null;
	}

	public static String getMimeType(String filename) {
		String type = null;
		String extension = MimeTypeMap.getFileExtensionFromUrl(filename);

		if(extension != null) {
			type = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
		}

		return type;
	}
}
