package com.example.util.simpletimetracker.feature_goals.viewModel

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.example.util.simpletimetracker.core.base.SingleLiveEvent
import com.example.util.simpletimetracker.core.extension.set
import com.example.util.simpletimetracker.core.interactor.StatisticsDetailNavigationInteractor
import com.example.util.simpletimetracker.core.model.NavigationTab
import com.example.util.simpletimetracker.feature_base_adapter.ViewHolderType
import com.example.util.simpletimetracker.feature_base_adapter.loader.LoaderViewData
import com.example.util.simpletimetracker.feature_base_adapter.statisticsGoal.StatisticsGoalViewData
import com.example.util.simpletimetracker.feature_goals.interactor.GoalsViewDataInteractor
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class GoalsViewModel @Inject constructor(
    private val goalsViewDataInteractor: GoalsViewDataInteractor,
    private val statisticsDetailNavigationInteractor: StatisticsDetailNavigationInteractor,
) : ViewModel() {

    val goals: LiveData<List<ViewHolderType>> by lazy {
        MutableLiveData(listOf(LoaderViewData() as ViewHolderType))
    }
    val resetScreen: SingleLiveEvent<Unit> = SingleLiveEvent()

    private var isVisible: Boolean = false
    private var timerJob: Job? = null

    fun onVisible() {
        isVisible = true
        startUpdate()
    }

    fun onHidden() {
        isVisible = false
        stopUpdate()
    }

    fun onTabReselected(tab: NavigationTab?) {
        if (isVisible && tab is NavigationTab.Goals) {
            resetScreen.set(Unit)
        }
    }

    fun onGoalClick(item: StatisticsGoalViewData) = viewModelScope.launch {
        statisticsDetailNavigationInteractor.navigateByGoal(
            goalId = item.id,
            shift = 0,
        )
    }

    private fun updateStatistics() = viewModelScope.launch {
        val data = loadStatisticsViewData()
        goals.set(data)
    }

    private suspend fun loadStatisticsViewData(): List<ViewHolderType> {
        return goalsViewDataInteractor.getViewData()
    }

    private fun startUpdate() {
        timerJob = viewModelScope.launch {
            timerJob?.cancelAndJoin()
            while (isActive) {
                updateStatistics()
                delay(TIMER_UPDATE)
            }
        }
    }

    private fun stopUpdate() {
        viewModelScope.launch {
            timerJob?.cancelAndJoin()
        }
    }

    companion object {
        private const val TIMER_UPDATE = 1000L
    }
}
