package com.prof18.feedflow.shared.presentation

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.prof18.feedflow.core.model.FeedSourceNotificationPreference
import com.prof18.feedflow.core.model.NotificationMode
import com.prof18.feedflow.core.model.NotificationSettingState
import com.prof18.feedflow.database.DatabaseHelper
import com.prof18.feedflow.shared.data.SettingsRepository
import com.prof18.feedflow.shared.domain.BackgroundSyncScheduler
import com.prof18.feedflow.shared.domain.model.SyncPeriod
import kotlinx.collections.immutable.persistentListOf
import kotlinx.collections.immutable.toImmutableList
import kotlinx.collections.immutable.toPersistentList
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

class NotificationsViewModel internal constructor(
    private val databaseHelper: DatabaseHelper,
    private val settingsRepository: SettingsRepository,
    private val backgroundSyncScheduler: BackgroundSyncScheduler,
) : ViewModel() {

    private val notificationSettingsMutableStateFlow = MutableStateFlow(
        NotificationSettingState(
            feedSources = persistentListOf<FeedSourceNotificationPreference>().toImmutableList(),
            isEnabledForAll = false,
            notificationMode = NotificationMode.FEED_SOURCE,
        ),
    )
    val notificationSettingState = notificationSettingsMutableStateFlow.asStateFlow()

    val syncPeriodFlow: StateFlow<SyncPeriod> = settingsRepository.syncPeriodFlow

    init {
        viewModelScope.launch {
            databaseHelper.getFeedSourcesFlow().collect { feedSources ->
                notificationSettingsMutableStateFlow.update {
                    NotificationSettingState(
                        feedSources = feedSources.map { feedSource ->
                            FeedSourceNotificationPreference(
                                feedSourceId = feedSource.id,
                                feedSourceTitle = feedSource.title,
                                isEnabled = feedSource.isNotificationEnabled,
                            )
                        }.toPersistentList(),
                        isEnabledForAll = if (feedSources.isEmpty()) {
                            false
                        } else {
                            feedSources.all { it.isNotificationEnabled }
                        },
                        notificationMode = settingsRepository.getNotificationMode(),
                    )
                }
            }
        }
    }

    fun updateAllNotificationStatus(status: Boolean) {
        viewModelScope.launch {
            databaseHelper.updateAllNotificationsEnabledStatus(enabled = status)
            notificationSettingsMutableStateFlow.update { oldValue ->
                oldValue.copy(
                    isEnabledForAll = status,
                )
            }
            if (status) {
                forceUpdateSyncPeriod()
            }
        }
    }

    fun updateNotificationStatus(status: Boolean, feedSourceId: String) {
        viewModelScope.launch {
            databaseHelper.updateNotificationEnabledStatus(
                feedSourceId = feedSourceId,
                enabled = status,
            )
            if (status) {
                forceUpdateSyncPeriod()
            }
        }
    }

    private fun forceUpdateSyncPeriod() {
        if (syncPeriodFlow.value == SyncPeriod.NEVER) {
            settingsRepository.setSyncPeriod(SyncPeriod.ONE_HOUR)
            backgroundSyncScheduler.updateSyncPeriod(SyncPeriod.ONE_HOUR)
        }
    }

    fun updateSyncPeriod(period: SyncPeriod) {
        viewModelScope.launch {
            settingsRepository.setSyncPeriod(period)
            backgroundSyncScheduler.updateSyncPeriod(period)
        }
    }

    fun updateNotificationMode(mode: NotificationMode) {
        viewModelScope.launch {
            settingsRepository.setNotificationMode(mode)
            notificationSettingsMutableStateFlow.update { oldValue ->
                oldValue.copy(
                    notificationMode = mode,
                )
            }
        }
    }
}
