import org.jetbrains.kotlin.gradle.dsl.JvmTarget
import java.util.*

plugins {
    alias(libs.plugins.android.application)
    alias(libs.plugins.kotlin.android)
    alias(libs.plugins.compose.multiplatform)
    alias(libs.plugins.compose.compiler)
    alias(libs.plugins.triplet.play)
    alias(libs.plugins.about.libraries.android)
    alias(libs.plugins.google.services)
    alias(libs.plugins.feedflow.detekt)
    alias(libs.plugins.kotlin.serialization)
}

apply(from = "../versioning.gradle.kts")

val appVersionCode: () -> Int by extra
val appVersionName: () -> String by extra

val local = Properties()
val localProperties: File = rootProject.file("keystore.properties")
if (localProperties.exists()) {
    localProperties.inputStream().use { local.load(it) }
}

val dropboxAppKey: String = local.getProperty("dropbox_key").orEmpty()
if (dropboxAppKey.isEmpty()) {
    println("Dropbox key not set in keystore.properties. Please add it to the file with the key 'dropbox_key'")
}

android {
    namespace = "com.prof18.feedflow.android"
    compileSdk = libs.versions.android.compile.sdk.get().toInt()
    defaultConfig {
        applicationId = "com.prof18.feedflow"
        minSdk = libs.versions.android.min.sdk.get().toInt()
        targetSdk = libs.versions.android.target.sdk.get().toInt()
        versionCode = 6089
        versionName = "1.9.0"

        addManifestPlaceholders(
            mapOf(
                "dropboxKey" to dropboxAppKey,
            ),
        )
    }

    buildFeatures {
        compose = true
        buildConfig = true
    }

    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_21
        targetCompatibility = JavaVersion.VERSION_21
    }


    buildTypes {
        getByName("debug") {
            applicationIdSuffix = ".debug"
            versionNameSuffix = "-debug"
            buildConfigField("String", "DROPBOX_APP_KEY", "\"$dropboxAppKey\"")
        }

        getByName("release") {
            isMinifyEnabled = true
            isShrinkResources = true
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro",
            )
            buildConfigField("String", "DROPBOX_APP_KEY", "\"$dropboxAppKey\"")
        }
    }

    flavorDimensions += "version"
    productFlavors {
        create("fdroid") {
            dimension = "version"
        }

        create("googlePlay") {
            dimension = "version"
            isDefault = true
        }
    }

    aboutLibraries {
        export {
            excludeFields.set(listOf("generated"))
        }
    }

    packaging {
        resources {
            excludes += setOf(
                "META-INF/DEPENDENCIES",
                "META-INF/LICENSE",
                "META-INF/LICENSE.txt",
                "META-INF/license.txt",
                "META-INF/NOTICE",
                "META-INF/NOTICE.txt",
                "META-INF/notice.txt",
                "META-INF/ASL2.0",
                "META-INF/INDEX.LIST",
            )
        }
    }
}

tasks.withType<org.jetbrains.kotlin.gradle.tasks.KotlinCompile>().configureEach {
    compilerOptions {
        freeCompilerArgs = listOf(
            "-opt-in=androidx.compose.material3.ExperimentalMaterial3Api",
            "-opt-in=androidx.compose.material3.ExperimentalMaterial3ExpressiveApi",
            "-opt-in=androidx.compose.animation.ExperimentalAnimationApi",
            "-opt-in=co.touchlab.kermit.ExperimentalKermitApi",
            "-opt-in=kotlin.ExperimentalStdlibApi",
        )
        jvmTarget.set(JvmTarget.JVM_21)
    }
}

dependencies {
    implementation(project(":shared"))
    implementation(project(":sharedUI"))
 
    implementation(libs.compose.multiplatform.runtime)
    implementation(libs.compose.multiplatform.foundation)
    implementation(libs.compose.material3)
    implementation(libs.compose.multiplatform.material.icons.extended)
    implementation(libs.compose.multiplatform.ui)
    implementation(libs.compose.multiplatform.ui.tooling)
    implementation(libs.compose.multiplatform.material3.adaptive.navigationsuite)
 
    implementation(libs.androidx.activity.compose)
    implementation(libs.androidx.lifecycle.viewModel.compose)
    implementation(libs.androidx.lifecycle.runtime.compose)
    implementation(libs.androidx.navigation3.runtime)
    implementation(libs.androidx.navigation3.ui)
    implementation(libs.androidx.lifecycle.viewmodel.navigation3)

    implementation(libs.bundles.about.libraries)

    implementation(libs.androidx.browser)
    implementation(libs.compose.webview)
    implementation(libs.colorpicker.compose)
    implementation(libs.telephoto.zoomable)

    implementation(platform(libs.koin.bom))
    implementation(libs.koin.core)
    implementation(libs.koin.android)
    implementation(libs.koin.workmanager)
    implementation(libs.koin.composeVM)

    implementation(libs.dropbox.core.android)
    implementation(libs.workmanager)
    implementation(libs.androidx.lifecycle.process)
    implementation(libs.kotlinx.serialization.json)
    implementation(libs.glance.appwidget)
    implementation(libs.glance.material)

    "googlePlayImplementation"(platform(libs.firebase.bom))
    "googlePlayImplementation"(libs.touchlab.kermit.crash)
    "googlePlayImplementation"(libs.play.review)
    "googlePlayImplementation"(libs.telemetry.deck)
    "googlePlayImplementation"(libs.google.identity.googleid)
    "googlePlayImplementation"(libs.google.play.services.auth)
    "googlePlayImplementation"(libs.kotlinx.coroutines.play.services)
    "googlePlayImplementation"(libs.google.api.client.android)
    "googlePlayImplementation"(libs.google.api.services.drive)

    debugImplementation(libs.compose.multiplatform.ui.tooling)

    testImplementation(libs.koin.test)
}

play {
    // The play_config.json file will be provided on CI
    serviceAccountCredentials.set(file("../play_config.json"))
    track.set("internal")
}

android.applicationVariants.configureEach {
    val name = name.replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.getDefault()) else it.toString() }
    val googleTask = tasks.findByName("process${name}GoogleServices")
    val uploadTask = tasks.findByName("uploadCrashlyticsMappingFile$name")
    googleTask?.enabled = !name.contains("Fdroid")
    uploadTask?.enabled = !name.contains("Fdroid")
}
