/**
 * Parse timestamp from value
 *
 * Supported formats must match {@link PATTERN_TIMESTAMP}
 *
 * @param value Value to parse timestamp from. If number, assumes already a timestamp in seconds.
 * @returns Parsed timestamp in seconds
 * @throws {TypeError} When value is not a string
 * @throws {TypeError} When value does not match {@link PATTERN_TIMESTAMP}
 */
export declare const parseTimestamp: (value: string | number) => number;
/**
 * Timestamp formatter
 */
export interface FormatterCallback {
    (timestamp: number): string;
}
/**
 * Provides a way to convert numeric timestamp to a formatted string.
 *
 * A custom formatter may be registered.
 * If one isn't registered, the default formatter will be used and the data will be formatted as HH:mm:SS.fff
 */
export declare class TimestampFormatterManager {
    static _instance: TimestampFormatterManager;
    private customFormatter;
    /**
     * Create the formatter
     */
    constructor();
    /**
     * Default formatter where the timestamp number is formatted to a human readable string in the format HH:mm:SS.fff
     *
     * @param timestamp Time in seconds to format
     * @returns formatted timestamp string
     */
    private static defaultFormatter;
    /**
     * Format the timestamp to a human readable string.
     *
     * If a custom formatter has been registered, it will be used.
     * If not, the default formatter is used and the data will be returned in the format HH:mm:SS.fff
     *
     * @param timestamp Time in seconds to format
     * @returns formatted timestamp string
     */
    format: (timestamp: number) => string;
    /**
     * Register a custom timestamp formatter
     *
     * @param formatter function to call to format timestamp to string
     */
    registerCustomFormatter: (formatter: FormatterCallback) => void;
    /**
     * Remove the custom formatter (if one registered)
     */
    unregisterCustomFormatter: () => void;
}
export declare const TimestampFormatter: TimestampFormatterManager;
