"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addSegment = void 0;
const options_1 = require("./options");
const timestamp_1 = require("./timestamp");
const types_1 = require("./types");
/**
 * Regular Expression for detecting punctuation that should not be prefixed with a space
 */
const PATTERN_PUNCTUATIONS = /^ *[.,?!}\]>) *$]/;
/**
 * Append `addition` to `body` with the character(s) specified.
 *
 * If `addition` matches the {@link PATTERN_PUNCTUATIONS} pattern, no character is added before the additional data.
 *
 * @param body Current body text
 * @param addition Additional text to add to `body`
 * @param separator Character(s) to use to separate data. If undefined, uses `\n`.
 * @returns Combined data
 */
const joinBody = (body, addition, separator = undefined) => {
    if (body) {
        let separatorToUse = separator || "\n";
        if (PATTERN_PUNCTUATIONS.exec(addition)) {
            separatorToUse = "";
        }
        return `${body}${separatorToUse}${addition}`;
    }
    return addition;
};
/**
 * Combine one or more {@link Segment}
 *
 * @param segments Array of Segment objects to combine
 * @param bodySeparator Character(s) to use to separate body data. If undefined, uses `\n`.
 * @returns Combined segment where:
 *
 *   - `startTime`: from first segment
 *   - `startTimeFormatted`: from first segment
 *   - `endTime`: from last segment
 *   - `endTimeFormatted`: from last segment
 *   - `speaker`: from first segment
 *   - `body`: combination of all segments
 */
const joinSegments = (segments, bodySeparator = undefined) => {
    const newSegment = { ...segments[0] };
    segments.slice(1).forEach((segment) => {
        newSegment.endTime = segment.endTime;
        newSegment.endTimeFormatted = timestamp_1.TimestampFormatter.format(segment.endTime);
        newSegment.body = joinBody(newSegment.body, segment.body, bodySeparator);
    });
    return newSegment;
};
/**
 * Checks if the new and prior segments have the same speaker.
 *
 * If so, combines segments where:
 *   - `startTime`: from priorSegment
 *   - `startTimeFormatted`: from priorSegment
 *   - `endTime`: from newSegment
 *   - `endTimeFormatted`: from newSegment
 *   - `speaker`: from priorSegment
 *   - `body`: body of priorSegment with body of newSegment separated with space
 *
 * @param newSegment segment being created
 * @param priorSegment prior parsed segment
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns result of combination.
 * If segments were combined, {@link CombineResult.replace} and {@link CombineResult.combined} set to true.
 */
const doCombineSpeaker = (newSegment, priorSegment, lastSpeaker) => {
    if (newSegment.speaker === priorSegment.speaker || newSegment.speaker === lastSpeaker) {
        return {
            segment: joinSegments([priorSegment, newSegment], " "),
            replace: true,
            combined: true,
        };
    }
    return {
        segment: newSegment,
        replace: false,
        combined: false,
    };
};
/**
 * Checks if the new and prior segments have the same speaker and combining body results in new body shorter than
 * max length
 *
 * If so, combines segments where:
 *   - `startTime`: from priorSegment
 *   - `startTimeFormatted`: from priorSegment
 *   - `endTime`: from newSegment
 *   - `endTimeFormatted`: from newSegment
 *   - `speaker`: from priorSegment
 *   - `body`: body of priorSegment with body of newSegment separated with space
 *
 * @param newSegment segment being created
 * @param priorSegment prior parsed segment
 * @param maxLength maximum allowed length of combined body. If undefined, uses {@link DEFAULT_COMBINE_SEGMENTS_LENGTH}
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns result of combination.
 * If segments were combined, {@link CombineResult.replace} and {@link CombineResult.combined} set to true.
 */
const doCombineSegments = (newSegment, priorSegment, maxLength, lastSpeaker) => {
    if (priorSegment === undefined) {
        return {
            segment: newSegment,
            replace: false,
            combined: false,
        };
    }
    const combineSegmentsLength = maxLength || types_1.DEFAULT_COMBINE_SEGMENTS_LENGTH;
    if ((newSegment.speaker === priorSegment.speaker || newSegment.speaker === lastSpeaker) &&
        joinBody(priorSegment.body, newSegment.body, " ").length <= combineSegmentsLength) {
        return {
            segment: joinSegments([priorSegment, newSegment], " "),
            replace: true,
            combined: true,
        };
    }
    return {
        segment: newSegment,
        replace: false,
        combined: false,
    };
};
/**
 * Checks if the new and prior segments have the same speaker, startTime and endTime.
 *
 * If so, combines segments where:
 *   - `startTime`: from priorSegment
 *   - `startTimeFormatted`: from priorSegment
 *   - `endTime`: from newSegment
 *   - `endTimeFormatted`: from newSegment
 *   - `speaker`: from priorSegment
 *   - `body`: body of priorSegment with body of newSegment separated with value of separator argument
 *
 * @param newSegment segment being created
 * @param priorSegment prior parsed segment
 * @param separator string to use to combine body values. If undefined, uses "\n"
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns result of combination.
 * If segments were combined, {@link CombineResult.replace} and {@link CombineResult.combined} set to true.
 */
const doCombineEqualTimes = (newSegment, priorSegment, separator, lastSpeaker) => {
    const combineEqualTimesSeparator = separator || "\n";
    if (newSegment.startTime === priorSegment.startTime &&
        newSegment.endTime === priorSegment.endTime &&
        (newSegment.speaker === priorSegment.speaker || newSegment.speaker === lastSpeaker)) {
        return {
            segment: joinSegments([priorSegment, newSegment], combineEqualTimesSeparator),
            replace: true,
            combined: true,
        };
    }
    return {
        segment: newSegment,
        replace: false,
        combined: false,
    };
};
/**
 * Checks if the new and prior segments have the same speaker. If so, sets the speaker value to undefined
 *
 * If so, combines segments where:
 *   - `startTime`: from priorSegment
 *   - `startTimeFormatted`: from priorSegment
 *   - `endTime`: from newSegment
 *   - `endTimeFormatted`: from newSegment
 *   - `speaker`: from newSegment if different from priorSegment else undefined
 *   - `body`: body of priorSegment with body of newSegment separated with space
 *
 * @param newSegment segment being created
 * @param priorSegment prior parsed segment. For the first segment, this shall be undefined.
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns result of combination.
 * If segments were combined, {@link CombineResult.replace} set to false and {@link CombineResult.combined} set to true.
 */
const doSpeakerChange = (newSegment, priorSegment, lastSpeaker) => {
    const result = {
        segment: newSegment,
        replace: false,
        combined: false,
    };
    if (priorSegment === undefined) {
        if (newSegment.speaker === lastSpeaker) {
            const segment = { ...newSegment };
            segment.speaker = undefined;
            return {
                segment,
                replace: false,
                combined: true,
            };
        }
        return result;
    }
    if (newSegment.speaker === undefined) {
        return result;
    }
    if (newSegment.speaker === "" ||
        newSegment.speaker === priorSegment.speaker ||
        newSegment.speaker === lastSpeaker) {
        const segment = { ...newSegment };
        segment.speaker = undefined;
        return {
            segment,
            replace: false,
            combined: true,
        };
    }
    return result;
};
/**
 * Determine how {@link Options.speakerChange is applied based an past options being applied}
 *
 * @param currentResult current result object from any prior options
 * @param priorSegment prior parsed segment
 * @param lastSpeaker last speaker name.
 * @returns result of combination.
 * If segments were combined, {@link CombineResult.replace} set to false and {@link CombineResult.combined} set to true.
 */
const applyOptionsAndDoSpeakerChange = (currentResult, priorSegment, lastSpeaker) => {
    const { combineSegments, combineEqualTimes } = options_1.Options;
    let result = currentResult;
    if (combineSegments && currentResult.combined && currentResult.replace) {
        result = doSpeakerChange(currentResult.segment, undefined, undefined);
    }
    else if (combineEqualTimes) {
        if (result.combined && result.replace) {
            result = doSpeakerChange(currentResult.segment, undefined, undefined);
        }
        else {
            result = doSpeakerChange(currentResult.segment, priorSegment, lastSpeaker);
        }
    }
    else {
        result = doSpeakerChange(currentResult.segment, priorSegment, lastSpeaker);
    }
    if (result) {
        result = {
            segment: result.segment,
            replace: currentResult.replace || result.replace,
            combined: currentResult.combined || result.combined,
        };
    }
    return result;
};
/**
 * Apply convert rules when no prior segment exits.
 *
 * NOTE: not all rules applicable when no prior segment
 *
 * @param newSegment segment before any rules options to it
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns the updated segment info
 */
const doCombineNoPrior = (newSegment, lastSpeaker) => {
    const { speakerChange } = options_1.Options;
    let result = {
        segment: newSegment,
        replace: false,
        combined: false,
    };
    if (speakerChange) {
        result = doSpeakerChange(result.segment, undefined, lastSpeaker);
    }
    return result;
};
/**
 * Apply convert rules when prior segment exits.
 *
 * @param newSegment segment before any rules options to it
 * @param priorSegment prior parsed segment
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns the updated segment info
 */
const doCombineWithPrior = (newSegment, priorSegment, lastSpeaker) => {
    const { combineEqualTimes, combineEqualTimesSeparator, combineSegments, combineSegmentsLength, combineSpeaker, speakerChange, } = options_1.Options;
    let result = {
        segment: { ...newSegment },
        replace: false,
        combined: false,
    };
    let combinedSpeaker = false;
    if (combineSpeaker) {
        result = doCombineSpeaker(result.segment, priorSegment, lastSpeaker);
        combinedSpeaker = result.combined;
    }
    if (!combinedSpeaker && combineEqualTimes) {
        result = doCombineEqualTimes(result.segment, priorSegment, combineEqualTimesSeparator, lastSpeaker);
    }
    if (!combinedSpeaker && combineSegments) {
        let combineResult;
        if (combineEqualTimes && result.combined) {
            combineResult = doCombineSegments(result.segment, undefined, combineSegmentsLength, lastSpeaker);
        }
        else {
            combineResult = doCombineSegments(result.segment, priorSegment, combineSegmentsLength, lastSpeaker);
        }
        if (combineResult) {
            result = {
                segment: combineResult.segment,
                replace: result.replace || combineResult.replace,
                combined: result.combined || combineResult.combined,
            };
        }
    }
    if (speakerChange) {
        result = applyOptionsAndDoSpeakerChange(result, priorSegment, lastSpeaker);
    }
    return result;
};
/**
 * Apply any options to the current segment
 *
 * @param newSegment segment before any rules options to it
 * @param priorSegment prior parsed segment. For the first segment, this shall be undefined.
 * @param lastSpeaker last speaker name.
 * Used when speaker in segment has been removed via {@link Options.speakerChange} rule
 * @returns the updated segment info
 */
const applyOptions = (newSegment, priorSegment, lastSpeaker = undefined) => {
    if (!options_1.Options.optionsSet()) {
        return {
            segment: newSegment,
            replace: false,
            combined: false,
        };
    }
    let result;
    // if no prior segment, limited additional checking
    if (priorSegment === undefined) {
        result = doCombineNoPrior(newSegment, lastSpeaker);
    }
    else {
        result = doCombineWithPrior(newSegment, priorSegment, lastSpeaker);
    }
    return result;
};
/**
 * Get the last speaker name from the previously parsed segments
 *
 * @param priorSegment prior parsed segment
 * @param priorSegments array of all previous segments
 * @returns the name of the last speaker
 */
const getLastSpeaker = (priorSegment, priorSegments) => {
    let lastSpeaker;
    if (priorSegment) {
        lastSpeaker = priorSegment.speaker;
    }
    if (lastSpeaker === undefined && priorSegments.length > 0) {
        lastSpeaker = priorSegments[0].speaker;
        for (let i = priorSegments.length - 1; i > 0; i--) {
            if (priorSegments[i].speaker !== undefined) {
                lastSpeaker = priorSegments[i].speaker;
                break;
            }
        }
    }
    return lastSpeaker;
};
/**
 * Helper for adding segment to or updating last segment in array of segments
 *
 * @param newSegment segment to add or replace
 * @param priorSegments array of all previous segments
 * @returns updated array of segments with new segment added or last segment updated (per options)
 */
const addSegment = (newSegment, priorSegments) => {
    const { speakerChange } = options_1.Options;
    const outSegments = priorSegments || [];
    const priorSegment = outSegments.length > 0 ? outSegments[outSegments.length - 1] : undefined;
    // don't worry about identifying the last speaker if speaker is not being removed by speakerChange
    let lastSpeaker;
    if (speakerChange) {
        lastSpeaker = getLastSpeaker(priorSegment, outSegments);
    }
    const newSegmentInfo = applyOptions(newSegment, priorSegment, lastSpeaker);
    if (newSegmentInfo.replace && outSegments.length > 0) {
        outSegments[outSegments.length - 1] = newSegmentInfo.segment;
    }
    else {
        outSegments.push(newSegmentInfo.segment);
    }
    return outSegments;
};
exports.addSegment = addSegment;
