"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertFile = exports.determineFormat = exports.Options = exports.TimestampFormatter = exports.TranscriptFormat = void 0;
const html_1 = require("./formats/html");
const json_1 = require("./formats/json");
const srt_1 = require("./formats/srt");
const vtt_1 = require("./formats/vtt");
const types_1 = require("./types");
var types_2 = require("./types");
Object.defineProperty(exports, "TranscriptFormat", { enumerable: true, get: function () { return types_2.TranscriptFormat; } });
var timestamp_1 = require("./timestamp");
Object.defineProperty(exports, "TimestampFormatter", { enumerable: true, get: function () { return timestamp_1.TimestampFormatter; } });
var options_1 = require("./options");
Object.defineProperty(exports, "Options", { enumerable: true, get: function () { return options_1.Options; } });
/**
 * Determines the format of transcript by inspecting the data
 *
 * @param data The transcript data
 * @returns The determined transcript format
 * @throws {TypeError} Cannot determine format of data or error parsing data
 */
const determineFormat = (data) => {
    const normalizedData = data.trim();
    if ((0, vtt_1.isVTT)(normalizedData)) {
        return types_1.TranscriptFormat.VTT;
    }
    if ((0, json_1.isJSON)(normalizedData)) {
        return types_1.TranscriptFormat.JSON;
    }
    if ((0, html_1.isHTML)(normalizedData)) {
        return types_1.TranscriptFormat.HTML;
    }
    if ((0, srt_1.isSRT)(normalizedData)) {
        return types_1.TranscriptFormat.SRT;
    }
    throw new TypeError(`Cannot determine format for data`);
};
exports.determineFormat = determineFormat;
/**
 * Convert the data to an Array of {@link Segment}
 *
 * @param data The transcript data
 * @param transcriptFormat The format of the data.
 * @returns An Array of Segment objects from the parsed data
 * @throws {TypeError} When `transcriptFormat` is unknown
 */
const convertFile = (data, transcriptFormat = undefined) => {
    const format = transcriptFormat ?? (0, exports.determineFormat)(data);
    const normalizedData = data.trimStart();
    let outSegments = [];
    switch (format) {
        case types_1.TranscriptFormat.HTML:
            outSegments = (0, html_1.parseHTML)(normalizedData);
            break;
        case types_1.TranscriptFormat.JSON:
            outSegments = (0, json_1.parseJSON)(normalizedData);
            break;
        case types_1.TranscriptFormat.SRT:
            outSegments = (0, srt_1.parseSRT)(normalizedData);
            break;
        case types_1.TranscriptFormat.VTT:
            outSegments = (0, vtt_1.parseVTT)(normalizedData);
            break;
        default:
            throw new TypeError(`Unknown transcript format: ${format}`);
    }
    return outSegments;
};
exports.convertFile = convertFile;
