"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseSRT = exports.isSRT = exports.parseSRTSegment = void 0;
const segments_1 = require("../segments");
const speaker_1 = require("../speaker");
const timestamp_1 = require("../timestamp");
const types_1 = require("../types");
/**
 * Parse lines looking for data to be SRT format
 *
 * @param lines Lines containing SRT data
 * @returns Parsed segment
 * @throws {Error} When no non-empty strings in `lines`
 * @throws {Error} When the minimum required number of lines is not received
 * @throws {Error} When segment lines does not start with a number
 * @throws {Error} When second segment line does not follow the timestamp format
 */
const parseSRTSegment = (lines) => {
    do {
        if (lines.length === 0) {
            throw new Error("SRT segment lines empty");
        }
        else if (lines[0].trim() === "") {
            lines.shift();
        }
        else {
            break;
        }
    } while (lines.length > 0);
    if (lines.length < 3) {
        throw new Error(`SRT requires at least 3 lines, ${lines.length} received`);
    }
    const index = parseInt(lines[0], 10);
    if (!index) {
        throw new Error(`First line of SRT segment is not a number`);
    }
    const timestampLine = lines[1];
    if (!timestampLine.includes("-->")) {
        throw new Error(`SRT timestamp line does not include --> separator`);
    }
    const timestampParts = timestampLine.split("-->");
    if (timestampParts.length !== 2) {
        throw new Error(`SRT timestamp line contains more than 2 --> separators`);
    }
    const startTime = (0, timestamp_1.parseTimestamp)(timestampParts[0].trim());
    const endTime = (0, timestamp_1.parseTimestamp)(timestampParts[1].trim());
    let bodyLines = lines.slice(2);
    const emptyLineIndex = bodyLines.findIndex((v) => v.trim() === "");
    if (emptyLineIndex > 0) {
        bodyLines = bodyLines.slice(0, emptyLineIndex);
    }
    const { speaker, message } = (0, speaker_1.parseSpeaker)(bodyLines.shift());
    bodyLines = [message].concat(bodyLines);
    return {
        startTime,
        endTime,
        speaker,
        body: bodyLines.join("\n"),
        index,
    };
};
exports.parseSRTSegment = parseSRTSegment;
/**
 * Create Segment from lines containing an SRT segment/cue
 *
 * @param segmentLines Lines containing SRT data
 * @param lastSpeaker Name of last speaker. Will be used if no speaker found in `segmentLines`
 * @returns Created segment
 */
const createSegmentFromSRTLines = (segmentLines, lastSpeaker) => {
    const srtSegment = (0, exports.parseSRTSegment)(segmentLines);
    const calculatedSpeaker = srtSegment.speaker ? srtSegment.speaker : lastSpeaker;
    return {
        startTime: srtSegment.startTime,
        startTimeFormatted: timestamp_1.TimestampFormatter.format(srtSegment.startTime),
        endTime: srtSegment.endTime,
        endTimeFormatted: timestamp_1.TimestampFormatter.format(srtSegment.endTime),
        speaker: calculatedSpeaker,
        body: srtSegment.body,
    };
};
/**
 * Determines if the value of data is a valid SRT transcript format
 *
 * @param data The transcript data
 * @returns True: data is valid SRT transcript format
 */
const isSRT = (data) => {
    try {
        return (0, exports.parseSRTSegment)(data.split(types_1.PATTERN_LINE_SEPARATOR).slice(0, 20)) !== undefined;
    }
    catch (e) {
        return false;
    }
};
exports.isSRT = isSRT;
/**
 * Parse SRT data to an Array of {@link Segment}
 *
 * @param data The transcript data
 * @returns An array of Segments from the parsed data
 * @throws {TypeError} When `data` is not valid SRT format
 */
const parseSRT = (data) => {
    if (!(0, exports.isSRT)(data)) {
        throw new TypeError(`Data is not valid SRT format`);
    }
    let outSegments = [];
    let lastSpeaker = "";
    let segmentLines = [];
    data.split(types_1.PATTERN_LINE_SEPARATOR).forEach((line, count) => {
        // separator line found, handle previous data
        if (line.trim() === "") {
            // handle consecutive multiple blank lines
            if (segmentLines.length !== 0) {
                try {
                    outSegments = (0, segments_1.addSegment)(createSegmentFromSRTLines(segmentLines, lastSpeaker), outSegments);
                    lastSpeaker = outSegments[outSegments.length - 1].speaker;
                }
                catch (e) {
                    console.error(`Error parsing SRT segment lines (source line ${count}): ${e}`);
                    console.error(segmentLines);
                }
            }
            segmentLines = []; // clear buffer
        }
        else {
            segmentLines.push(line);
        }
    });
    // handle data when trailing line not included
    if (segmentLines.length !== 0) {
        try {
            outSegments = (0, segments_1.addSegment)(createSegmentFromSRTLines(segmentLines, lastSpeaker), outSegments);
            lastSpeaker = outSegments[outSegments.length - 1].speaker;
        }
        catch (e) {
            console.error(`Error parsing final SRT segment lines: ${e}`);
            console.error(segmentLines);
        }
    }
    return outSegments;
};
exports.parseSRT = parseSRT;
