"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseTranscriptFile = exports.getTranscriptForLocale = void 0;
var htmlHelpers_1 = require("./htmlHelpers");
var timeHelpers_1 = require("./timeHelpers");
var convertJSONSRTItemToTranscriptRow = function (item, line) {
    var body = item.body, endTime = item.endTime, startTime = item.startTime, speaker = item.speaker;
    if (!startTime && startTime !== 0)
        return null;
    var startTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(startTime);
    if (!endTime && endTime !== 0)
        return null;
    var endTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(endTime);
    return {
        line: line,
        startTime: startTime,
        startTimeFormatted: startTimeFormatted,
        endTime: endTime,
        endTimeFormatted: endTimeFormatted,
        speaker: speaker,
        body: body
    };
};
var convertParsedHTMLItemToTranscriptRow = function (item, line) {
    /*
      item[0] = full parsed as single line
      item[1] = speaker
      item[2] = start time
      item[3] = html body
    */
    var speaker = item[1];
    var startTime = convertTranscriptTimestampToSeconds(item[2]);
    if (!startTime && startTime !== 0)
        return null;
    var startTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(startTime);
    var body = (0, htmlHelpers_1.decodeHTMLString)(item[3]);
    return {
        line: line,
        startTime: startTime,
        startTimeFormatted: startTimeFormatted,
        speaker: speaker,
        body: body
    };
};
var convertParsedSRTItemToTranscriptRow = function (item) {
    /*
      item[0] = full parsed as single line
      item[1] = line
      item[2] = start time
      item[3] = end time
      item[4] = speaker
      item[5] = speaker
      item[6] = body line 1
      item[7] = body line 2
    */
    var line = parseInt(item[1], 10);
    var startTime = convertTranscriptTimestampToSeconds(item[2]);
    if (!startTime && startTime !== 0)
        return null;
    var startTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(startTime);
    var endTime = convertTranscriptTimestampToSeconds(item[3]);
    if (!endTime && endTime !== 0)
        return null;
    var endTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(endTime);
    var speaker = item[4];
    var body = item[6];
    if (item[7])
        body += " ".concat(item[7]);
    return {
        line: line,
        startTime: startTime,
        startTimeFormatted: startTimeFormatted,
        endTime: endTime,
        endTimeFormatted: endTimeFormatted,
        speaker: speaker,
        body: body
    };
};
var convertParsedVTTItemToTranscriptRow = function (item, line) {
    /*
      item[0] = full parsed as single line
      item[1] = start time
      item[2] = end time
      item[3] = speaker
      item[4] = speaker
      item[5] = body line 1
      item[6] = body line 2
    */
    var startTime = convertTranscriptTimestampToSeconds(item[1]);
    if (!startTime && startTime !== 0)
        return null;
    var startTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(startTime);
    var endTime = convertTranscriptTimestampToSeconds(item[2]);
    if (!endTime && endTime !== 0)
        return null;
    var endTimeFormatted = (0, timeHelpers_1.convertSecToHHMMSS)(endTime);
    var speaker = item[3];
    var body = item[5];
    if (item[6])
        body += " ".concat(item[6]);
    return {
        line: line,
        startTime: startTime,
        startTimeFormatted: startTimeFormatted,
        endTime: endTime,
        endTimeFormatted: endTimeFormatted,
        speaker: speaker,
        body: body
    };
};
var convertTranscriptTimestampToSeconds = function (timestamp) {
    // SRT time stamps use this formatting: 00:02:45,170
    var hhmmss = timestamp.split(',')[0];
    // VTT time stamps use this formatting: 00:02:45.170
    hhmmss = timestamp.split('.')[0];
    return (0, timeHelpers_1.convertHHMMSSToSeconds)(hhmmss);
};
var getTranscriptForLocale = function (transcripts, locale) {
    var transcript = null;
    if ((transcripts === null || transcripts === void 0 ? void 0 : transcripts.length) > 0 && locale) {
        transcript = transcripts.find(function (t) { return t.language === locale; });
        if (!transcript) {
            var baseLanguageArray = locale.split('-');
            var baseLanguage_1 = baseLanguageArray[0];
            transcript = transcripts.find(function (t) { var _a; return (_a = t.language) === null || _a === void 0 ? void 0 : _a.startsWith(baseLanguage_1); });
        }
        if (!transcript) {
            transcript = transcripts[0];
        }
    }
    return transcript;
};
exports.getTranscriptForLocale = getTranscriptForLocale;
var parseHTMLFile = function (data) {
    data = (data && data.trim()) || '';
    // eslint-disable-next-line max-len
    var pattern = /\x3ccite\x3e(.+?)\x3a\x3c\x2fcite\x3e\n\s{1,}?\x3ctime\x3e([0-9\x3a\x2c]{1,12})\x3c\x2ftime\x3e\n\s{1,}?\x3cp\x3e(.+?)\x3c\x2fp\x3e/gim;
    var matches;
    var result = [];
    data = data.replace(/\r\n|\r|\n/g, '\n');
    var index = 0;
    while ((matches = pattern.exec(data)) !== null) {
        var item = convertParsedHTMLItemToTranscriptRow(matches, index);
        index++;
        if (item)
            result.push(item);
    }
    /*
      If a text/html file does not have parsable fields, but it has contents,
      then convert it into a single row that starts at 0 seconds.
    */
    if (result.length === 0 && data) {
        var speaker = '';
        var startTime = '00:00:00,000';
        var index_1 = 0;
        var strippedHtml = (0, htmlHelpers_1.removeHTMLFromString)(data);
        var htmlItem = convertParsedHTMLItemToTranscriptRow([
            data,
            speaker,
            startTime,
            strippedHtml
        ], index_1);
        if (htmlItem)
            result.push(htmlItem);
    }
    return result;
};
var parseJSONFile = function (data) {
    var segments = data.segments;
    var result = [];
    if (Array.isArray(segments)) {
        for (var i = 0; i < segments.length; i++) {
            var item = convertJSONSRTItemToTranscriptRow(segments[i], i);
            if (item)
                result.push(item);
        }
    }
    return result;
};
var parseSRTFile = function (data) {
    var pattern = /(\d{1,})\n([0-9\x3a\x2c]{12})\s\x2d\x2d\x3e\s([0-9\x3a\x2c]{12})\n((.+?)\x3a\s)?(.*)\n(.*)\n/gim;
    var matches;
    var result = [];
    data = data.replace(/\r\n|\r|\n/g, '\n');
    while ((matches = pattern.exec(data)) !== null) {
        var item = convertParsedSRTItemToTranscriptRow(matches);
        if (item)
            result.push(item);
    }
    return result;
};
var parseTranscriptFile = function (data, transcriptType) {
    if (!data)
        return [];
    var parsedTranscript = [];
    if (transcriptType === 'application/json') {
        if (typeof data === 'string') {
            try {
                data = JSON.parse(data);
            }
            catch (error) {
                //
            }
        }
        parsedTranscript = parseJSONFile(data);
    }
    else if (transcriptType === 'application/srt' || transcriptType === 'text/srt' || transcriptType === 'application/x-subrip') {
        parsedTranscript = parseSRTFile(data);
    }
    else if (transcriptType === 'text/html') {
        // parseHTMLFile isn't working for at least this RSS feed https://feeds.buzzsprout.com/1.rss
        parsedTranscript = parseHTMLFile(data);
    }
    else if (transcriptType === 'text/vtt') {
        parsedTranscript = parseVTTFile(data);
    }
    return parsedTranscript;
};
exports.parseTranscriptFile = parseTranscriptFile;
var parseVTTFile = function (data) {
    var pattern = /([0-9\x3a\x2e]{12})\s\x2d\x2d\x3e\s([0-9\x3a\x2e]{12})\n((.+?)\x3a\s)?(.*)\n(.*)\n\n/gim;
    var matches;
    var result = [];
    data = data.replace(/\r\n|\r|\n/g, '\n');
    var index = 0;
    while ((matches = pattern.exec(data)) !== null) {
        var item = convertParsedVTTItemToTranscriptRow(matches, index);
        index++;
        if (item)
            result.push(item);
    }
    return result;
};
