"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.soundbite = exports.chapters = exports.funding = exports.transcript = exports.TranscriptType = exports.locked = void 0;
/* eslint-disable @typescript-eslint/no-unsafe-member-access */
/* eslint-disable @typescript-eslint/no-unsafe-assignment */
const shared_1 = require("../shared");
const logger_1 = require("../../logger");
/**
 * https://github.com/Podcastindex-org/podcast-namespace/blob/main/docs/1.0.md#locked
 *
 * This tag may be set to yes or no. The purpose is to tell other podcast platforms whether they are
 * allowed to import this feed. A value of yes means that any attempt to import this feed into a new
 * platform should be rejected.
 */
exports.locked = {
    phase: 1,
    tag: "podcast:locked",
    name: "locked",
    nodeTransform: shared_1.firstIfArray,
    supportCheck: (node) => Boolean(shared_1.getAttribute(node, "owner")),
    fn(node) {
        const feedUpdate = {};
        const lockedText = shared_1.getText(node).toLowerCase();
        const owner = shared_1.getAttribute(node, "owner");
        logger_1.logger.debug(`- Owner: ${owner !== null && owner !== void 0 ? owner : ""}`);
        logger_1.logger.debug(`- Locked: ${lockedText}`);
        if (["yes", "true"].includes(lockedText)) {
            feedUpdate.locked = true;
        }
        else if (["no", "false"].includes(lockedText)) {
            feedUpdate.locked = false;
        }
        if (owner) {
            feedUpdate.podcastOwner = owner;
        }
        return feedUpdate;
    },
};
var TranscriptType;
(function (TranscriptType) {
    TranscriptType["Plain"] = "text/plain";
    TranscriptType["HTML"] = "text/html";
    TranscriptType["SRT"] = "application/srt";
    TranscriptType["JSON"] = "application/json";
    TranscriptType["UNKNOWN"] = "unknown";
})(TranscriptType = exports.TranscriptType || (exports.TranscriptType = {}));
function deriveMimeType(typeAttribute) {
    switch (typeAttribute) {
        case "application/srt":
        case "application/x-subrip":
        case "text/srt":
            return TranscriptType.SRT;
        case "text/plain":
            return TranscriptType.Plain;
        case "text/html":
            return TranscriptType.HTML;
        case "application/json":
            return TranscriptType.JSON;
        default:
            console.warn("Unexpected transcript type", typeAttribute);
            console.warn(" Please open an issue - https://github.com/RyanHirsch/partytime/issues");
            return TranscriptType.UNKNOWN;
    }
}
exports.transcript = {
    phase: 1,
    tag: "podcast:transcript",
    name: "transcript",
    nodeTransform: shared_1.ensureArray,
    supportCheck: (node) => node.some((transcriptNode) => Boolean(shared_1.getAttribute(transcriptNode, "url")) &&
        Boolean(shared_1.getAttribute(transcriptNode, "type"))),
    fn(node, feed) {
        const itemUpdate = { podcastTranscripts: [] };
        node.forEach((transcriptNode) => {
            const feedLanguage = feed ? feed.rss.channel.language : null;
            const url = shared_1.getAttribute(transcriptNode, "url");
            const type = deriveMimeType(shared_1.getAttribute(transcriptNode, "type"));
            const language = shared_1.getAttribute(transcriptNode, "language") || feedLanguage;
            const rel = shared_1.getAttribute(transcriptNode, "rel");
            logger_1.logger.debug(`- Feed Language: ${feedLanguage}`);
            logger_1.logger.debug(`- URL: ${url !== null && url !== void 0 ? url : "<null>"}`);
            logger_1.logger.debug(`- Type: ${type}`);
            logger_1.logger.debug(`- Language: ${language}`);
            logger_1.logger.debug(`- Rel: ${rel !== null && rel !== void 0 ? rel : "<null>"}`);
            if (url && type) {
                const transcriptValue = {
                    url,
                    type,
                };
                if (language) {
                    transcriptValue.language = language;
                }
                if (rel) {
                    transcriptValue.rel = "captions";
                }
                itemUpdate.podcastTranscripts.push(transcriptValue);
            }
        });
        return itemUpdate;
    },
};
exports.funding = {
    phase: 1,
    tag: "podcast:funding",
    name: "funding",
    nodeTransform: shared_1.ensureArray,
    supportCheck: (node) => Boolean(node.find((x) => shared_1.getAttribute(x, "url"))),
    fn(node) {
        return {
            podcastFunding: node
                .map((n) => {
                const message = shared_1.getText(n);
                const url = shared_1.getAttribute(n, "url");
                if (url) {
                    return {
                        message: message !== null && message !== void 0 ? message : "",
                        url,
                    };
                }
                return undefined;
            })
                .filter(Boolean),
        };
    },
};
exports.chapters = {
    phase: 1,
    tag: "podcast:chapters",
    name: "chapters",
    nodeTransform: shared_1.firstIfArray,
    supportCheck: (node) => Boolean(shared_1.getAttribute(node, "url")) && Boolean(shared_1.getAttribute(node, "type")),
    fn(node) {
        return {
            podcastChapters: {
                url: shared_1.getKnownAttribute(node, "url"),
                type: shared_1.getKnownAttribute(node, "type"),
            },
        };
    },
};
exports.soundbite = {
    phase: 1,
    tag: "podcast:soundbite",
    name: "soundbite",
    nodeTransform: shared_1.ensureArray,
    supportCheck: (node) => node.some((n) => shared_1.getAttribute(n, "duration") && shared_1.getAttribute(n, "startTime")),
    fn(node, feed) {
        const itemUpdate = { podcastSoundbites: [] };
        node.forEach((soundbiteNode) => {
            var _a;
            const duration = parseFloat(shared_1.getKnownAttribute(soundbiteNode, "duration"));
            const startTime = parseFloat(shared_1.getKnownAttribute(soundbiteNode, "startTime"));
            const title = shared_1.getText(soundbiteNode);
            if (duration && startTime) {
                const bite = {
                    duration,
                    startTime,
                    title: title || ((_a = feed.rss.channel.title) !== null && _a !== void 0 ? _a : "").trim(),
                };
                itemUpdate.podcastSoundbites.push(bite);
            }
        });
        return itemUpdate;
    },
};
//# sourceMappingURL=phase-1.js.map