/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * @format
 */
'use strict';

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var HermesParserDecodeUTF8String = require('./HermesParserDecodeUTF8String');

var NODE_DESERIALIZERS = require('./HermesParserNodeDeserializers');

var HermesParserDeserializer = /*#__PURE__*/function () {
  function HermesParserDeserializer(programBuffer, positionBuffer, positionBufferSize, wasmParser, options) {
    _classCallCheck(this, HermesParserDeserializer);

    _defineProperty(this, "commentTypes", ['CommentLine', 'CommentBlock', 'InterpreterDirective']);

    _defineProperty(this, "tokenTypes", ['Boolean', 'Identifier', 'Keyword', 'Null', 'Numeric', 'Punctuator', 'String', 'RegularExpression', 'Template', 'JSXText']);

    // Program and position buffer are memory addresses, so we must convert
    // into indices into HEAPU32 (an array of 4-byte integers).
    this.programBufferIdx = programBuffer / 4;
    this.positionBufferIdx = positionBuffer / 4;
    this.positionBufferSize = positionBufferSize;
    this.locMap = {};
    this.HEAPU8 = wasmParser.HEAPU8;
    this.HEAPU32 = wasmParser.HEAPU32;
    this.HEAPF64 = wasmParser.HEAPF64;
    this.options = options;
  }
  /**
   * Consume and return the next 4 bytes in the program buffer.
   */


  _createClass(HermesParserDeserializer, [{
    key: "next",
    value: function next() {
      var num = this.HEAPU32[this.programBufferIdx++];
      return num;
    }
  }, {
    key: "deserialize",
    value: function deserialize() {
      var program = {
        type: 'Program',
        loc: this.addEmptyLoc(),
        body: this.deserializeNodeList(),
        comments: this.deserializeComments()
      };

      if (this.options.tokens) {
        program.tokens = this.deserializeTokens();
      }

      this.fillLocs();
      return program;
    }
    /**
     * Booleans are serialized as a single 4-byte integer.
     */

  }, {
    key: "deserializeBoolean",
    value: function deserializeBoolean() {
      return Boolean(this.next());
    }
    /**
     * Numbers are serialized directly into program buffer, taking up 8 bytes
     * preceded by 4 bytes of alignment padding if necessary.
     */

  }, {
    key: "deserializeNumber",
    value: function deserializeNumber() {
      var floatIdx; // Numbers are aligned on 8-byte boundaries, so skip padding if we are at
      // an odd index into the 4-byte aligned program buffer.

      if (this.programBufferIdx % 2 === 0) {
        floatIdx = this.programBufferIdx / 2;
        this.programBufferIdx += 2;
      } else {
        floatIdx = (this.programBufferIdx + 1) / 2;
        this.programBufferIdx += 3;
      }

      return this.HEAPF64[floatIdx];
    }
    /**
     * Strings are serialized as a 4-byte pointer into the heap, followed
     * by their size as a 4-byte integer. The size is only present if the
     * pointer is non-null.
     */

  }, {
    key: "deserializeString",
    value: function deserializeString() {
      var ptr = this.next();

      if (ptr === 0) {
        return null;
      }

      var size = this.next();
      return HermesParserDecodeUTF8String(ptr, size, this.HEAPU8);
    }
    /**
     * Nodes are serialized as a 4-byte integer denoting their node kind,
     * followed by a 4-byte loc ID, followed by serialized node properties.
     *
     * If the node kind is 0 the node is null, otherwise the node kind - 1 is an
     * index into the array of node deserialization functions.
     */

  }, {
    key: "deserializeNode",
    value: function deserializeNode() {
      var nodeType = this.next();

      if (nodeType === 0) {
        return null;
      }

      var nodeDeserializer = NODE_DESERIALIZERS[nodeType - 1].bind(this);
      return nodeDeserializer();
    }
    /**
     * Node lists are serialized as a 4-byte integer denoting the number of
     * elements in the list, followed by the serialized elements.
     */

  }, {
    key: "deserializeNodeList",
    value: function deserializeNodeList() {
      var size = this.next();
      var nodeList = [];

      for (var i = 0; i < size; i++) {
        nodeList.push(this.deserializeNode());
      }

      return nodeList;
    } // Matches StoredComment::Kind enum in JSLexer.h

  }, {
    key: "deserializeComments",

    /**
     * Comments are serialized as a node list, where each comment is serialized
     * as a 4-byte integer denoting comment type, followed by a 4-byte value
     * denoting the loc ID, followed by a serialized string for the comment value.
     */
    value: function deserializeComments() {
      var size = this.next();
      var comments = [];

      for (var i = 0; i < size; i++) {
        var commentType = this.commentTypes[this.next()];
        var loc = this.addEmptyLoc();
        var value = this.deserializeString();
        comments.push({
          type: commentType,
          loc: loc,
          value: value
        });
      }

      return comments;
    }
  }, {
    key: "deserializeTokens",
    value: function deserializeTokens() {
      var size = this.next();
      var tokens = [];

      for (var i = 0; i < size; i++) {
        var tokenType = this.tokenTypes[this.next()];
        var loc = this.addEmptyLoc();
        var value = this.deserializeString();
        tokens.push({
          type: tokenType,
          loc: loc,
          value: value
        });
      }

      return tokens;
    }
    /**
     * While deserializing the AST locations are represented by
     * a 4-byte loc ID. This is used to create a map of loc IDs to empty loc
     * objects that are filled after the AST has been deserialized.
     */

  }, {
    key: "addEmptyLoc",
    value: function addEmptyLoc() {
      var loc = {};
      this.locMap[this.next()] = loc;
      return loc;
    }
    /**
     * Positions are serialized as a loc ID which denotes which loc it is associated with,
     * followed by kind which denotes whether it is a start or end position,
     * followed by line, column, and offset (4-bytes each).
     */

  }, {
    key: "fillLocs",
    value: function fillLocs() {
      for (var i = 0; i < this.positionBufferSize; i++) {
        var locId = this.HEAPU32[this.positionBufferIdx++];
        var kind = this.HEAPU32[this.positionBufferIdx++];
        var line = this.HEAPU32[this.positionBufferIdx++];
        var column = this.HEAPU32[this.positionBufferIdx++];
        var offset = this.HEAPU32[this.positionBufferIdx++];
        var loc = this.locMap[locId];

        if (kind === 0) {
          loc.start = {
            line: line,
            column: column
          };
          loc.rangeStart = offset;
        } else {
          loc.end = {
            line: line,
            column: column
          };
          loc.rangeEnd = offset;
        }
      }
    }
  }]);

  return HermesParserDeserializer;
}();

module.exports = HermesParserDeserializer;