/*
 * Copyright 2013 Josei [hi at jrubio dot me]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.memetro.android.ui.register;
import android.content.Intent;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Base64;
import android.util.Log;
import android.view.View;
import android.widget.*;

import androidx.annotation.Nullable;
import androidx.lifecycle.ViewModelProvider;
import com.bumptech.glide.Glide;
import com.memetro.android.R;
import com.memetro.android.databinding.ActivityRegisterCredentialsBinding;
import com.memetro.android.api.ResultState;
import com.memetro.android.api.captcha.models.CaptchaDataModel;
import com.memetro.android.api.login.models.LoginResponseModel;
import com.memetro.android.api.signup.models.SignupResponseModel;
import com.memetro.android.api.sync.models.City;
import com.memetro.android.api.sync.models.Country;
import com.memetro.android.api.sync.models.StaticDataResponseModel;
import com.memetro.android.ui.BaseActivity;
import com.memetro.android.ui.common.LayoutUtils;

import com.memetro.android.ui.common.MemetroDialog;
import com.memetro.android.ui.common.MemetroProgress;
import com.memetro.android.ui.dashboard.DashboardActivity;
import com.memetro.android.ui.login.LoginActivity;
import dagger.hilt.android.AndroidEntryPoint;

import java.util.Objects;

import static com.memetro.android.ui.register.CredentialsViewModel.PASS_MIN_LENGTH;

@AndroidEntryPoint
public class CredentialsActivity extends BaseActivity<ActivityRegisterCredentialsBinding> {

    private static String TAG = "Memetro Register Credentials";
    private CredentialsViewModel viewModel;
    private MemetroProgress pdialog;

    private City city;
    private Country country;

    private String captchaKey;

    @Override
    protected void setUpViews(@Nullable @org.jetbrains.annotations.Nullable Bundle savedInstanceState) {
        pdialog = new MemetroProgress(this);
        binding.register.setOnClickListener(view -> {
            if(!Objects.equals(viewModel.passwordState.getValue(), CredentialsViewModel.PasswordStatus.OK)) {
                String message = view.getRootView().getContext(). getString(R.string.error_register) + " \n" + binding.passwordStatusMessage.getText().toString();
                MemetroDialog.showDialog(CredentialsActivity.this, null, message);
            }
            else if (viewModel.emailStatus.getValue() == null || !viewModel.emailStatus.getValue()) {
                String message = view.getRootView().getContext(). getString(R.string.error_register_valid_mail) ;
                MemetroDialog.showDialog(CredentialsActivity.this, null, message);
            }
            else if(city == null || country == null){
                String message = view.getRootView().getContext(). getString(R.string.error_register_city_country);
                MemetroDialog.showDialog(CredentialsActivity.this, null, message);
            }
            else if(binding.solveCaptcha.getText().toString().isEmpty()){
                String message = view.getRootView().getContext(). getString(R.string.error_register_resolve_captcha);
                MemetroDialog.showDialog(CredentialsActivity.this, null, message);
            }
            else {
                viewModel.register(
                        binding.username.getText().toString(),
                        binding.email.getText().toString(),
                        binding.password.getText().toString(),
                        country.getId(),
                        city.getId(),
                        binding.solveCaptcha.getText().toString(),
                        captchaKey
                );
            }
        });
        binding.spinnerCountry.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int i, long l) {
                country = (Country) adapterView.getAdapter().getItem(i);
                viewModel.getCities(country.getId());
            }
            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {

            }
        });

        binding.spinnerCity.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int i, long l) {
                city = (City) adapterView.getAdapter().getItem(i);
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {

            }
        });

        binding.password.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {}

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                _passwordCheck();
            }
            @Override
            public void afterTextChanged(Editable editable) {}
        });

        binding.repeatPassword.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {}

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                _passwordCheck();
            }
            @Override
            public void afterTextChanged(Editable editable) {}
        });

        binding.email.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {}

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                viewModel.emailCheck(binding.email.getText().toString());
            }
            @Override
            public void afterTextChanged(Editable editable) {}
        });

        binding.captchaReload.setOnClickListener(view -> {
           viewModel.getCaptcha();
        });
    }

    private void _passwordCheck() {
        viewModel.passwordCheck(binding.password.getText().toString(), binding.repeatPassword.getText().toString());
    }

    protected void handleSignupState(ResultState<SignupResponseModel> signupState) {
        switch (signupState.getStatus()) {
            case SUCCESS:
                if (pdialog.isShowing()) {
                    pdialog.dismiss();
                    Toast.makeText(context, getString(R.string.register_ok), Toast.LENGTH_SHORT).show();
                    finish();
                }
                break;
            case ERROR:
                if (pdialog.isShowing()) pdialog.dismiss();
                String message = "Error doing sign up!";
                MemetroDialog.showDialog(CredentialsActivity.this, null, message);
                break;
            case LOADING:
                pdialog.show();
                break;
        }
    }
    protected void handlePasswordState(CredentialsViewModel.PasswordStatus status) {
        switch (status) {
            case OK:
                binding.register.setEnabled(true);
                binding.passwordStatusMessage.setText("");
                break;
            case EMPTY:
                binding.passwordStatusMessage.setText("Password can not be empty!");
                break;
            case TOO_SHORT:
                binding.passwordStatusMessage.setText("Password have to be minimum of " + String.valueOf(PASS_MIN_LENGTH));
                break;
            case NOT_MATCH:
                binding.passwordStatusMessage.setText("Passwords not match!");
                break;
            case NOT_NUMS:
                binding.passwordStatusMessage.setText("Password have to contain numbers!");
                break;
            default:
                binding.passwordStatusMessage.setText("");
                break;
        }
    }

    private void handleStaticDataState(ResultState<StaticDataResponseModel> result){
        switch (result.getStatus()) {
            case SUCCESS:
                if (pdialog.isShowing()) pdialog.dismiss();
                break;
            case ERROR:
                if (pdialog.isShowing()) pdialog.dismiss();
                MemetroDialog.showDialog(CredentialsActivity.this, null, getString(R.string.reach_server_error), this::finish);
                break;
            case LOADING:
                pdialog.show();
                break;
        }
    }

    private void handleCaptchaState(ResultState<CaptchaDataModel> result){
        switch (result.getStatus()) {
            case SUCCESS:
                if (pdialog.isShowing()) pdialog.dismiss();
                byte[] imageByteArray = Base64.decode(result.getData().getCaptchaImage(), Base64.DEFAULT);
                Glide.with(this)
                        .asBitmap()
                        .load(imageByteArray)
                        .into(binding.captchaImage);
                this.captchaKey = result.getData().getCaptchaKey();
                break;
            case ERROR:
                if (pdialog.isShowing()) pdialog.dismiss();
                MemetroDialog.showDialog(CredentialsActivity.this, null, getString(R.string.reach_captcha_error));
                break;
            case LOADING:
                pdialog.show();
                break;
        }
    }

    @Override
    protected ActivityRegisterCredentialsBinding getBinding() {
        return ActivityRegisterCredentialsBinding.inflate(getLayoutInflater());
    }

    @Override
    protected void observeViewModel() {
        viewModel.countries.observe(this, countries -> {
            if(countries != null) {
                LayoutUtils.setDefaultSpinner(
                        this,
                        binding.spinnerCountry,
                        countries
                );
                if(countries.size()>0) viewModel.getCities(countries.get(0).getId());
            }
        });
        viewModel.cities.observe(this, cities -> {
            if(cities != null) {
                LayoutUtils.setDefaultSpinner(
                        this,
                        binding.spinnerCity,
                        cities
                );
            }
        });
        viewModel.signupState.observe(this, this::handleSignupState);
        viewModel.passwordState.observe(this, this::handlePasswordState);
        viewModel.staticDataState.observe(this, this::handleStaticDataState);
        viewModel.captchaState.observe(this, this::handleCaptchaState);
    }

    @Override
    protected void initViewModels() {
        viewModel = new ViewModelProvider(this).get(CredentialsViewModel.class);
    }

}
