/*
 * Copyright 2013 Nytyr [me at nytyr dot me]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.memetro.android.api.oauth;

import android.content.Context;
import android.content.Intent;
import android.util.Log;

import com.memetro.android.R;
import com.memetro.android.ui.login.LoginActivity;
import com.memetro.android.api.Config;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class OAuth {

    private Context context;
    private String oauthServer = Config.OAUTHSERVER;
    private String clientId = Config.OAUTHCLIENTID;
    private String clientSecret = Config.OAUTHCLIENTSECRET;

    private boolean firstAuthCall = true;
    private int lastHttpStatus = -1;

    public OAuth(Context context) {
        this.context = context;
    }

    public JSONObject login (String username, String password){

        Map<String, String> params = new HashMap<String, String>(5);

        params.put("username", username);
        params.put("password", password);
        params.put("client_id", clientId);
        params.put("client_secret", clientSecret);
        params.put("grant_type", "password");

        JSONObject result = call("oauth", "token", params);

        try{
            if(!result.getBoolean("success")){
                JSONObject errorResult = new JSONObject();
                errorResult.put("error", result.getString("message"));
                return errorResult;
            }
        }catch(Exception e){
            return result;
        }
        return result;
    }

    public JSONObject call(String controller, String action, Map<String, String> params) {
        HttpClient httpclient = new DefaultHttpClient();
        HttpPost httppost = new HttpPost(oauthServer+controller+"/"+action);

        try {
            // Add data
            httppost.setEntity(new UrlEncodedFormEntity(Map2NameValuePair(params)));

            // Execute Post
            HttpResponse response = httpclient.execute(httppost);

            // Catch headers
            int statusCode = response.getStatusLine().getStatusCode();
            lastHttpStatus = statusCode;
            if (statusCode != 200){
                JSONObject returnJ = new JSONObject();
                returnJ.put("success", false);
                returnJ.put("data", new JSONArray());

                Log.d("Http Status Code", String.valueOf(statusCode));

                switch (statusCode){
                    case 401:
                        if (refreshToken() && firstAuthCall) {
                            firstAuthCall = false;
                            Utils utils = new Utils();
                            params.put("access_token", utils.getToken(context));
                            return call(controller, action, params);
                        }

                        returnJ.put("message", context.getString(R.string.session_expired));
                        Utils utils = new Utils();
                        utils.setToken(context, "", "");
                        Intent intent = new Intent(context, LoginActivity.class);
                        intent.setFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_NEW_TASK);
                        context.startActivity(intent);
                        return returnJ;
                    case 404:
                        returnJ.put("message", context.getString(R.string.action_not_found));
                        return returnJ;
                    case 500:
                        returnJ.put("message", context.getString(R.string.server_error));
                        return returnJ;
                    default:
                        returnJ.put("message", context.getString(R.string.internal_error));
                        return returnJ;
                }
            }

            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String json = reader.readLine();
            JSONTokener tokener = new JSONTokener(json);

            return new JSONObject(tokener);

        } catch (IOException | JSONException e) {

            // TODO Auto-generated catch block

        }
        return new JSONObject();
    }

    private boolean refreshToken() {
        Map<String, String> params = new HashMap<String, String>(4);

        Utils Utils = new Utils();
        params.put("refresh_token", Utils.getRefreshToken(context));
        params.put("client_id", clientId);
        params.put("client_secret", clientSecret);
        params.put("grant_type", "refresh_token");

        JSONObject result = call("oauth", "token", params);

        if (lastHttpStatus == 200) {
            try{
                Utils.setToken(context, result.getString("access_token"), result.getString("refresh_token"));
                return true;
            } catch(Exception e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private List<NameValuePair> Map2NameValuePair(Map<String, String> params) {
        List<NameValuePair> returnParams = new ArrayList<NameValuePair>();
        if (params == null) return returnParams;
        for (Map.Entry<String, String> entry : params.entrySet()) {
            returnParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
        }
        return returnParams;
    }
}
