package com.mcsnowflake.worktimer.configuration

import android.content.SharedPreferences
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import com.mcsnowflake.worktimer.configuration.Preference.AUTO_SWITCH_MODE
import com.mcsnowflake.worktimer.configuration.Preference.DND_MODE
import com.mcsnowflake.worktimer.configuration.Preference.HYDRATION_MODE
import com.mcsnowflake.worktimer.configuration.Preference.OVERTIME_MODE
import com.mcsnowflake.worktimer.configuration.Preference.STATISTICS_ON
import com.mcsnowflake.worktimer.configuration.TimerInput.LONG_BREAK_DURATION
import com.mcsnowflake.worktimer.configuration.TimerInput.SHORT_BREAK_DURATION
import com.mcsnowflake.worktimer.configuration.TimerInput.SHORT_BREAK_REPETITIONS
import com.mcsnowflake.worktimer.configuration.TimerInput.WORK_SESSION_DURATION
import kotlin.time.Duration
import kotlin.time.Duration.Companion.minutes
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn

class ConfigurationStore(
    private val dataStore: DataStore<Preferences>,
) {

    suspend operator fun <P> set(configuration: Configuration<P>, value: P) = dataStore.edit { it[configuration.key] = value }
    suspend operator fun <P> get(configuration: Configuration<P>) = getAsFlow(configuration).firstOrNull() ?: configuration.default
    fun <P> getAsFlow(configuration: Configuration<P>) = dataStore.data.map { it[configuration.key] ?: configuration.default }

    fun <P> getAsStateFlow(configuration: Configuration<P>, coroutineScope: CoroutineScope) = getAsFlow(configuration)
        .distinctUntilChanged()
        .stateIn(coroutineScope, SharingStarted.Eagerly, configuration.default)

    suspend fun getDuration(configuration: TimerInput): Duration = get(configuration).minutes

    suspend fun recover(sharedPreferences: SharedPreferences) {
        if (sharedPreferences.all.isNotEmpty()) dataStore.edit {

            if (sharedPreferences.contains(STATISTICS_ON.name))
                it[STATISTICS_ON.key] = sharedPreferences.getBoolean(STATISTICS_ON.name, STATISTICS_ON.default)

            // mode settings
            if (sharedPreferences.contains(DND_MODE.name))
                it[DND_MODE.key] = sharedPreferences.getBoolean(DND_MODE.name, DND_MODE.default)

            if (sharedPreferences.contains(HYDRATION_MODE.name))
                it[HYDRATION_MODE.key] = sharedPreferences.getBoolean(HYDRATION_MODE.name, HYDRATION_MODE.default)

            if (sharedPreferences.contains(AUTO_SWITCH_MODE.name))
                it[AUTO_SWITCH_MODE.key] = sharedPreferences.getBoolean(AUTO_SWITCH_MODE.name, AUTO_SWITCH_MODE.default)

            if (sharedPreferences.contains(OVERTIME_MODE.name))
                it[OVERTIME_MODE.key] = sharedPreferences.getBoolean(OVERTIME_MODE.name, OVERTIME_MODE.default)

            // timer settings
            if (sharedPreferences.contains(WORK_SESSION_DURATION.name))
                it[WORK_SESSION_DURATION.key] = sharedPreferences.getInt(WORK_SESSION_DURATION.name, WORK_SESSION_DURATION.default)

            if (sharedPreferences.contains(SHORT_BREAK_DURATION.name))
                it[SHORT_BREAK_DURATION.key] = sharedPreferences.getInt(SHORT_BREAK_DURATION.name, SHORT_BREAK_DURATION.default)

            if (sharedPreferences.contains(LONG_BREAK_DURATION.name))
                it[LONG_BREAK_DURATION.key] = sharedPreferences.getInt(LONG_BREAK_DURATION.name, LONG_BREAK_DURATION.default)

            if (sharedPreferences.contains(SHORT_BREAK_REPETITIONS.name))
                it[SHORT_BREAK_REPETITIONS.key] = sharedPreferences.getInt(SHORT_BREAK_REPETITIONS.name, SHORT_BREAK_REPETITIONS.default)
        }
    }
}

