package com.mcsnowflake.worktimer.ui.views.dashboard

import androidx.compose.runtime.Composable
import androidx.compose.runtime.State
import androidx.compose.runtime.collectAsState
import androidx.core.text.isDigitsOnly
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.TimerInput
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted.Companion.Eagerly
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

class SetupViewModel(
    configuration: ConfigurationStore
) : ViewModel() {

    private val inputs = TimerInput.entries.associateWith { MutableStateFlow("") }

    init {
        inputs.forEach { input ->
            // write back inputs
            viewModelScope.launch {
                input.value
                    .filter { it.isDigitsOnly() && it.isNotEmpty() }
                    .map { it.toInt() }
                    .collect { configuration[input.key] = it }
            }

            // Load initial value
            viewModelScope.launch { input.value.emit(configuration.getAsFlow(input.key).first().toString()) }
        }
    }

    @Composable
    operator fun get(timerInput: TimerInput): State<String> = inputs[timerInput]!!.asStateFlow().collectAsState()
    operator fun set(timerInput: TimerInput, value: String) = viewModelScope.launch { inputs[timerInput]!!.emit(value) }

    val isInputValid = combine(inputs.values) { inputs: Array<String> -> inputs.all { it.isDigitsOnly() && it.isNotEmpty() && it.toInt() > 0 } }
        .stateIn(viewModelScope, Eagerly, false)
}
