@file:OptIn(FlowPreview::class)

package com.mcsnowflake.worktimer.ui.views.settings

import android.Manifest
import android.os.Build
import android.util.Log
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.animation.core.animateDpAsState
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.SegmentedButton
import androidx.compose.material3.SegmentedButtonDefaults
import androidx.compose.material3.SingleChoiceSegmentedButtonRow
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.State
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.mcsnowflake.worktimer.configuration.Preference
import com.mcsnowflake.worktimer.configuration.TimerInput
import com.mcsnowflake.worktimer.ui.views.settings.SettingsCardData.Companion.SettingsCardHeight
import com.mcsnowflake.worktimer.ui.views.settings.SettingsCardData.Companion.expandedSettingsCardHeight
import kotlinx.coroutines.FlowPreview

@Preview(widthDp = 400)
@Composable
fun Simple() = SettingsCard(
    enabled = true,
    primary = Preference.OVERTIME_MODE,
    primaryValue = remember { mutableStateOf(false) },
)

@Preview(widthDp = 400)
@Composable
fun Disabled() = SettingsCard(
    enabled = false,
    primary = Preference.OVERTIME_MODE,
    primaryValue = remember { mutableStateOf(true) },
)

@Preview(widthDp = 400)
@Composable
fun Extended() = SettingsCard(
    enabled = true,
    primary = Preference.OVERTIME_MODE,
    primaryValue = remember { mutableStateOf(true) },
    secondaryValue = remember { mutableIntStateOf(10) },
)

@Composable
fun SettingsCard(
    modifier: Modifier = Modifier,
    enabled: Boolean = true,
    primary: Preference = Preference.OVERTIME_MODE,
    primaryValue: State<Boolean> = mutableStateOf(true),
    primaryOnChange: (Boolean) -> Unit = {},
    secondaryValue: State<Int>? = null,
    secondaryOnChange: (Int) -> Unit = {}
) {
    val data: SettingsCardData = remember(primary) { SettingsCardData[primary] }
    val height by animateDpAsState(if (primaryValue.value && secondaryValue != null) expandedSettingsCardHeight else SettingsCardHeight)

    HorizontalDivider()
    Row(
        modifier = modifier
            .fillMaxWidth()
            .padding(vertical = 8.dp)
            .height(height),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.SpaceEvenly
    ) {

        Icon(
            modifier = Modifier
                .padding(horizontal = 16.dp)
                .size(36.dp),
            imageVector = data.icon,
            contentDescription = "${data.label} icon"
        )

        Column(
            verticalArrangement = Arrangement.spacedBy(8.dp),
            horizontalAlignment = Alignment.Start
        ) {

            Primary(data, primaryValue, primaryOnChange, enabled)

            if (primaryValue.value && secondaryValue != null) Secondary(secondaryValue, secondaryOnChange)
        }
    }
}

@Composable
fun Primary(data: SettingsCardData, primaryValue: State<Boolean>, primaryOnChange: (Boolean) -> Unit, enabled: Boolean) = Row(
    verticalAlignment = Alignment.CenterVertically
) {
    Column(
        Modifier.weight(1f),
        verticalArrangement = Arrangement.spacedBy(8.dp, Alignment.CenterVertically),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text(
            modifier = Modifier.fillMaxWidth(),
            text =  stringResource(data.label),
            style = MaterialTheme.typography.bodyLarge,
            fontWeight = FontWeight.Bold

        )
        Text(
            modifier = Modifier.fillMaxWidth(),
            text = stringResource(if (primaryValue.value) data.onText else data.offText),
            style = MaterialTheme.typography.bodyMedium,
        )

    }
    Switch(modifier = Modifier.padding(horizontal = 16.dp), checked = primaryValue.value, onCheckedChange = primaryOnChange, enabled = enabled)
}

@Composable
fun Secondary(secondaryValue: State<Int>, secondaryOnChange: (Int) -> Unit) = Row(
    verticalAlignment = Alignment.CenterVertically,
    horizontalArrangement = Arrangement.spacedBy(8.dp, Alignment.CenterHorizontally)
) {

    val options = listOf(5, 10, 15)

    SingleChoiceSegmentedButtonRow {
        options.forEachIndexed { index, label ->
            SegmentedButton(
                shape = SegmentedButtonDefaults.itemShape(
                    index = index,
                    count = options.size
                ),
                onClick = { secondaryOnChange(label) },
                selected = label == secondaryValue.value,
                label = {
                    Text(
                        text =  stringResource(label),
                        style = MaterialTheme.typography.bodyMedium
                    )
                }
            )
        }
    }

    Text(
        text = "minutes",
        style = MaterialTheme.typography.bodyMedium,
    )

}

@Composable
fun SimpleSettingsCard(
    modifier: Modifier,
    viewModel: SettingsViewModel,
    preference: Preference,
    onChange: (Boolean) -> Unit = { viewModel[preference] = it }
) = SettingsCard(
    modifier = modifier,
    primary = preference,
    primaryValue = viewModel[preference],
    primaryOnChange = onChange
)

@Composable
fun ExtendedSettingsCard(
    modifier: Modifier,
    viewModel: SettingsViewModel,
    preference: Preference,
    input: TimerInput,
    onChange: (Boolean) -> Unit = { viewModel[preference] = it }
) = SettingsCard(
    modifier = modifier,
    primary = preference,
    primaryValue = viewModel[preference],
    primaryOnChange = onChange,
    secondaryValue = viewModel[input],
    secondaryOnChange = { viewModel[input] = it }
)

@Composable
fun NotificationsSettingsCard(
    modifier: Modifier,
    viewModel: SettingsViewModel,
    preference: Preference,
    areNotificationsAllowed: () -> Boolean
) {
    val requestLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.RequestPermission()
    ) { viewModel[preference] = it }

    SimpleSettingsCard(
        modifier = modifier,
        preference = preference,
        viewModel = viewModel,
        onChange = {
            Log.d("NotificationsSettingsCard", "prefrence $preference set to $it / notifications: ${areNotificationsAllowed()}")
            if (it && Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU && !areNotificationsAllowed()) {
                Log.d("NotificationsSettingsCard", "request launcher launched")
                requestLauncher.launch(Manifest.permission.POST_NOTIFICATIONS)
            } else viewModel[preference] = it
        }
    )
}
