@file:OptIn(ExperimentalTime::class)

package com.mcsnowflake.worktimer.ui.components.timer

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.togetherWith
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.LinearProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.State
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.StrokeCap
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.mcsnowflake.worktimer.TextHandler
import com.mcsnowflake.worktimer.state.SessionData
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerState
import kotlin.time.Clock.System.now
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlinx.coroutines.delay

@Composable
fun MiddleBar(
    modifier: Modifier = Modifier,
    timerState: State<TimerState> = mutableStateOf(TimerState.Stopped)
) = AnimatedContent(
    timerState.value,
    modifier = modifier.heightIn(min = 6.dp),
    transitionSpec = {

        when (targetState) {
            is TimerState.Session.Finished -> {
                fadeIn() togetherWith fadeOut()
            }

            is TimerState.Session.Running -> {
                fadeIn(initialAlpha = 1f) togetherWith fadeOut(targetAlpha = 1f)
            }

            TimerState.Stopped -> {
                fadeIn(initialAlpha = 1f) togetherWith fadeOut(targetAlpha = 1f)
            }
        }
    }

) { state ->
    when (state) {
        is TimerState.Session.Finished -> InfoText(state.session)
        is TimerState.Session.Running -> ProgressBar(state.session)
        TimerState.Stopped -> HorizontalDivider(
            color = MaterialTheme.colorScheme.secondary.copy(alpha = .2f),
            thickness = 6.dp,
            modifier = Modifier
                .clip(RoundedCornerShape(4.dp))
            // .padding(top = 4.dp, bottom = 2.dp)
        )
    }

}

@Composable
fun InfoText(session: SessionData) {
    val infoText = remember(session) {
        TextHandler.getInfoText(session.type)
    }

    Text(
        modifier = Modifier.fillMaxWidth(),
        text = stringResource(infoText),
        textAlign = TextAlign.Center,
        fontSize = 20.sp
    )
}

@Composable
private fun ProgressBar(session: SessionData) {

    var progress by remember { mutableFloatStateOf(progress(session)) }

    LaunchedEffect(session) {
        while (true) {
            progress = progress(session)
            delay(1.seconds)
        }
    }

    LinearProgressIndicator(
        modifier = Modifier
            .fillMaxWidth()
            .height(5.dp),
        progress = { progress },
        color = if (session.type == WORK_SESSION) MaterialTheme.colorScheme.primary
        else MaterialTheme.colorScheme.tertiary,
        trackColor = if (session.type == WORK_SESSION) MaterialTheme.colorScheme.secondary.copy(alpha = .2f)
        else MaterialTheme.colorScheme.tertiary.copy(alpha = .2f),
        strokeCap = StrokeCap.Round
    )
}

private fun progress(session: SessionData): Float {
    val progressTime = session.start - now()
    val totalTime = session.start - session.end
    return (progressTime / totalTime).toFloat()
}
