package com.mcsnowflake.worktimer.ui.components.banners

import android.Manifest
import android.app.NotificationManager
import android.os.Build
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.rounded.NotificationsOff
import androidx.compose.material3.Icon
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import com.mcsnowflake.worktimer.R
import com.mcsnowflake.worktimer.configuration.Preference
import org.koin.compose.koinInject

@Preview
@Composable
fun NotificationBanner(
    bannersViewModel: BannersViewModel = koinInject(),
    notificationManager: NotificationManager = koinInject(),
) {

    var hasNotificationsPermission by remember { mutableStateOf(false) }
    val notificationsEnabled by bannersViewModel[Preference.NOTIFICATIONS_ON].collectAsState()
    val reevaluate by bannersViewModel.getReevaluateBanners()

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        LaunchedEffect(reevaluate) {
            hasNotificationsPermission = !notificationManager.areNotificationsEnabled()
        }

        val permissionRequestLauncher = rememberLauncherForActivityResult(
            contract = ActivityResultContracts.RequestPermission()
        ) { bannersViewModel.setReevaluateBanners() }

        AnimatedBanner(
            isVisible = hasNotificationsPermission && notificationsEnabled,
            icon = { Icon(Icons.Rounded.NotificationsOff, stringResource(id = R.string.notification_banner_icon_description)) },
            text = stringResource(id = R.string.notification_banner_text),
            actions = mapOf(
                stringResource(id = R.string.notification_banner_action_allow) to { permissionRequestLauncher.launch(Manifest.permission.POST_NOTIFICATIONS) },
                stringResource(id = R.string.notification_banner_action_dismiss) to { bannersViewModel[Preference.NOTIFICATIONS_ON] = false }
            )
        )
    }
}
