/*
 * Copyright 2025 Nicolas Maltais
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.maltaisn.notes.ui.settings

import android.app.Activity
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.view.View
import android.widget.TextView
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.RequiresApi
import androidx.annotation.StringRes
import androidx.core.net.toUri
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updatePadding
import androidx.fragment.app.viewModels
import androidx.navigation.fragment.findNavController
import androidx.preference.DropDownPreference
import androidx.preference.Preference
import androidx.preference.PreferenceFragmentCompat
import androidx.preference.SwitchPreferenceCompat
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.color.DynamicColors
import com.google.android.material.snackbar.Snackbar
import com.maltaisn.notes.App
import com.maltaisn.notes.BuildConfig
import com.maltaisn.notes.R
import com.maltaisn.notes.TAG
import com.maltaisn.notes.databinding.FragmentSettingsBinding
import com.maltaisn.notes.model.DefaultPrefsManager
import com.maltaisn.notes.navigateSafe
import com.maltaisn.notes.setEnterExitTransitions
import com.maltaisn.notes.ui.AppTheme
import com.maltaisn.notes.ui.common.ConfirmDialog
import com.maltaisn.notes.ui.main.MainActivity
import com.maltaisn.notes.ui.notification.NotificationPermission
import com.maltaisn.notes.ui.observeEvent
import com.maltaisn.notes.ui.reminder.ReminderPermission
import dagger.hilt.android.AndroidEntryPoint
import java.text.DateFormat

@AndroidEntryPoint
class SettingsFragment : PreferenceFragmentCompat(), ConfirmDialog.Callback, ExportPasswordDialog.Callback,
    ImportPasswordDialog.Callback {

    private val viewModel: SettingsViewModel by viewModels()

    private var exportDataLauncher: ActivityResultLauncher<Intent>? = null
    private var autoExportLauncher: ActivityResultLauncher<Intent>? = null
    private var importDataLauncher: ActivityResultLauncher<Intent>? = null

    private var notificationPermission: NotificationPermission? = null
    private var reminderPermission: ReminderPermission? = null

    override fun onCreate(state: Bundle?) {
        super.onCreate(state)
        val context = requireContext()

        exportDataLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            val uri = result.data?.data
            if (result.resultCode == Activity.RESULT_OK && uri != null) {
                val output = try {
                    // write and *truncate*. Otherwise the file is not overwritten!
                    context.contentResolver.openOutputStream(uri, "wt")
                } catch (e: Exception) {
                    Log.i(TAG, "Data export failed", e)
                    null
                }
                if (output != null) {
                    viewModel.exportData(output, getString(R.string.edit_copy_untitled_name))
                } else {
                    showMessage(R.string.export_fail)
                }
            }
        }

        autoExportLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            val uri = result.data?.data
            if (result.resultCode == Activity.RESULT_OK && uri != null) {
                val output = try {
                    val cr = context.contentResolver
                    cr.takePersistableUriPermission(uri, Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
                    cr.openOutputStream(uri)
                } catch (e: Exception) {
                    Log.i(TAG, "Data export failed", e)
                    null
                }
                if (output != null) {
                    viewModel.setupAutoExport(output, uri.toString())
                } else {
                    showMessage(R.string.export_fail)
                    autoExportPref.isChecked = false
                }
            }
        }

        importDataLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            val uri = result.data?.data
            if (result.resultCode == Activity.RESULT_OK && uri != null) {
                val input = try {
                    context.contentResolver.openInputStream(uri)
                } catch (e: Exception) {
                    Log.i(TAG, "Data import failed", e)
                    null
                }
                if (input != null) {
                    viewModel.importData(input)
                } else {
                    showMessage(R.string.import_bad_input)
                }
            }
        }

        notificationPermission = NotificationPermission(this)
        reminderPermission = getContext()?.let { ReminderPermission(this, it) }

        setEnterExitTransitions()
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        val binding = FragmentSettingsBinding.bind(view)

        binding.toolbar.setNavigationOnClickListener {
            findNavController().popBackStack()
        }

        // Apply padding so that the settings don't overlap with the navigation bar
        val rcv = view.findViewById<RecyclerView>(R.id.recycler_view)

        ViewCompat.setOnApplyWindowInsetsListener(rcv) { _, insets ->
            val sysWindow = insets.getInsets(WindowInsetsCompat.Type.systemBars() or WindowInsetsCompat.Type.ime())
            rcv.updatePadding(bottom = sysWindow.bottom)
            insets
        }

        setupViewModelObservers()
    }

    private fun setupViewModelObservers() {
        viewModel.messageEvent.observeEvent(viewLifecycleOwner, ::showMessage)
        viewModel.lastAutoExport.observe(viewLifecycleOwner) { date ->
            updateAutoExportSummary(autoExportPref.isChecked, date)
        }
        viewModel.exportData.observeEvent(viewLifecycleOwner) { extension ->
            val intent = Intent(Intent.ACTION_CREATE_DOCUMENT)
                .setType("application/$extension").addCategory(Intent.CATEGORY_OPENABLE).apply {
                    putExtra(Intent.EXTRA_TITLE, "notes.$extension")
                }
            exportDataLauncher?.launch(intent)
        }
        viewModel.releasePersistableUriEvent.observeEvent(viewLifecycleOwner) { uri ->
            try {
                requireContext().contentResolver.releasePersistableUriPermission(uri.toUri(),
                    Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
            } catch (e: Exception) {
                // Permission was revoked? will probably happen sometimes
                Log.i(TAG, "Failed to release persistable URI permission", e)
            }
        }
        viewModel.showImportPasswordDialogEvent.observeEvent(viewLifecycleOwner) {
            ImportPasswordDialog.newInstance()
                .show(childFragmentManager, null)
        }
        viewModel.askNotificationPermission.observeEvent(viewLifecycleOwner) {
            notificationPermission?.request()
        }
        viewModel.askReminderPermission.observeEvent(viewLifecycleOwner) {
            reminderPermission?.request()
        }
    }

    override fun onCreatePreferences(savedInstanceState: Bundle?, rootKey: String?) {
        val context = requireContext()
        setPreferencesFromResource(R.xml.prefs, rootKey)

        requirePreference<DropDownPreference>(DefaultPrefsManager.THEME).setOnPreferenceChangeListener { _, theme ->
            (context.applicationContext as App).updateTheme(AppTheme.fromValue(theme as String))
            true
        }

        requirePreference<Preference>(DefaultPrefsManager.DYNAMIC_COLORS).apply {
            if (DynamicColors.isDynamicColorAvailable()) {
                setOnPreferenceClickListener {
                    ConfirmDialog.newInstance(
                        title = R.string.pref_restart_dialog_title,
                        message = R.string.pref_restart_dialog_description,
                        btnPositive = R.string.action_ok
                    ).show(childFragmentManager, RESTART_DIALOG_TAG)
                    true
                }
            } else {
                // Hide dynamic color / material you preference on unsupported Android versions
                isVisible = false
            }
        }

        requirePreference<Preference>(DefaultPrefsManager.PREVIEW_LINES).setOnPreferenceClickListener {
            findNavController().navigateSafe(SettingsFragmentDirections.actionNestedSettings(
                R.xml.prefs_preview_lines, R.string.pref_preview_lines))
            true
        }

        requirePreference<Preference>(DefaultPrefsManager.EXPORT_DATA).setOnPreferenceClickListener {
            viewModel.onExportJsonRequested()
            true
        }

        val encryptedExportPref: SwitchPreferenceCompat = requirePreference(DefaultPrefsManager.ENCRYPTED_EXPORT)
        // Older versions don't support PBKDF2withHmacSHA512
        if (Build.VERSION.SDK_INT < 26) {
            encryptedExportPref.isVisible = false
        }

        encryptedExportPref.setOnPreferenceChangeListener { _, newValue ->
            if (newValue == true) {
                ExportPasswordDialog.newInstance()
                    .show(childFragmentManager, null)
            } else {
                viewModel.deleteExportKey()
            }
            true
        }

        autoExportPref.setOnPreferenceChangeListener { _, newValue ->
            if (newValue == true) {
                ConfirmDialog.newInstance(
                    title = R.string.pref_data_auto_export,
                    message = R.string.auto_export_message,
                    btnPositive = R.string.action_ok
                ).show(childFragmentManager, AUTOMATIC_EXPORT_DIALOG_TAG)
            } else {
                updateAutoExportSummary(false)
                viewModel.disableAutoExport()
            }
            true
        }

        requirePreference<Preference>(DefaultPrefsManager.IMPORT_DATA).setOnPreferenceClickListener {
            // note: explicit mimetype fails for some devices, see #11
            val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                .setType("*/*").addCategory(Intent.CATEGORY_OPENABLE)
            importDataLauncher?.launch(intent)
            true
        }

        requirePreference<Preference>(DefaultPrefsManager.EXPORT_ARCHIVE).setOnPreferenceClickListener {
            viewModel.onExportZipRequested()
            true
        }

        requirePreference<Preference>(DefaultPrefsManager.CLEAR_DATA).setOnPreferenceClickListener {
            ConfirmDialog.newInstance(
                title = R.string.pref_data_clear,
                message = R.string.pref_data_clear_confirm_message,
                btnPositive = R.string.action_clear
            ).show(childFragmentManager, CLEAR_DATA_DIALOG_TAG)
            true
        }

        // Set version name as summary text for version preference
        requirePreference<Preference>(DefaultPrefsManager.VERSION).summary = BuildConfig.VERSION_NAME
    }

    override fun onDestroy() {
        super.onDestroy()
        exportDataLauncher = null
        importDataLauncher = null
        autoExportLauncher = null
        notificationPermission = null
        reminderPermission = null
    }

    private fun showMessage(@StringRes messageId: Int) {
        val snackbar = Snackbar.make(requireView(), messageId, Snackbar.LENGTH_SHORT)
            .setGestureInsetBottomIgnored(true)
        snackbar.view.findViewById<TextView>(com.google.android.material.R.id.snackbar_text).maxLines = 5
        snackbar.show()
    }

    private fun <T : Preference> requirePreference(key: CharSequence) =
        checkNotNull(findPreference<T>(key)) { "Could not find preference with key '$key'." }

    private val autoExportPref: SwitchPreferenceCompat
        get() = requirePreference(DefaultPrefsManager.AUTO_EXPORT)

    private fun updateAutoExportSummary(enabled: Boolean, date: Long = 0) {
        if (enabled) {
            autoExportPref.summary = buildString {
                appendLine(getString(R.string.pref_data_auto_export_summary))
                append(getString(R.string.pref_data_auto_export_date,
                    DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.SHORT).format(date)))
            }
        } else {
            autoExportPref.summary = getString(R.string.pref_data_auto_export_summary)
        }
    }

    override fun onDialogPositiveButtonClicked(tag: String?) {
        when (tag) {
            RESTART_DIALOG_TAG -> {
                // Reload MainActivity to apply theming changes
                requireActivity().finish()
                startActivity(Intent(requireContext(), MainActivity::class.java))
            }

            CLEAR_DATA_DIALOG_TAG -> {
                viewModel.clearData()
            }

            AUTOMATIC_EXPORT_DIALOG_TAG -> {
                val intent = Intent(Intent.ACTION_CREATE_DOCUMENT)
                    .setType("application/json")
                    .addCategory(Intent.CATEGORY_OPENABLE)
                autoExportLauncher?.launch(intent)
            }

            NOTIF_PERMISSION_DIALOG -> notificationPermission?.onDialogPositiveButtonClicked(tag)
            else -> reminderPermission?.onDialogPositiveButtonClicked(tag)
        }
    }

    override fun onDialogNegativeButtonClicked(tag: String?) {
        when (tag) {
            AUTOMATIC_EXPORT_DIALOG_TAG -> {
                // No file chosen for auto export, disable it.
                autoExportPref.isChecked = false
            }

            NOTIF_PERMISSION_DIALOG -> notificationPermission?.onDialogNegativeButtonClicked(tag)
            else -> reminderPermission?.onDialogPositiveButtonClicked(tag)
        }
    }

    override fun onDialogCancelled(tag: String?) {
        when (tag) {
            AUTOMATIC_EXPORT_DIALOG_TAG -> {
                // No file chosen for auto export, disable it.
                autoExportPref.isChecked = false
            }

            NOTIF_PERMISSION_DIALOG -> notificationPermission?.onDialogCancelled(tag)
            else -> reminderPermission?.onDialogPositiveButtonClicked(tag)
        }
    }

    private val exportEncryptionPref: SwitchPreferenceCompat
        get() = requirePreference(DefaultPrefsManager.ENCRYPTED_EXPORT)

    @RequiresApi(Build.VERSION_CODES.M)
    override fun onExportPasswordDialogPositiveButtonClicked(password: String) {
        viewModel.generateExportKeyFromPassword(password)
    }

    override fun onExportPasswordDialogNegativeButtonClicked() {
        exportEncryptionPref.isChecked = false
    }

    override fun onExportPasswordDialogCancelled() {
        exportEncryptionPref.isChecked = false
    }

    @RequiresApi(Build.VERSION_CODES.M)
    override fun onImportPasswordDialogPositiveButtonClicked(password: String) {
        viewModel.importSavedEncryptedJsonData(password)
    }

    companion object {
        private const val RESTART_DIALOG_TAG = "restart_dialog"
        private const val CLEAR_DATA_DIALOG_TAG = "clear_data_dialog"
        private const val AUTOMATIC_EXPORT_DIALOG_TAG = "automatic_export_dialog"
        private const val NOTIF_PERMISSION_DIALOG = "notif-permission-dialog"
    }
}
