// Copyright (c) 2024, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:test/test.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import '../../dart/resolution/node_text_expectations.dart';
import '../elements_base.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(ClassElementTest_keepLinking);
    defineReflectiveTests(ClassElementTest_fromBytes);
    // TODO(scheglov): implement augmentation
    // defineReflectiveTests(ClassElementTest_augmentation_keepLinking);
    // defineReflectiveTests(ClassElementTest_augmentation_fromBytes);
    defineReflectiveTests(UpdateNodeTextExpectations);
  });
}

abstract class ClassElementTest extends ElementsBaseTest {
  test_class_abstract() async {
    var library = await buildLibrary('abstract class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_base() async {
    var library = await buildLibrary('base class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @11
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    base class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_const() async {
    var library = await buildLibrary('class C { const C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 16
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_const_external() async {
    var library = await buildLibrary('class C { external const C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            external const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 25
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        external const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  C();
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              documentationComment: /**\n   * Docs\n   */
              typeName: C
              typeNameOffset: 34
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          documentationComment: /**\n   * Docs\n   */
''');
  }

  test_class_constructor_explicit_named() async {
    var library = await buildLibrary('class C { C.foo(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            foo @12
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              element: <testLibraryFragment>::@class::C::@constructor::foo#element
              typeName: C
              typeNameOffset: 10
              periodOffset: 11
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::C::@constructor::foo
''');
  }

  test_class_constructor_explicit_type_params() async {
    var library = await buildLibrary('class C<T, U> { C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 16
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_explicit_unnamed() async {
    var library = await buildLibrary('class C { C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_external() async {
    var library = await buildLibrary('class C { external C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            external new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 19
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        external new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_factory() async {
    var library = await buildLibrary('class C { factory C() => throw 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_field_formal_dynamic_dynamic() async {
    var library =
        await buildLibrary('class C { dynamic x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @36
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_dynamic_typed() async {
    var library = await buildLibrary('class C { dynamic x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_dynamic_untyped() async {
    var library = await buildLibrary('class C { dynamic x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_noReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(this.x(double b));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic Function(double)
              formalParameters
                requiredPositional b
                  type: double
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(int this.x(double b));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int Function(double)
              formalParameters
                requiredPositional b
                  type: double
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType_generic() async {
    var library = await buildLibrary(r'''
class C {
  Function() f;
  C(List<U> this.f<T, U>(T t));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            f @23
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              getter2: <testLibraryFragment>::@class::C::@getter::f
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 28
              formalParameters
                this.f @43
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::f#element
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::C::@getter::f
              element: <testLibraryFragment>::@class::C::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                _f
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::_f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: dynamic Function()
          getter: <testLibraryFragment>::@class::C::@getter::f#element
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final f
              type: List<U> Function<T, U>(T)
              formalParameters
                requiredPositional t
                  type: T
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::C::@getter::f
          returnType: dynamic Function()
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional _f
              type: dynamic Function()
          returnType: void
''');
  }

  test_class_constructor_field_formal_multiple_matching_fields() async {
    // This is a compile-time error but it should still analyze consistently.
    var library = await buildLibrary('class C { C(this.x); int x; String x; }');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x::@def::0
              element: <testLibraryFragment>::@class::C::@field::x::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::x::@def::0
              setter2: <testLibraryFragment>::@class::C::@setter::x::@def::0
            x @35
              reference: <testLibraryFragment>::@class::C::@field::x::@def::1
              element: <testLibraryFragment>::@class::C::@field::x::@def::1#element
              getter2: <testLibraryFragment>::@class::C::@getter::x::@def::1
              setter2: <testLibraryFragment>::@class::C::@setter::x::@def::1
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                this.x @17
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::0
              element: <testLibraryFragment>::@class::C::@getter::x::@def::0#element
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::1
              element: <testLibraryFragment>::@class::C::@getter::x::@def::1#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::0
              element: <testLibraryFragment>::@class::C::@setter::x::@def::0#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@def::0::@parameter::_x#element
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::1
              element: <testLibraryFragment>::@class::C::@setter::x::@def::1#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@def::1::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x::@def::0#element
          setter: <testLibraryFragment>::@class::C::@setter::x::@def::0#element
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x::@def::1
          type: String
          getter: <testLibraryFragment>::@class::C::@getter::x::@def::1#element
          setter: <testLibraryFragment>::@class::C::@setter::x::@def::1#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x::@def::0
          returnType: int
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x::@def::1
          returnType: String
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x::@def::0
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x::@def::1
          formalParameters
            requiredPositional _x
              type: String
          returnType: void
''');
  }

  test_class_constructor_field_formal_no_matching_field() async {
    // This is a compile-time error but it should still analyze consistently.
    var library = await buildLibrary('class C { C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                this.x @17
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_typed_dynamic() async {
    var library = await buildLibrary('class C { num x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: num
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
          returnType: void
''');
  }

  test_class_constructor_field_formal_typed_typed() async {
    var library = await buildLibrary('class C { num x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: num
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
          returnType: void
''');
  }

  test_class_constructor_field_formal_typed_untyped() async {
    var library = await buildLibrary('class C { num x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: num
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: num
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
          returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_dynamic() async {
    var library = await buildLibrary('class C { var x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_typed() async {
    var library = await buildLibrary('class C { var x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_untyped() async {
    var library = await buildLibrary('class C { var x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_constructor_fieldFormal_named_noDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x}); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType x
              firstFragment: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_constructor_fieldFormal_named_withDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x: 42}); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 42 @28
                      staticType: int
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType x
              firstFragment: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                expression: expression_0
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_constructor_fieldFormal_optional_noDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x]); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalPositional final hasImplicitType x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_constructor_fieldFormal_optional_withDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x = 42]); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 42 @29
                      staticType: int
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalPositional final hasImplicitType x
              type: int
              constantInitializer
                expression: expression_0
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_constructor_implicit_type_params() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_initializers_assertInvocation() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              formalParameters
                x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional x
              type: int
          constantInitializers
            AssertInitializer
              assertKeyword: assert @29
              leftParenthesis: ( @35
              condition: BinaryExpression
                leftOperand: SimpleIdentifier
                  token: x @36
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                  staticType: int
                operator: >= @38
                rightOperand: IntegerLiteral
                  literal: 42 @41
                  staticType: int
                element: dart:core::<fragment>::@class::num::@method::>=#element
                staticInvokeType: bool Function(num)
                staticType: bool
              rightParenthesis: ) @43
''');
  }

  test_class_constructor_initializers_assertInvocation_message() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42, 'foo');
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              formalParameters
                x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional x
              type: int
          constantInitializers
            AssertInitializer
              assertKeyword: assert @29
              leftParenthesis: ( @35
              condition: BinaryExpression
                leftOperand: SimpleIdentifier
                  token: x @36
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                  staticType: int
                operator: >= @38
                rightOperand: IntegerLiteral
                  literal: 42 @41
                  staticType: int
                element: dart:core::<fragment>::@class::num::@method::>=#element
                staticInvokeType: bool Function(num)
                staticType: bool
              comma: , @43
              message: SimpleStringLiteral
                literal: 'foo' @45
              rightParenthesis: ) @50
''');
  }

  test_class_constructor_initializers_field() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = 42;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @35
                element: <testLibraryFragment>::@class::C::@field::x#element
                staticType: null
              equals: = @37
              expression: IntegerLiteral
                literal: 42 @39
                staticType: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
''');
  }

  test_class_constructor_initializers_field_notConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = foo();
}
int foo() => 42;
''');
    // It is OK to keep non-constant initializers.
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
      functions
        foo @52
          reference: <testLibraryFragment>::@function::foo
          element: <testLibrary>::@function::foo
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @35
                element: <testLibraryFragment>::@class::C::@field::x#element
                staticType: null
              equals: = @37
              expression: MethodInvocation
                methodName: SimpleIdentifier
                  token: foo @39
                  element: <testLibrary>::@function::foo
                  staticType: int Function()
                argumentList: ArgumentList
                  leftParenthesis: ( @42
                  rightParenthesis: ) @43
                staticInvokeType: int Function()
                staticType: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
  functions
    foo
      reference: <testLibrary>::@function::foo
      firstFragment: <testLibraryFragment>::@function::foo
      returnType: int
''');
  }

  test_class_constructor_initializers_field_optionalPositionalParameter() async {
    var library = await buildLibrary('''
class A {
  final int _f;
  const A([int f = 0]) : _f = f;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _f @22
              reference: <testLibraryFragment>::@class::A::@field::_f
              element: <testLibraryFragment>::@class::A::@field::_f#element
              getter2: <testLibraryFragment>::@class::A::@getter::_f
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 34
              formalParameters
                default f @41
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 0 @45
                      staticType: int
          getters
            synthetic get _f
              reference: <testLibraryFragment>::@class::A::@getter::_f
              element: <testLibraryFragment>::@class::A::@getter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _f
          firstFragment: <testLibraryFragment>::@class::A::@field::_f
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::_f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalPositional f
              type: int
              constantInitializer
                expression: expression_0
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: _f @51
                element: <testLibraryFragment>::@class::A::@field::_f#element
                staticType: null
              equals: = @54
              expression: SimpleIdentifier
                token: f @56
                element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f#element
                staticType: int
      getters
        synthetic get _f
          firstFragment: <testLibraryFragment>::@class::A::@getter::_f
          returnType: int
''');
  }

  test_class_constructor_initializers_field_recordLiteral() async {
    var library = await buildLibrary('''
class C {
  final Object x;
  const C(int a) : x = (0, a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 36
              formalParameters
                a @42
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional a
              type: int
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @47
                element: <testLibraryFragment>::@class::C::@field::x#element
                staticType: null
              equals: = @49
              expression: RecordLiteral
                leftParenthesis: ( @51
                fields
                  IntegerLiteral
                    literal: 0 @52
                    staticType: int
                  SimpleIdentifier
                    token: a @55
                    element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
                    staticType: int
                rightParenthesis: ) @56
                staticType: (int, int)
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: Object
''');
  }

  test_class_constructor_initializers_field_withParameter() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C(int p) : x = 1 + p;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
              formalParameters
                p @35
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional p
              type: int
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @40
                element: <testLibraryFragment>::@class::C::@field::x#element
                staticType: null
              equals: = @42
              expression: BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @44
                  staticType: int
                operator: + @46
                rightOperand: SimpleIdentifier
                  token: p @48
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p#element
                  staticType: int
                element: dart:core::<fragment>::@class::num::@method::+#element
                staticInvokeType: num Function(num)
                staticType: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
''');
  }

  test_class_constructor_initializers_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  const B(dynamic x);
  const B.f()
   : this(A<Function()>());
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 46
              formalParameters
                x @56
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::x#element
            const f @70
              reference: <testLibraryFragment>::@class::B::@constructor::f
              element: <testLibraryFragment>::@class::B::@constructor::f#element
              typeName: B
              typeNameOffset: 68
              periodOffset: 69
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional x
              type: dynamic
        const f
          firstFragment: <testLibraryFragment>::@class::B::@constructor::f
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @79
              argumentList: ArgumentList
                leftParenthesis: ( @83
                arguments
                  InstanceCreationExpression
                    constructorName: ConstructorName
                      type: NamedType
                        name: A @84
                        typeArguments: TypeArgumentList
                          leftBracket: < @85
                          arguments
                            GenericFunctionType
                              functionKeyword: Function @86
                              parameters: FormalParameterList
                                leftParenthesis: ( @94
                                rightParenthesis: ) @95
                              declaredElement: GenericFunctionTypeElement
                                parameters
                                returnType: dynamic
                                type: dynamic Function()
                              type: dynamic Function()
                          rightBracket: > @96
                        element2: <testLibrary>::@class::A
                        type: A<dynamic Function()>
                      element: ConstructorMember
                        baseElement: <testLibraryFragment>::@class::A::@constructor::new#element
                        substitution: {T: dynamic Function()}
                    argumentList: ArgumentList
                      leftParenthesis: ( @97
                      rightParenthesis: ) @98
                    staticType: A<dynamic Function()>
                rightParenthesis: ) @99
              element: <testLibraryFragment>::@class::B::@constructor::new#element
          redirectedConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
''');
  }

  test_class_constructor_initializers_superInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
}
class B extends A {
  const B() : super(const []);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                values @33
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::values#element
        class B @50
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 72
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional values
              type: List<String>
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @78
              argumentList: ArgumentList
                leftParenthesis: ( @83
                arguments
                  ListLiteral
                    constKeyword: const @84
                    leftBracket: [ @90
                    rightBracket: ] @91
                    staticType: List<String>
                rightParenthesis: ) @92
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_superInvocation_named() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(int p);
}
class C extends A {
  const C() : super.aaa(42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
              formalParameters
                p @28
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::p#element
        class C @40
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 62
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const aaa
          firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa
          formalParameters
            requiredPositional p
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @68
              period: . @73
              constructorName: SimpleIdentifier
                token: aaa @74
                element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @77
                arguments
                  IntegerLiteral
                    literal: 42 @78
                    staticType: int
                rightParenthesis: ) @80
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa#element
''');
  }

  test_class_constructor_initializers_superInvocation_named_underscore() async {
    var library = await buildLibrary('''
class A {
  const A._();
}
class B extends A {
  const B() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const _ @20
              reference: <testLibraryFragment>::@class::A::@constructor::_
              element: <testLibraryFragment>::@class::A::@constructor::_#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 55
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const _
          firstFragment: <testLibraryFragment>::@class::A::@constructor::_
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @61
              period: . @66
              constructorName: SimpleIdentifier
                token: _ @67
                element: <testLibraryFragment>::@class::A::@constructor::_#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @68
                rightParenthesis: ) @69
              element: <testLibraryFragment>::@class::A::@constructor::_#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::_#element
''');
  }

  test_class_constructor_initializers_superInvocation_namedExpression() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(a, {int b});
}
class C extends A {
  const C() : super.aaa(1, b: 2);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
              formalParameters
                a @24
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::a#element
                default b @32
                  reference: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b#element
        class C @45
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 67
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const aaa
          firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @73
              period: . @78
              constructorName: SimpleIdentifier
                token: aaa @79
                element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @82
                arguments
                  IntegerLiteral
                    literal: 1 @83
                    staticType: int
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: b @86
                        element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b#element
                        staticType: null
                      colon: : @87
                    expression: IntegerLiteral
                      literal: 2 @89
                      staticType: int
                rightParenthesis: ) @90
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa#element
''');
  }

  test_class_constructor_initializers_superInvocation_unnamed() async {
    var library = await buildLibrary('''
class A {
  const A(int p);
}
class C extends A {
  const C.ccc() : super(42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                p @24
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::p#element
        class C @36
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const ccc @60
              reference: <testLibraryFragment>::@class::C::@constructor::ccc
              element: <testLibraryFragment>::@class::C::@constructor::ccc#element
              typeName: C
              typeNameOffset: 58
              periodOffset: 59
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional p
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const ccc
          firstFragment: <testLibraryFragment>::@class::C::@constructor::ccc
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @68
              argumentList: ArgumentList
                leftParenthesis: ( @73
                arguments
                  IntegerLiteral
                    literal: 42 @74
                    staticType: int
                rightParenthesis: ) @76
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_thisInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
  const A.empty() : this(const []);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                values @33
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::values#element
            const empty @52
              reference: <testLibraryFragment>::@class::A::@constructor::empty
              element: <testLibraryFragment>::@class::A::@constructor::empty#element
              typeName: A
              typeNameOffset: 50
              periodOffset: 51
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional values
              type: List<String>
        const empty
          firstFragment: <testLibraryFragment>::@class::A::@constructor::empty
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @62
              argumentList: ArgumentList
                leftParenthesis: ( @66
                arguments
                  ListLiteral
                    constKeyword: const @67
                    leftBracket: [ @73
                    rightBracket: ] @74
                    staticType: List<String>
                rightParenthesis: ) @75
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          redirectedConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, 'bbb');
  const C.named(int a, String b);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
            const named @56
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 54
              periodOffset: 55
              formalParameters
                a @66
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::a#element
                b @76
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @24
              period: . @28
              constructorName: SimpleIdentifier
                token: named @29
                element: <testLibraryFragment>::@class::C::@constructor::named#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @34
                arguments
                  IntegerLiteral
                    literal: 1 @35
                    staticType: int
                  SimpleStringLiteral
                    literal: 'bbb' @38
                rightParenthesis: ) @43
              element: <testLibraryFragment>::@class::C::@constructor::named#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: String
''');
  }

  test_class_constructor_initializers_thisInvocation_namedExpression() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, b: 2);
  const C.named(a, {int b});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
            const named @55
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 53
              periodOffset: 54
              formalParameters
                a @61
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::a#element
                default b @69
                  reference: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @24
              period: . @28
              constructorName: SimpleIdentifier
                token: named @29
                element: <testLibraryFragment>::@class::C::@constructor::named#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @34
                arguments
                  IntegerLiteral
                    literal: 1 @35
                    staticType: int
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: b @38
                        element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
                        staticType: null
                      colon: : @39
                    expression: IntegerLiteral
                      literal: 2 @41
                      staticType: int
                rightParenthesis: ) @42
              element: <testLibraryFragment>::@class::C::@constructor::named#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
              type: int
''');
  }

  test_class_constructor_initializers_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C.named() : this(1, 'bbb');
  const C(int a, String b);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 18
              periodOffset: 19
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 54
              formalParameters
                a @60
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
                b @70
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @30
              argumentList: ArgumentList
                leftParenthesis: ( @34
                arguments
                  IntegerLiteral
                    literal: 1 @35
                    staticType: int
                  SimpleStringLiteral
                    literal: 'bbb' @38
                rightParenthesis: ) @43
              element: <testLibraryFragment>::@class::C::@constructor::new#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: String
''');
  }

  test_class_constructor_parameters_super_explicitType_function() async {
    var library = await buildLibrary('''
class A {
  A(Object? a);
}

class B extends A {
  B(int super.a<T extends num>(T d)?);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @22
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @35
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 51
              formalParameters
                super.a @63
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: Object?
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int Function<T extends num>(T)?
              formalParameters
                requiredPositional d
                  type: T
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_explicitType_interface() async {
    var library = await buildLibrary('''
class A {
  A(num a);
}

class B extends A {
  B(int super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
              formalParameters
                super.a @59
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: num
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_explicitType_interface_nullable() async {
    var library = await buildLibrary('''
class A {
  A(num? a);
}

class B extends A {
  B(int? super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @19
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @32
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 48
              formalParameters
                super.a @61
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: num?
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int?
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_invalid_topFunction() async {
    var library = await buildLibrary('''
void f(super.a) {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      functions
        f @5
          reference: <testLibraryFragment>::@function::f
          element: <testLibrary>::@function::f
          formalParameters
            super.a @13
              element: <testLibraryFragment>::@function::f::@parameter::a#element
  functions
    f
      reference: <testLibrary>::@function::f
      firstFragment: <testLibraryFragment>::@function::f
      formalParameters
        requiredPositional final hasImplicitType a
          type: dynamic
      returnType: void
''');
  }

  test_class_constructor_parameters_super_optionalNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({String o1, super.a, String o2, super.b}) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 77
              formalParameters
                default o1 @87
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @97
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @107
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @117
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
            requiredNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed o1
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
              type: String
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
            optionalNamed o2
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
              type: String
            optionalNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_defaultValue() async {
    var library = await buildLibrary('''
class A {
  A({int a = 0});
}

class B extends A {
  B({super.a});
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 0 @23
                      staticType: int
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              formalParameters
                default super.a @62
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                expression: expression_0
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved() async {
    var library = await buildLibrary('''
class A {
  A({required int a});
}

class B extends A {
  B({super.b});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @42
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 58
              formalParameters
                default super.b @67
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A(int a);
}

class B extends A {
  B({super.a});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
              formalParameters
                default super.a @56
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B([String o1, super.a, String o2, super.b]) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                b @28
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                default o1 @67
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @77
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @87
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @97
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalPositional o1
              type: String
            optionalPositional final hasImplicitType a
              type: int
            optionalPositional o2
              type: String
            optionalPositional final hasImplicitType b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({
    required String o1,
    required super.a,
    required String o2,
    required super.b,
  }) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 77
              formalParameters
                default o1 @101
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @124
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @147
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @170
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
            requiredNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredNamed o1
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
              type: String
            requiredNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
            requiredNamed o2
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
              type: String
            requiredNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredNamed_defaultValue() async {
    var library = await buildLibrary('''
class A {
  A({int a = 0});
}

class B extends A {
  B({required super.a});
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 0 @23
                      staticType: int
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              formalParameters
                default super.a @71
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                expression: expression_0
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B(String o1, super.a, String o2, super.b) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                b @28
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                o1 @66
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                super.a @76
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                o2 @86
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                super.b @96
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional o1
              type: String
            requiredPositional final hasImplicitType a
              type: int
            requiredPositional o2
              type: String
            requiredPositional final hasImplicitType b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder() async {
    // It is important that `B` is declared after `C`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
abstract class A {
  A(int a);
}

class C extends B {
  C(super.a);
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
              formalParameters
                a @27
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class C @40
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 56
              formalParameters
                super.a @64
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
        class B @77
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 93
              formalParameters
                super.a @101
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder_generic() async {
    // It is important that `C` is declared before `B`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
class A {
  A(int a);
}

class C extends B<String> {
  C(super.a);
}

class B<T> extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class C @31
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 55
              formalParameters
                super.a @63
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
        class B @76
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @78
              element: T@78
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 95
              formalParameters
                super.a @103
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B<String>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved() async {
    var library = await buildLibrary('''
class A {}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @18
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 34
              formalParameters
                super.a @42
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A({required int a})
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                super.a @65
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_params() async {
    var library = await buildLibrary('class C { C(x, int y); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                x @12
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                y @19
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional hasImplicitType x
              type: dynamic
            requiredPositional y
              type: int
''');
  }

  test_class_constructor_redirected_factory_named() async {
    var library = await buildLibrary('''
class C {
  factory C() = D.named;
  C._();
}
class D extends C {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 20
            _ @39
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 37
              periodOffset: 38
        class D @52
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            named @70
              reference: <testLibraryFragment>::@class::D::@constructor::named
              element: <testLibraryFragment>::@class::D::@constructor::named#element
              typeName: D
              typeNameOffset: 68
              periodOffset: 69
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::D::@constructor::named
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_named_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 26
            _ @51
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 49
              periodOffset: 50
        class D @64
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @66
              element: T@66
            U @69
              element: U@69
          constructors
            named @94
              reference: <testLibraryFragment>::@class::D::@constructor::named
              element: <testLibraryFragment>::@class::D::@constructor::named#element
              typeName: D
              typeNameOffset: 92
              periodOffset: 93
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::D::@constructor::named#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
        U
      supertype: C<U, T>
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::D::@constructor::named
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_named_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>.named;
  B._();
}
class C<T, U> extends A<U, T> {
  C.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @35
              element: T@35
            U @38
              element: U@38
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
            _ @78
              reference: <testLibraryFragment>::@class::B::@constructor::_
              element: <testLibraryFragment>::@class::B::@constructor::_#element
              typeName: B
              typeNameOffset: 76
              periodOffset: 77
        class C @91
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @93
              element: T@93
            U @96
              element: U@96
          constructors
            named @121
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 119
              periodOffset: 120
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            T @10
              element: T@10
            U @13
              element: U@13
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::C::@constructor::named#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::B::@constructor::_
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      supertype: C<U, T>
        alias: <testLibrary>::@typeAlias::A
          typeArguments
            U
            T
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        T
        U
      aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_named_imported() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
            _ @58
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 56
              periodOffset: 57
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_imported_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @27
              element: T@27
            U @30
              element: U@30
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 45
            _ @70
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 68
              periodOffset: 69
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_prefixed() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
            _ @69
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 67
              periodOffset: 68
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_prefixed_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @34
              element: T@34
            U @37
              element: U@37
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 52
            _ @81
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 79
              periodOffset: 80
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_class() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D.named<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @8
              element: E@8
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_constructor() async {
    var library = await buildLibrary('''
class D {}
class C<E> {
  factory C() = D.named<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class D @6
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class C @17
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @19
              element: E@19
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 34
  classes
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_unnamed() async {
    var library = await buildLibrary('''
class C {
  factory C() = D;
  C._();
}
class D extends C {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 20
            _ @33
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 31
              periodOffset: 32
        class D @46
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 62
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 26
            _ @45
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 43
              periodOffset: 44
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @60
              element: T@60
            U @63
              element: U@63
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 86
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::D::@constructor::new#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
        U
      supertype: C<U, T>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>;
  B_();
}
class C<T, U> extends B<U, T> {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @35
              element: T@35
            U @38
              element: U@38
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
          methods
            B_ @70
              reference: <testLibraryFragment>::@class::B::@method::B_
              element: <testLibraryFragment>::@class::B::@method::B_#element
        class C @84
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @86
              element: T@86
            U @89
              element: U@89
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 112
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            T @10
              element: T@10
            U @13
              element: U@13
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::C::@constructor::new#element
            substitution: {T: U, U: T}
      methods
        abstract B_
          firstFragment: <testLibraryFragment>::@class::B::@method::B_
          returnType: dynamic
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      supertype: B<U, T>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        T
        U
      aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 50
              periodOffset: 51
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @27
              element: T@27
            U @30
              element: U@30
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 45
            _ @64
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 62
              periodOffset: 63
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_viaTypeAlias() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = A;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 50
              periodOffset: 51
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 61
              periodOffset: 62
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @34
              element: T@34
            U @37
              element: U@37
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 52
            _ @75
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 73
              periodOffset: 74
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
            substitution: {T: U, U: T}
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_viaTypeAlias() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.A;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 61
              periodOffset: 62
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_unresolved() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @8
              element: E@8
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_unnamed_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A = C;
class B {
  factory B() = A;
  B._();
}
class C extends B {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @21
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 35
            _ @48
              reference: <testLibraryFragment>::@class::B::@constructor::_
              element: <testLibraryFragment>::@class::B::@constructor::_#element
              typeName: B
              typeNameOffset: 46
              periodOffset: 47
        class C @61
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 77
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::B::@constructor::_
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::_#element
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      aliasedType: C
''');
  }

  test_class_constructor_redirected_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 18
              periodOffset: 19
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 37
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @43
              period: . @47
              constructorName: SimpleIdentifier
                token: named @48
                element: <testLibraryFragment>::@class::C::@constructor::named#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @53
                rightParenthesis: ) @54
              element: <testLibraryFragment>::@class::C::@constructor::named#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_named_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            const named @23
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 21
              periodOffset: 22
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 40
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @46
              period: . @50
              constructorName: SimpleIdentifier
                token: named @51
                element: <testLibraryFragment>::@class::C::@constructor::named#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @56
                rightParenthesis: ) @57
              element: <testLibraryFragment>::@class::C::@constructor::named#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_named_notConst() async {
    var library = await buildLibrary('''
class C {
  C.named();
  C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            named @14
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 25
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
            const named @33
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 31
              periodOffset: 32
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @43
              argumentList: ArgumentList
                leftParenthesis: ( @47
                rightParenthesis: ) @48
              element: <testLibraryFragment>::@class::C::@constructor::new#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
            const named @36
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 34
              periodOffset: 35
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          constantInitializers
            RedirectingConstructorInvocation
              thisKeyword: this @46
              argumentList: ArgumentList
                leftParenthesis: ( @50
                rightParenthesis: ) @51
              element: <testLibraryFragment>::@class::C::@constructor::new#element
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_notConst() async {
    var library = await buildLibrary('''
class C {
  C();
  C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
            named @21
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 19
              periodOffset: 20
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_redirectedConstructor_generic01() async {
    // Note, this code has compile-time errors.
    // `A` returned  by the redirected constructor is not `B<U>`.
    // But we still have some element model.
    var library = await buildLibrary(r'''
class A implements B<int> {}

class B<U> implements C<U> {
  factory B() = A;
}

class C<V> {
  factory C() = B<V>;
}
''');

    configuration
      ..forClassConstructors(classNames: {'C'})
      ..elementPrinterConfiguration.withRedirectedConstructors = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @87
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 104
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::B::@constructor::new#element
            substitution: {U: V}
            redirectedConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_redirectedConstructor_generic11() async {
    var library = await buildLibrary(r'''
class A<T> implements B<T> {}

class B<U> implements C<U> {
  factory B() = A<U>;
}

class C<V> {
  factory C() = B<V>;
}
''');

    configuration
      ..forClassConstructors(classNames: {'C'})
      ..elementPrinterConfiguration.withRedirectedConstructors = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @91
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 108
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: ConstructorMember
            baseElement: <testLibraryFragment>::@class::B::@constructor::new#element
            substitution: {U: V}
            redirectedConstructor: ConstructorMember
              baseElement: <testLibraryFragment>::@class::A::@constructor::new#element
              substitution: {T: V}
              redirectedConstructor: <null>
''');
  }

  test_class_constructor_superConstructor_generic01() async {
    var library = await buildLibrary(r'''
class A {}
class B<U> extends A {}
class C extends B<int> {}
''');

    configuration
      ..forClassConstructors(classNames: {'C'})
      ..elementPrinterConfiguration.withSuperConstructors = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @41
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_generic11() async {
    var library = await buildLibrary(r'''
class A<T> {}
class B<U> extends A<String> {}
class C extends B<int> {}
''');

    configuration
      ..forClassConstructors(classNames: {'C'})
      ..elementPrinterConfiguration.withSuperConstructors = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @52
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_generic_named() async {
    var library = await buildLibrary('''
class A<T> {
  A.named(T a);
}
class B extends A<int> {
  B() : super.named(0);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            named @17
              reference: <testLibraryFragment>::@class::A::@constructor::named
              element: <testLibraryFragment>::@class::A::@constructor::named#element
              typeName: A
              typeNameOffset: 15
              periodOffset: 16
              formalParameters
                a @25
                  element: <testLibraryFragment>::@class::A::@constructor::named::@parameter::a#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 58
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::A::@constructor::named
          formalParameters
            requiredPositional a
              type: T
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<int>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::named#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_named() async {
    var library = await buildLibrary('''
class A {
  A.named();
}
class B extends A {
  B() : super.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            named @14
              reference: <testLibraryFragment>::@class::A::@constructor::named
              element: <testLibraryFragment>::@class::A::@constructor::named#element
              typeName: A
              typeNameOffset: 12
              periodOffset: 13
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::A::@constructor::named
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::named#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_explicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B() : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 33
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 33
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit2() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_unnamed_implicit() async {
    var library = await buildLibrary('class C {}');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_withCycles_const() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = const D();
}
class D {
  final x;
  const D() : x = const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            x @70
              reference: <testLibraryFragment>::@class::D::@field::x
              element: <testLibraryFragment>::@class::D::@field::x#element
              getter2: <testLibraryFragment>::@class::D::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 81
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::D::@getter::x
              element: <testLibraryFragment>::@class::D::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @35
                element: <testLibraryFragment>::@class::C::@field::x#element
                staticType: null
              equals: = @37
              expression: InstanceCreationExpression
                keyword: const @39
                constructorName: ConstructorName
                  type: NamedType
                    name: D @45
                    element2: <testLibrary>::@class::D
                    type: D
                  element: <testLibraryFragment>::@class::D::@constructor::new#element
                argumentList: ArgumentList
                  leftParenthesis: ( @46
                  rightParenthesis: ) @47
                staticType: D
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::D::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::D::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          constantInitializers
            ConstructorFieldInitializer
              fieldName: SimpleIdentifier
                token: x @87
                element: <testLibraryFragment>::@class::D::@field::x#element
                staticType: null
              equals: = @89
              expression: InstanceCreationExpression
                keyword: const @91
                constructorName: ConstructorName
                  type: NamedType
                    name: C @97
                    element2: <testLibrary>::@class::C
                    type: C
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
                argumentList: ArgumentList
                  leftParenthesis: ( @98
                  rightParenthesis: ) @99
                staticType: C
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::D::@getter::x
          returnType: dynamic
''');
  }

  test_class_constructor_withCycles_nonConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  C() : x = new D();
}
class D {
  final x;
  D() : x = new C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
        class D @50
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            x @62
              reference: <testLibraryFragment>::@class::D::@field::x
              element: <testLibraryFragment>::@class::D::@field::x#element
              getter2: <testLibraryFragment>::@class::D::@getter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 67
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::D::@getter::x
              element: <testLibraryFragment>::@class::D::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::D::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::D::@getter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::D::@getter::x
          returnType: dynamic
''');
  }

  test_class_constructors_named() async {
    var library = await buildLibrary('''
class C {
  C.foo();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            foo @14
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              element: <testLibraryFragment>::@class::C::@constructor::foo#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::C::@constructor::foo
''');
  }

  test_class_constructors_unnamed() async {
    var library = await buildLibrary('''
class C {
  C();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructors_unnamed_new() async {
    var library = await buildLibrary('''
class C {
  C.new();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new @14
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented() async {
    var library = await buildLibrary('''
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @22
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_mix() async {
    var library = await buildLibrary('''
/**
 * aaa
 */
/**
 * bbb
 */
class A {}

/**
 * aaa
 */
/// bbb
/// ccc
class B {}

/// aaa
/// bbb
/**
 * ccc
 */
class C {}

/// aaa
/// bbb
/**
 * ccc
 */
/// ddd
class D {}

/**
 * aaa
 */
// bbb
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @79
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @122
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @173
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @207
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      documentationComment: /**\n * bbb\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      documentationComment: /// bbb\n/// ccc
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * ccc\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      documentationComment: /// ddd
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      documentationComment: /**\n * aaa\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// first
/// second
/// third
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @37
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// first\n/// second\n/// third
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_with_references() async {
    var library = await buildLibrary('''
/**
 * Docs referring to [D] and [E]
 */
class C {}

class D {}
class E {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @47
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @59
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @70
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs referring to [D] and [E]\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_documented_with_windows_line_endings() async {
    var library = await buildLibrary('/**\r\n * Docs\r\n */\r\nclass C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_withLeadingNotDocumentation() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @66
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_withMetadata() async {
    var library = await buildLibrary('''
/// Comment 1
/// Comment 2
@Annotation()
class BeforeMeta {}

/// Comment 1
/// Comment 2
@Annotation.named()
class BeforeMetaNamed {}

@Annotation()
/// Comment 1
/// Comment 2
class AfterMeta {}

/// Comment 1
@Annotation()
/// Comment 2
class AroundMeta {}

/// Doc comment.
@Annotation()
// Not doc comment.
class DocBeforeMetaNotDocAfter {}

class Annotation {
  const Annotation();
  const Annotation.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class BeforeMeta @48
          reference: <testLibraryFragment>::@class::BeforeMeta
          element: <testLibrary>::@class::BeforeMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::BeforeMeta::@constructor::new
              element: <testLibraryFragment>::@class::BeforeMeta::@constructor::new#element
              typeName: BeforeMeta
        class BeforeMetaNamed @117
          reference: <testLibraryFragment>::@class::BeforeMetaNamed
          element: <testLibrary>::@class::BeforeMetaNamed
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new
              element: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new#element
              typeName: BeforeMetaNamed
        class AfterMeta @185
          reference: <testLibraryFragment>::@class::AfterMeta
          element: <testLibrary>::@class::AfterMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::AfterMeta::@constructor::new
              element: <testLibraryFragment>::@class::AfterMeta::@constructor::new#element
              typeName: AfterMeta
        class AroundMeta @247
          reference: <testLibraryFragment>::@class::AroundMeta
          element: <testLibrary>::@class::AroundMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::AroundMeta::@constructor::new
              element: <testLibraryFragment>::@class::AroundMeta::@constructor::new#element
              typeName: AroundMeta
        class DocBeforeMetaNotDocAfter @319
          reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
          element: <testLibrary>::@class::DocBeforeMetaNotDocAfter
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new
              element: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new#element
              typeName: DocBeforeMetaNotDocAfter
        class Annotation @354
          reference: <testLibraryFragment>::@class::Annotation
          element: <testLibrary>::@class::Annotation
          constructors
            const new
              reference: <testLibraryFragment>::@class::Annotation::@constructor::new
              element: <testLibraryFragment>::@class::Annotation::@constructor::new#element
              typeName: Annotation
              typeNameOffset: 375
            const named @408
              reference: <testLibraryFragment>::@class::Annotation::@constructor::named
              element: <testLibraryFragment>::@class::Annotation::@constructor::named#element
              typeName: Annotation
              typeNameOffset: 397
              periodOffset: 407
  classes
    class BeforeMeta
      reference: <testLibrary>::@class::BeforeMeta
      firstFragment: <testLibraryFragment>::@class::BeforeMeta
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::BeforeMeta::@constructor::new
    class BeforeMetaNamed
      reference: <testLibrary>::@class::BeforeMetaNamed
      firstFragment: <testLibraryFragment>::@class::BeforeMetaNamed
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new
    class AfterMeta
      reference: <testLibrary>::@class::AfterMeta
      firstFragment: <testLibraryFragment>::@class::AfterMeta
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::AfterMeta::@constructor::new
    class AroundMeta
      reference: <testLibrary>::@class::AroundMeta
      firstFragment: <testLibraryFragment>::@class::AroundMeta
      documentationComment: /// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::AroundMeta::@constructor::new
    class DocBeforeMetaNotDocAfter
      reference: <testLibrary>::@class::DocBeforeMetaNotDocAfter
      firstFragment: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
      documentationComment: /// Doc comment.
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new
    class Annotation
      reference: <testLibrary>::@class::Annotation
      firstFragment: <testLibraryFragment>::@class::Annotation
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::Annotation::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::Annotation::@constructor::named
''');
  }

  test_class_field_abstract() async {
    var library = await buildLibrary('''
abstract class C {
  abstract int i;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                _i
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        abstract i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic abstract get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
      setters
        synthetic abstract set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
          returnType: void
''');
  }

  test_class_field_const() async {
    var library = await buildLibrary('class C { static const int i = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer i @27
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @31
                  staticType: int
              getter2: <testLibraryFragment>::@class::C::@getter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static const hasInitializer i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          constantInitializer
            fragment: <testLibraryFragment>::@class::C::@field::i
            expression: expression_0
          getter: <testLibraryFragment>::@class::C::@getter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
''');
  }

  test_class_field_const_late() async {
    var library =
        await buildLibrary('class C { static late const int i = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer i @32
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @36
                  staticType: int
              getter2: <testLibraryFragment>::@class::C::@getter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static late const hasInitializer i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          constantInitializer
            fragment: <testLibraryFragment>::@class::C::@field::i
            expression: expression_0
          getter: <testLibraryFragment>::@class::C::@getter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
''');
  }

  test_class_field_covariant() async {
    var library = await buildLibrary('''
class C {
  covariant int x;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @26
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        covariant x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional covariant _x
              type: int
          returnType: void
''');
  }

  test_class_field_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  var x;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @38
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_field_duplicate_getter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  int get foo => 0;
}
''');
    configuration
      ..withAugmentedWithoutAugmentation = true
      ..withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              element: <testLibraryFragment>::@class::C::@field::foo::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo::@def::0
              setter2: <testLibraryFragment>::@class::C::@setter::foo
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              element: <testLibraryFragment>::@class::C::@field::foo::@def::1#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo::@def::1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::0
              element: <testLibraryFragment>::@class::C::@getter::foo::@def::0#element
            get foo @35
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::1
              element: <testLibraryFragment>::@class::C::@getter::foo::@def::1#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo::@def::0#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::1
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo::@def::1#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo::@def::0
          returnType: int
        get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo::@def::1
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
''');
  }

  test_class_field_duplicate_setter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  set foo(int _) {}
}
''');
    configuration
      ..withAugmentedWithoutAugmentation = true
      ..withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              element: <testLibraryFragment>::@class::C::@field::foo::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo::@def::0
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              element: <testLibraryFragment>::@class::C::@field::foo::@def::1#element
              setter2: <testLibraryFragment>::@class::C::@setter::foo::@def::1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::0
              element: <testLibraryFragment>::@class::C::@setter::foo::@def::0#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@def::0::@parameter::_foo#element
            set foo @31
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::1
              element: <testLibraryFragment>::@class::C::@setter::foo::@def::1#element
              formalParameters
                _ @39
                  element: <testLibraryFragment>::@class::C::@setter::foo::@def::1::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo::@def::0#element
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::1
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::foo::@def::1#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo::@def::0
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
        set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo::@def::1
          formalParameters
            requiredPositional _
              type: int
          returnType: void
''');
  }

  test_class_field_external() async {
    var library = await buildLibrary('''
abstract class C {
  external int i;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                _i
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        external i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
      setters
        synthetic set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
          returnType: void
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
  const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              initializer: expression_0
                IntegerLiteral
                  literal: 42 @22
                  staticType: int
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 34
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          constantInitializer
            fragment: <testLibraryFragment>::@class::C::@field::x
            expression: expression_0
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  final f = const A<int Function(double a)>();
  const B();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @46
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              initializer: expression_0
                InstanceCreationExpression
                  keyword: const @50
                  constructorName: ConstructorName
                    type: NamedType
                      name: A @56
                      typeArguments: TypeArgumentList
                        leftBracket: < @57
                        arguments
                          GenericFunctionType
                            returnType: NamedType
                              name: int @58
                              element2: dart:core::@class::int
                              type: int
                            functionKeyword: Function @62
                            parameters: FormalParameterList
                              leftParenthesis: ( @70
                              parameter: SimpleFormalParameter
                                type: NamedType
                                  name: double @71
                                  element2: dart:core::@class::double
                                  type: double
                                name: a @78
                                declaredElement: a@78
                                  type: double
                              rightParenthesis: ) @79
                            declaredElement: GenericFunctionTypeElement
                              parameters
                                a
                                  kind: required positional
                                  type: double
                              returnType: int
                              type: int Function(double)
                            type: int Function(double)
                        rightBracket: > @80
                      element2: <testLibrary>::@class::A
                      type: A<int Function(double)>
                    element: ConstructorMember
                      baseElement: <testLibraryFragment>::@class::A::@constructor::new#element
                      substitution: {T: int Function(double)}
                  argumentList: ArgumentList
                    leftParenthesis: ( @81
                    rightParenthesis: ) @82
                  staticType: A<int Function(double)>
              getter2: <testLibraryFragment>::@class::B::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 93
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: A<int Function(double)>
          constantInitializer
            fragment: <testLibraryFragment>::@class::B::@field::f
            expression: expression_0
          getter: <testLibraryFragment>::@class::B::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
          returnType: A<int Function(double)>
''');
  }

  test_class_field_final_hasInitializer_noConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_final_withSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int foo;
  A(this.foo);
  set foo(int newValue) {}
}
''');
    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            foo @22
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 29
              formalParameters
                this.foo @36
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::foo#element
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            set foo @48
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                newValue @56
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::newValue#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType foo
              type: int
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional newValue
              type: int
          returnType: void
''');
  }

  test_class_field_formal_param_inferred_type_implicit() async {
    var library = await buildLibrary('class C extends D { var v; C(this.v); }'
        ' abstract class D { int get v; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 27
              formalParameters
                this.v @34
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::v#element
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                _v
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
        class D @55
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic v
              reference: <testLibraryFragment>::@class::D::@field::v
              element: <testLibraryFragment>::@class::D::@field::v#element
              getter2: <testLibraryFragment>::@class::D::@getter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          getters
            get v @67
              reference: <testLibraryFragment>::@class::D::@getter::v
              element: <testLibraryFragment>::@class::D::@getter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType v
              type: int
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
          returnType: int
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
          returnType: void
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic v
          firstFragment: <testLibraryFragment>::@class::D::@field::v
          type: int
          getter: <testLibraryFragment>::@class::D::@getter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        abstract get v
          firstFragment: <testLibraryFragment>::@class::D::@getter::v
          returnType: int
''');
  }

  test_class_field_implicit_type() async {
    var library = await buildLibrary('class C { var x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_field_implicit_type_late() async {
    var library = await buildLibrary('class C { late var x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @19
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
          returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_explicit_initialized() async {
    var library = await buildLibrary('class C { num v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                _v
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
          returnType: num
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: num
          returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_initialized() async {
    var library = await buildLibrary('class C { var v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                _v
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
          returnType: int
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
          returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_uninitialized() async {
    var library = await buildLibrary(
        'class C extends D { var v; } abstract class D { int get v; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                _v
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
        class D @44
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic v
              reference: <testLibraryFragment>::@class::D::@field::v
              element: <testLibraryFragment>::@class::D::@field::v#element
              getter2: <testLibraryFragment>::@class::D::@getter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          getters
            get v @56
              reference: <testLibraryFragment>::@class::D::@getter::v
              element: <testLibraryFragment>::@class::D::@getter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
          returnType: int
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
          returnType: void
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic v
          firstFragment: <testLibraryFragment>::@class::D::@field::v
          type: int
          getter: <testLibraryFragment>::@class::D::@getter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        abstract get v
          firstFragment: <testLibraryFragment>::@class::D::@getter::v
          returnType: int
''');
  }

  test_class_field_inferred_type_nonStatic_inherited_resolveInitializer() async {
    var library = await buildLibrary(r'''
const a = 0;
abstract class A {
  const A();
  List<int> get f;
}
class B extends A {
  const B();
  final f = [a];
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @28
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::A::@field::f
              element: <testLibraryFragment>::@class::A::@field::f#element
              getter2: <testLibraryFragment>::@class::A::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 40
          getters
            get f @61
              reference: <testLibraryFragment>::@class::A::@getter::f
              element: <testLibraryFragment>::@class::A::@getter::f#element
        class B @72
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @107
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              initializer: expression_0
                ListLiteral
                  leftBracket: [ @111
                  elements
                    SimpleIdentifier
                      token: a @112
                      element: <testLibraryFragment>::@getter::a#element
                      staticType: int
                  rightBracket: ] @113
                  staticType: List<int>
              getter2: <testLibraryFragment>::@class::B::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 94
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
      topLevelVariables
        hasInitializer a @6
          reference: <testLibraryFragment>::@topLevelVariable::a
          element: <testLibrary>::@topLevelVariable::a
          initializer: expression_1
            IntegerLiteral
              literal: 0 @10
              staticType: int
          getter2: <testLibraryFragment>::@getter::a
      getters
        synthetic get a
          reference: <testLibraryFragment>::@getter::a
          element: <testLibraryFragment>::@getter::a#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::A::@field::f
          type: List<int>
          getter: <testLibraryFragment>::@class::A::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get f
          firstFragment: <testLibraryFragment>::@class::A::@getter::f
          returnType: List<int>
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: List<int>
          constantInitializer
            fragment: <testLibraryFragment>::@class::B::@field::f
            expression: expression_0
          getter: <testLibraryFragment>::@class::B::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
          returnType: List<int>
  topLevelVariables
    const hasInitializer a
      reference: <testLibrary>::@topLevelVariable::a
      firstFragment: <testLibraryFragment>::@topLevelVariable::a
      type: int
      constantInitializer
        fragment: <testLibraryFragment>::@topLevelVariable::a
        expression: expression_1
      getter: <testLibraryFragment>::@getter::a#element
  getters
    synthetic static get a
      firstFragment: <testLibraryFragment>::@getter::a
      returnType: int
''');
  }

  test_class_field_inferred_type_static_implicit_initialized() async {
    var library = await buildLibrary('class C { static var v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @21
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                _v
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
          returnType: int
      setters
        synthetic static set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
          returnType: void
''');
  }

  test_class_field_inheritedContextType_double() async {
    var library = await buildLibrary('''
abstract class A {
  const A();
  double get foo;
}
class B extends A {
  const B();
  final foo = 2;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 27
          getters
            get foo @45
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @58
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer foo @93
              reference: <testLibraryFragment>::@class::B::@field::foo
              element: <testLibraryFragment>::@class::B::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 2 @99
                  staticType: double
              getter2: <testLibraryFragment>::@class::B::@getter::foo
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 80
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::B::@getter::foo
              element: <testLibraryFragment>::@class::B::@getter::foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: double
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::B::@field::foo
          type: double
          constantInitializer
            fragment: <testLibraryFragment>::@class::B::@field::foo
            expression: expression_0
          getter: <testLibraryFragment>::@class::B::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::foo
          returnType: double
''');
  }

  test_class_field_isPromotable_abstractGetter() async {
    var library = await buildLibrary(r'''
abstract class A {
  int? get _foo;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic _foo
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get _foo @30
              reference: <testLibraryFragment>::@class::A::@getter::_foo
              element: <testLibraryFragment>::@class::A::@getter::_foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
          returnType: int?
''');
  }

  test_class_field_isPromotable_hasGetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibraryFragment>::@class::B::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_hasGetter_abstract() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_hasGetter_inPart() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibrary>::@fragment::package:test/a.dart::@class::B::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_hasGetter_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_hasNotFinalField() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::B::@field::_foo#element
''');
  }

  test_class_field_isPromotable_hasNotFinalField_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_hasSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  set _field(int? _) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_language217() async {
    var library = await buildLibrary(r'''
// @dart = 2.19
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInMixin() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B {
  final int? _foo = 0;
}

/// `_foo` is implemented in [M].
class C with M implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@mixin::M::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInSuperclass() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

class C {
  final int? _foo = 0;
}

/// `_foo` is implemented in [B].
class D extends B implements C {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inClassTypeAlias() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

mixin M {
  dynamic noSuchMethod(Invocation invocation) {}
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class E = Object with M implements B;
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@class::E
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inEnum() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
enum E implements B {
  v;
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@enum::E
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_getter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inDifferentLibrary() async {
    newFile('$testPackageLibPath/a.dart', r'''
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
import 'a.dart';

class A {
  final int? _foo;
  A(this._foo);
}

/// Has a noSuchMethod thrower for B._field, but since private names in
/// different libraries are distinct, this has no effect on promotion of
/// C._field.
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inheritedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B extends A {
  A(super.value);
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_mixedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B with M {}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@mixin::M::@getter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibrary>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_unusedMixin() async {
    // Mixins are implicitly abstract so the presence of a mixin that inherits
    // a field into its interface, and doesn't implement it, doesn't mean that
    // a noSuchMethod forwarder created for it. So,  this does not block that
    // field from promoting.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M implements A {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
''');
  }

  test_class_field_isPromotable_notFinal() async {
    var library = await buildLibrary(r'''
class A {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
          setter: <testLibraryFragment>::@class::A::@setter::_foo#element
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::A::@field::_foo#element
''');
  }

  test_class_field_isPromotable_notPrivate() async {
    var library = await buildLibrary(r'''
class A {
  int? field;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        field
          firstFragment: <testLibraryFragment>::@class::A::@field::field
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::field#element
          setter: <testLibraryFragment>::@class::A::@setter::field#element
''');
  }

  test_class_field_isPromotable_typeInference() async {
    // We decide that `_foo` is promotable before inferring the type of `bar`.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  final bar = _foo != null ? _foo : 0;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
        final hasInitializer bar
          firstFragment: <testLibraryFragment>::@class::A::@field::bar
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::bar#element
''');
  }

  test_class_field_missingName() async {
    var library = await buildLibrary('''
abstract class C {
  Object a,;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            a @28
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
            <null-name> (offset=30)
              reference: <testLibraryFragment>::@class::C::@field::0
              element: <testLibraryFragment>::@class::C::@field::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::0
              setter2: <testLibraryFragment>::@class::C::@setter::0
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
            synthetic get <null-name>
              reference: <testLibraryFragment>::@class::C::@getter::0
              element: <testLibraryFragment>::@class::C::@getter::0#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                _a
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
            synthetic set <null-name>
              reference: <testLibraryFragment>::@class::C::@setter::0
              element: <testLibraryFragment>::@class::C::@setter::0#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::0::@parameter::_#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
        <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@field::0
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::0#element
          setter: <testLibraryFragment>::@class::C::@setter::0#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
          returnType: Object
        synthetic get <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@getter::0
          returnType: Object
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: Object
          returnType: void
        synthetic set <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@setter::0
          formalParameters
            requiredPositional _
              type: Object
          returnType: void
''');
  }

  test_class_field_ofGeneric_refEnclosingTypeParameter_false() async {
    var library = await buildLibrary('''
class C<T> {
  late int foo;
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            foo @24
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        late foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
''');
  }

  test_class_field_ofGeneric_refEnclosingTypeParameter_true() async {
    var library = await buildLibrary('''
class C<T> {
  late T foo;
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            foo @22
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        late foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          hasEnclosingTypeParameterReference: true
          type: T
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          hasEnclosingTypeParameterReference: true
          returnType: T
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          hasEnclosingTypeParameterReference: true
          formalParameters
            requiredPositional _foo
              type: T
          returnType: void
''');
  }

  test_class_field_propagatedType_const_noDep() async {
    var library = await buildLibrary('''
class C {
  static const x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @29
                  staticType: int
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static const hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          constantInitializer
            fragment: <testLibraryFragment>::@class::C::@field::x
            expression: expression_0
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_propagatedType_final_dep_inLib() async {
    newFile('$testPackageLibPath/a.dart', 'final a = 1;');
    var library = await buildLibrary('''
import "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class C @23
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer b @35
              reference: <testLibraryFragment>::@class::C::@field::b
              element: <testLibraryFragment>::@class::C::@field::b#element
              getter2: <testLibraryFragment>::@class::C::@getter::b
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get b
              reference: <testLibraryFragment>::@class::C::@getter::b
              element: <testLibraryFragment>::@class::C::@getter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer b
          firstFragment: <testLibraryFragment>::@class::C::@field::b
          type: double
          getter: <testLibraryFragment>::@class::C::@getter::b#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get b
          firstFragment: <testLibraryFragment>::@class::C::@getter::b
          returnType: double
''');
  }

  test_class_field_propagatedType_final_dep_inPart() async {
    newFile('$testPackageLibPath/a.dart', 'part of lib; final a = 1;');
    var library = await buildLibrary('''
library lib;
part "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  name: lib
  fragments
    <testLibraryFragment> (offset=8)
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class C @34
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer b @46
              reference: <testLibraryFragment>::@class::C::@field::b
              element: <testLibraryFragment>::@class::C::@field::b#element
              getter2: <testLibraryFragment>::@class::C::@getter::b
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get b
              reference: <testLibraryFragment>::@class::C::@getter::b
              element: <testLibraryFragment>::@class::C::@getter::b#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      topLevelVariables
        hasInitializer a @19
          reference: <testLibrary>::@fragment::package:test/a.dart::@topLevelVariable::a
          element: <testLibrary>::@topLevelVariable::a
          getter2: <testLibrary>::@fragment::package:test/a.dart::@getter::a
      getters
        synthetic get a
          reference: <testLibrary>::@fragment::package:test/a.dart::@getter::a
          element: <testLibrary>::@fragment::package:test/a.dart::@getter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer b
          firstFragment: <testLibraryFragment>::@class::C::@field::b
          type: double
          getter: <testLibraryFragment>::@class::C::@getter::b#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get b
          firstFragment: <testLibraryFragment>::@class::C::@getter::b
          returnType: double
  topLevelVariables
    final hasInitializer a
      reference: <testLibrary>::@topLevelVariable::a
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@topLevelVariable::a
      type: int
      getter: <testLibrary>::@fragment::package:test/a.dart::@getter::a#element
  getters
    synthetic static get a
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@getter::a
      returnType: int
''');
  }

  test_class_field_propagatedType_final_noDep_instance() async {
    var library = await buildLibrary('''
class C {
  final x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_propagatedType_final_noDep_static() async {
    var library = await buildLibrary('''
class C {
  static final x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_static() async {
    var library = await buildLibrary('class C { static int i; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @21
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                _i
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
      setters
        synthetic static set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
          returnType: void
''');
  }

  test_class_field_static_final_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  static final f = 0;
  const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer f @25
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              getter2: <testLibraryFragment>::@class::C::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 40
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::C::@getter::f
              element: <testLibraryFragment>::@class::C::@getter::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get f
          firstFragment: <testLibraryFragment>::@class::C::@getter::f
          returnType: int
''');
  }

  test_class_field_static_final_untyped() async {
    var library = await buildLibrary('class C { static final x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @23
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_field_static_late() async {
    var library = await buildLibrary('class C { static late int i; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @26
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                _i
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static late i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
      setters
        synthetic static set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
          returnType: void
''');
  }

  test_class_field_type_explicit() async {
    var library = await buildLibrary(r'''
class C {
  int a = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                _a
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
          returnType: int
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: int
          returnType: void
''');
  }

  test_class_field_type_inferred_fromInitializer() async {
    var library = await buildLibrary(r'''
class C {
  var foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
''');
  }

  test_class_field_type_inferred_fromSuper() async {
    var library = await buildLibrary(r'''
abstract class A {
  int get foo;
}

class B extends A {
  final foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @29
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @43
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer foo @65
              reference: <testLibraryFragment>::@class::B::@field::foo
              element: <testLibraryFragment>::@class::B::@field::foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::B::@getter::foo
              element: <testLibraryFragment>::@class::B::@getter::foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::B::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::foo
          returnType: int
''');
  }

  test_class_field_type_inferred_Never() async {
    var library = await buildLibrary(r'''
class C {
  var a = throw 42;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                _a
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: Never
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
          returnType: Never
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: Never
          returnType: void
''');
  }

  test_class_field_typed() async {
    var library = await buildLibrary('class C { int x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_field_untyped() async {
    var library = await buildLibrary('class C { var x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
''');
  }

  test_class_fields() async {
    var library = await buildLibrary('class C { int i; int j; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @14
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
            j @21
              reference: <testLibraryFragment>::@class::C::@field::j
              element: <testLibraryFragment>::@class::C::@field::j#element
              getter2: <testLibraryFragment>::@class::C::@getter::j
              setter2: <testLibraryFragment>::@class::C::@setter::j
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
            synthetic get j
              reference: <testLibraryFragment>::@class::C::@getter::j
              element: <testLibraryFragment>::@class::C::@getter::j#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                _i
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
            synthetic set j
              reference: <testLibraryFragment>::@class::C::@setter::j
              element: <testLibraryFragment>::@class::C::@setter::j#element
              formalParameters
                _j
                  element: <testLibraryFragment>::@class::C::@setter::j::@parameter::_j#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
        j
          firstFragment: <testLibraryFragment>::@class::C::@field::j
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::j#element
          setter: <testLibraryFragment>::@class::C::@setter::j#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
          returnType: int
        synthetic get j
          firstFragment: <testLibraryFragment>::@class::C::@getter::j
          returnType: int
      setters
        synthetic set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
          returnType: void
        synthetic set j
          firstFragment: <testLibraryFragment>::@class::C::@setter::j
          formalParameters
            requiredPositional _j
              type: int
          returnType: void
''');
  }

  test_class_fields_late() async {
    var library = await buildLibrary('''
class C {
  late int foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            foo @21
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
''');
  }

  test_class_fields_late_final() async {
    var library = await buildLibrary('''
class C {
  late final int foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late final foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
''');
  }

  test_class_fields_late_final_initialized() async {
    var library = await buildLibrary('''
class C {
  late final int foo = 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
''');
  }

  test_class_fields_late_inference_usingSuper_methodInvocation() async {
    var library = await buildLibrary('''
class A {
  int foo() => 0;
}

class B extends A {
  late var f = super.foo();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @16
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @62
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
              setter2: <testLibraryFragment>::@class::B::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::B::@setter::f
              element: <testLibraryFragment>::@class::B::@setter::f#element
              formalParameters
                _f
                  element: <testLibraryFragment>::@class::B::@setter::f::@parameter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
          returnType: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        late hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::f#element
          setter: <testLibraryFragment>::@class::B::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
          returnType: int
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::B::@setter::f
          formalParameters
            requiredPositional _f
              type: int
          returnType: void
''');
  }

  test_class_fields_late_inference_usingSuper_propertyAccess() async {
    var library = await buildLibrary('''
class A {
  int get foo => 0;
}

class B extends A {
  late var f = super.foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @39
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @64
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
              setter2: <testLibraryFragment>::@class::B::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::B::@setter::f
              element: <testLibraryFragment>::@class::B::@setter::f#element
              formalParameters
                _f
                  element: <testLibraryFragment>::@class::B::@setter::f::@parameter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        late hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::f#element
          setter: <testLibraryFragment>::@class::B::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
          returnType: int
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::B::@setter::f
          formalParameters
            requiredPositional _f
              type: int
          returnType: void
''');
  }

  test_class_final() async {
    var library = await buildLibrary('final class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    final class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_getter_abstract() async {
    var library = await buildLibrary('abstract class C { int get x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        abstract get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_getter_external() async {
    var library = await buildLibrary('class C { external int get x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        external get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_getter_implicit_return_type() async {
    var library = await buildLibrary('class C { get x => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @14
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: dynamic
''');
  }

  test_class_getter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
''');
  }

  test_class_getter_invokesSuperSelf_getter_nestedInAssignment() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    (super.foo).foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
''');
  }

  test_class_getter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
          returnType: int
''');
  }

  test_class_getter_missingName() async {
    var library = await buildLibrary('''
class A {
  get () => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            get @12
              reference: <testLibraryFragment>::@class::A::@method::get
              element: <testLibraryFragment>::@class::A::@method::get#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        get
          firstFragment: <testLibraryFragment>::@class::A::@method::get
          returnType: dynamic
''');
  }

  test_class_getter_native() async {
    var library = await buildLibrary('''
class C {
  int get x() native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        external get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_getter_ofGeneric_refEnclosingTypeParameter_false() async {
    var library = await buildLibrary('''
class C<T> {
  int get foo {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
          getters
            get foo @23
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
''');
  }

  test_class_getter_ofGeneric_refEnclosingTypeParameter_true() async {
    var library = await buildLibrary('''
class C<T> {
  T get foo {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
          getters
            get foo @21
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          hasEnclosingTypeParameterReference: true
          type: T
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          hasEnclosingTypeParameterReference: true
          returnType: T
''');
  }

  test_class_getter_static() async {
    var library = await buildLibrary('class C { static int get x => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @25
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
''');
  }

  test_class_getters() async {
    var library =
        await buildLibrary('class C { int get x => null; get y => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
            synthetic y
              reference: <testLibraryFragment>::@class::C::@field::y
              element: <testLibraryFragment>::@class::C::@field::y#element
              getter2: <testLibraryFragment>::@class::C::@getter::y
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @18
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
            get y @33
              reference: <testLibraryFragment>::@class::C::@getter::y
              element: <testLibraryFragment>::@class::C::@getter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
        synthetic y
          firstFragment: <testLibraryFragment>::@class::C::@field::y
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::y#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
        get y
          firstFragment: <testLibraryFragment>::@class::C::@getter::y
          returnType: dynamic
''');
  }

  test_class_implicitField_getterFirst() async {
    var library = await buildLibrary('''
class C {
  int get x => 0;
  void set x(int value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            set x @39
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @45
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_implicitField_setterFirst() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  int get x => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @47
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: int
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_interface() async {
    var library = await buildLibrary('interface class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @16
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    interface class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_interfaces() async {
    var library = await buildLibrary('''
class C implements D, E {}
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @33
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @44
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        D
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_interfaces_extensionType() async {
    var library = await buildLibrary('''
class A {}
extension type B(int it) {}
class C {}
class D implements A, B, C {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
        class C @45
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
        class D @56
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
      extensionTypes
        extension type B @26
          reference: <testLibraryFragment>::@extensionType::B
          element: <testLibrary>::@extensionType::B
          fields
            it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              element: <testLibraryFragment>::@extensionType::B::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::B::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              element: <testLibraryFragment>::@extensionType::B::@getter::it#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      interfaces
        A
        C
  extensionTypes
    extension type B
      reference: <testLibrary>::@extensionType::B
      firstFragment: <testLibraryFragment>::@extensionType::B
      representation: <testLibraryFragment>::@extensionType::B::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::B::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::B::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::B::@getter::it
          returnType: int
''');
  }

  test_class_interfaces_Function() async {
    var library = await buildLibrary('''
class A {}
class B {}
class C implements A, Function, B {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @28
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        A
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_interfaces_unresolved() async {
    var library = await buildLibrary(
        'class C implements X, Y, Z {} class X {} class Z {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class X @36
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
        class Z @47
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        X
        Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
''');
  }

  test_class_lazy_constructors() async {
    var library = await buildLibrary('''
class A {
  A.named();
}
''');

    var constructors = library.getClass2('A')!.constructors2;
    expect(constructors, hasLength(1));
  }

  test_class_lazy_fields() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var fields = library.getClass2('A')!.fields2;
    expect(fields, hasLength(1));
  }

  test_class_lazy_getters() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var getters = library.getClass2('A')!.getters2;
    expect(getters, hasLength(1));
  }

  test_class_lazy_methods() async {
    var library = await buildLibrary('''
class A {
  void foo() {}
}
''');

    var methods = library.getClass2('A')!.methods2;
    expect(methods, hasLength(1));
  }

  test_class_lazy_setters() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var setters = library.getClass2('A')!.setters2;
    expect(setters, hasLength(1));
  }

  test_class_method_abstract() async {
    var library = await buildLibrary('abstract class C { f(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        abstract f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: dynamic
''');
  }

  test_class_method_async() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Future f() async {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        dart:async
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @40 async
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: Future<dynamic>
''');
  }

  test_class_method_asyncStar() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Stream f() async* {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        dart:async
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @40 async*
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: Stream<dynamic>
''');
  }

  test_class_method_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  f() {}
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @34
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              documentationComment: /**\n   * Docs\n   */
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          documentationComment: /**\n   * Docs\n   */
          returnType: dynamic
''');
  }

  test_class_method_external() async {
    var library = await buildLibrary('class C { external f(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: dynamic
''');
  }

  test_class_method_hasImplicitReturnType_false() async {
    var library = await buildLibrary('''
class C {
  int m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isFalse);
  }

  test_class_method_hasImplicitReturnType_true() async {
    var library = await buildLibrary('''
class C {
  m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isTrue);
  }

  test_class_method_inferred_type_nonStatic_implicit_param() async {
    var library = await buildLibrary('class C extends D { void f(value) {} }'
        ' abstract class D { void f(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @25
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::value#element
        class D @54
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          methods
            f @63
              reference: <testLibraryFragment>::@class::D::@method::f
              element: <testLibraryFragment>::@class::D::@method::f#element
              formalParameters
                value @69
                  element: <testLibraryFragment>::@class::D::@method::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          formalParameters
            requiredPositional hasImplicitType value
              type: int
          returnType: void
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      methods
        abstract f
          firstFragment: <testLibraryFragment>::@class::D::@method::f
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_method_inferred_type_nonStatic_implicit_return() async {
    var library = await buildLibrary('''
class C extends D {
  f() => null;
}
abstract class D {
  int f();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @22
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
        class D @52
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          methods
            f @62
              reference: <testLibraryFragment>::@class::D::@method::f
              element: <testLibraryFragment>::@class::D::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: int
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      methods
        abstract f
          firstFragment: <testLibraryFragment>::@class::D::@method::f
          returnType: int
''');
  }

  test_class_method_invokesSuperSelf() async {
    var library = await buildLibrary(r'''
class A {
  void foo() {
    super.foo();
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @17 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
          returnType: void
''');
  }

  test_class_method_missingName() async {
    var library = await buildLibrary('''
class A {
  () {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            <null-name> (offset=12)
              reference: <testLibraryFragment>::@class::A::@method::0
              element: <testLibraryFragment>::@class::A::@method::0#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        
          firstFragment: <testLibraryFragment>::@class::A::@method::0
          returnType: dynamic
''');
  }

  test_class_method_namedAsSupertype() async {
    var library = await buildLibrary(r'''
class A {}
class B extends A {
  void A() {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
          methods
            A @38
              reference: <testLibraryFragment>::@class::B::@method::A
              element: <testLibraryFragment>::@class::B::@method::A#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      methods
        A
          firstFragment: <testLibraryFragment>::@class::B::@method::A
          returnType: void
''');
  }

  test_class_method_native() async {
    var library = await buildLibrary('''
class C {
  int m() native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            m @16
              reference: <testLibraryFragment>::@class::C::@method::m
              element: <testLibraryFragment>::@class::C::@method::m#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external m
          firstFragment: <testLibraryFragment>::@class::C::@method::m
          returnType: int
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_false_hide() async {
    var library = await buildLibrary('''
class C<T> {
  void foo<T>(T _) {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          methods
            foo @20
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
              typeParameters
                T @24
                  element: T@24
              formalParameters
                _ @29
                  element: <testLibraryFragment>::@class::C::@method::foo::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          typeParameters
            T
          formalParameters
            requiredPositional _
              type: T
          returnType: void
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_formalParameter() async {
    var library = await buildLibrary('''
class C<T> {
  void foo(T _) {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          methods
            foo @20
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
              formalParameters
                _ @26
                  element: <testLibraryFragment>::@class::C::@method::foo::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          hasEnclosingTypeParameterReference: true
          formalParameters
            requiredPositional _
              type: T
          returnType: void
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_formalParameter2() async {
    var library = await buildLibrary('''
class C<T> {
  void foo(void Function(T) _) {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          methods
            foo @20
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
              formalParameters
                _ @41
                  element: <testLibraryFragment>::@class::C::@method::foo::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          hasEnclosingTypeParameterReference: true
          formalParameters
            requiredPositional _
              type: void Function(T)
          returnType: void
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_inferred() async {
    var library = await buildLibrary('''
class A<U> {
  U foo() {}
}

class B<T> extends A<T> {
  foo() {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            U @8
              element: U@8
          methods
            foo @17
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
        class B @35
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @37
              element: T@37
          methods
            foo @57
              reference: <testLibraryFragment>::@class::B::@method::foo
              element: <testLibraryFragment>::@class::B::@method::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        U
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
          hasEnclosingTypeParameterReference: true
          returnType: U
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
      supertype: A<T>
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::B::@method::foo
          hasEnclosingTypeParameterReference: true
          returnType: T
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_returnType() async {
    var library = await buildLibrary('''
class C<T> {
  T foo() {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          methods
            foo @17
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          hasEnclosingTypeParameterReference: true
          returnType: T
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_typeAlias() async {
    var library = await buildLibrary('''
typedef MyInt<U> = int;

class C<T> {
  MyInt<T> foo() {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @31
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @33
              element: T@33
          methods
            foo @49
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
      typeAliases
        MyInt @8
          reference: <testLibraryFragment>::@typeAlias::MyInt
          element: <testLibrary>::@typeAlias::MyInt
          typeParameters
            U @14
              element: U@14
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          hasEnclosingTypeParameterReference: true
          returnType: int
            alias: <testLibrary>::@typeAlias::MyInt
              typeArguments
                T
  typeAliases
    MyInt
      firstFragment: <testLibraryFragment>::@typeAlias::MyInt
      typeParameters
        U
      aliasedType: int
''');
  }

  test_class_method_ofGeneric_refEnclosingTypeParameter_true_typeParameter() async {
    var library = await buildLibrary('''
class C<T> {
  void foo<U extends T>() {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          methods
            foo @20
              reference: <testLibraryFragment>::@class::C::@method::foo
              element: <testLibraryFragment>::@class::C::@method::foo#element
              typeParameters
                U @24
                  element: U@24
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      methods
        foo
          firstFragment: <testLibraryFragment>::@class::C::@method::foo
          hasEnclosingTypeParameterReference: true
          typeParameters
            U
              bound: T
          returnType: void
''');
  }

  test_class_method_params() async {
    var library = await buildLibrary('class C { f(x, y) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              formalParameters
                x @12
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::x#element
                y @15
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          formalParameters
            requiredPositional hasImplicitType x
              type: dynamic
            requiredPositional hasImplicitType y
              type: dynamic
          returnType: dynamic
''');
  }

  test_class_method_static() async {
    var library = await buildLibrary('class C { static f() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @17
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        static f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: dynamic
''');
  }

  test_class_method_syncStar() async {
    var library = await buildLibrary(r'''
class C {
  Iterable<int> f() sync* {
    yield 42;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @26 sync*
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: Iterable<int>
''');
  }

  test_class_method_type_parameter() async {
    var library = await buildLibrary('class C { T f<T, U>(U u) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @12
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                T @14
                  element: T@14
                U @17
                  element: U@17
              formalParameters
                u @22
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::u#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          typeParameters
            T
            U
          formalParameters
            requiredPositional u
              type: U
          returnType: T
''');
  }

  test_class_method_type_parameter_in_generic_class() async {
    var library = await buildLibrary('''
class C<T, U> {
  V f<V, W>(T t, U u, W w) => null;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @20
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                V @22
                  element: V@22
                W @25
                  element: W@25
              formalParameters
                t @30
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::t#element
                u @35
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::u#element
                w @40
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::w#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          hasEnclosingTypeParameterReference: true
          typeParameters
            V
            W
          formalParameters
            requiredPositional t
              type: T
            requiredPositional u
              type: U
            requiredPositional w
              type: W
          returnType: V
''');
  }

  test_class_method_type_parameter_with_function_typed_parameter() async {
    var library = await buildLibrary('class C { void f<T, U>(T x(U u)) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @15
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                T @17
                  element: T@17
                U @20
                  element: U@20
              formalParameters
                x @25
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          typeParameters
            T
            U
          formalParameters
            requiredPositional x
              type: T Function(U)
              formalParameters
                requiredPositional u
                  type: U
          returnType: void
''');
  }

  test_class_methods() async {
    var library = await buildLibrary('class C { f() {} g() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
            g @17
              reference: <testLibraryFragment>::@class::C::@method::g
              element: <testLibraryFragment>::@class::C::@method::g#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: dynamic
        g
          firstFragment: <testLibraryFragment>::@class::C::@method::g
          returnType: dynamic
''');
  }

  test_class_missingName() async {
    var library = await buildLibrary(r'''
class {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class <null-name> (offset=6)
          reference: <testLibraryFragment>::@class::0
          element: <testLibrary>::@class::0
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::0::@constructor::new
              element: <testLibraryFragment>::@class::0::@constructor::new#element
              typeName: null
  classes
    class 
      reference: <testLibrary>::@class::0
      firstFragment: <testLibraryFragment>::@class::0
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::0::@constructor::new
''');
  }

  test_class_mixin_class() async {
    var library = await buildLibrary('mixin class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    mixin class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_mixins() async {
    var library = await buildLibrary('''
class C extends D with E, F, G {}
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @40
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @51
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
        class F @62
          reference: <testLibraryFragment>::@class::F
          element: <testLibrary>::@class::F
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::F::@constructor::new
              element: <testLibraryFragment>::@class::F::@constructor::new#element
              typeName: F
        class G @73
          reference: <testLibraryFragment>::@class::G
          element: <testLibrary>::@class::G
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::G::@constructor::new
              element: <testLibraryFragment>::@class::G::@constructor::new#element
              typeName: G
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
        F
        G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
    class F
      reference: <testLibrary>::@class::F
      firstFragment: <testLibraryFragment>::@class::F
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::F::@constructor::new
    class G
      reference: <testLibrary>::@class::G
      firstFragment: <testLibraryFragment>::@class::G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::G::@constructor::new
''');
  }

  test_class_mixins_extensionType() async {
    var library = await buildLibrary('''
mixin A {}
extension type B(int it) {}
mixin C {}
class D extends Object with A, B, C {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class D @56
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
      extensionTypes
        extension type B @26
          reference: <testLibraryFragment>::@extensionType::B
          element: <testLibrary>::@extensionType::B
          fields
            it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              element: <testLibraryFragment>::@extensionType::B::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::B::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              element: <testLibraryFragment>::@extensionType::B::@getter::it#element
      mixins
        mixin A @6
          reference: <testLibraryFragment>::@mixin::A
          element: <testLibrary>::@mixin::A
        mixin C @45
          reference: <testLibraryFragment>::@mixin::C
          element: <testLibrary>::@mixin::C
  classes
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: Object
      mixins
        A
        C
  extensionTypes
    extension type B
      reference: <testLibrary>::@extensionType::B
      firstFragment: <testLibraryFragment>::@extensionType::B
      representation: <testLibraryFragment>::@extensionType::B::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::B::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::B::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::B::@getter::it
          returnType: int
  mixins
    mixin A
      reference: <testLibrary>::@mixin::A
      firstFragment: <testLibraryFragment>::@mixin::A
      superclassConstraints
        Object
    mixin C
      reference: <testLibrary>::@mixin::C
      firstFragment: <testLibraryFragment>::@mixin::C
      superclassConstraints
        Object
''');
  }

  test_class_mixins_generic() async {
    var library = await buildLibrary('''
class Z extends A with B<int>, C<double> {}
class A {}
class B<B1> {}
class C<C1> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Z @6
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
        class A @50
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            B1 @63
              element: B1@63
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @76
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            C1 @78
              element: C1@78
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      supertype: A
      mixins
        B<int>
        C<double>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        B1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        C1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_mixins_generic_superAfter() async {
    var library = await buildLibrary('''
mixin M<T extends num> {}
mixin M2<T extends num> on M<T> {}
class Z extends S with M2 {}
class S with M<int> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Z @67
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
        class S @96
          reference: <testLibraryFragment>::@class::S
          element: <testLibrary>::@class::S
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::S::@constructor::new
              element: <testLibraryFragment>::@class::S::@constructor::new#element
              typeName: S
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          typeParameters
            T @8
              element: T@8
        mixin M2 @32
          reference: <testLibraryFragment>::@mixin::M2
          element: <testLibrary>::@mixin::M2
          typeParameters
            T @35
              element: T@35
  classes
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      supertype: S
      mixins
        M2<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
          superConstructor: <testLibraryFragment>::@class::S::@constructor::new#element
    class S
      reference: <testLibrary>::@class::S
      firstFragment: <testLibraryFragment>::@class::S
      supertype: Object
      mixins
        M<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::S::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      typeParameters
        T
          bound: num
      superclassConstraints
        Object
    mixin M2
      reference: <testLibrary>::@mixin::M2
      firstFragment: <testLibraryFragment>::@mixin::M2
      typeParameters
        T
          bound: num
      superclassConstraints
        M<T>
''');
  }

  test_class_mixins_genericMixin_tooManyArguments() async {
    var library = await buildLibrary('''
mixin M<T> {}
class A extends Object with M<int, String> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @20
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          typeParameters
            T @8
              element: T@8
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      supertype: Object
      mixins
        M<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      typeParameters
        T
      superclassConstraints
        Object
''');
  }

  test_class_mixins_typeParameter() async {
    var library = await buildLibrary('''
mixin M1 {}
mixin M2 {}
class A<T> extends Object with M1, T<int>, M2 {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @30
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @32
              element: T@32
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      mixins
        mixin M1 @6
          reference: <testLibraryFragment>::@mixin::M1
          element: <testLibrary>::@mixin::M1
        mixin M2 @18
          reference: <testLibraryFragment>::@mixin::M2
          element: <testLibrary>::@mixin::M2
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      supertype: Object
      mixins
        M1
        M2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  mixins
    mixin M1
      reference: <testLibrary>::@mixin::M1
      firstFragment: <testLibraryFragment>::@mixin::M1
      superclassConstraints
        Object
    mixin M2
      reference: <testLibrary>::@mixin::M2
      firstFragment: <testLibraryFragment>::@mixin::M2
      superclassConstraints
        Object
''');
  }

  test_class_mixins_unresolved() async {
    var library = await buildLibrary(
        'class C extends Object with X, Y, Z {} class X {} class Z {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class X @45
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
        class Z @56
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        X
        Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
''');
  }

  test_class_notSimplyBounded_circularity_via_typeAlias_recordType() async {
    var library = await buildLibrary('''
class C<T extends A> {}
typedef A = (C, int);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        A @32
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      aliasedType: (C<dynamic>, int)
''');
  }

  test_class_notSimplyBounded_circularity_via_typedef() async {
    // C's type parameter T is not simply bounded because its bound, F, expands
    // to `dynamic F(C)`, which refers to C.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(C value);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        F @32
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      aliasedType: dynamic Function(C<dynamic>)
''');
  }

  test_class_notSimplyBounded_circularity_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // C, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends C<dynamic>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_class() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends C> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: T@32
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    notSimplyBounded class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_functionType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = void Function();
typedef D<T extends C> = void Function();
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      typeAliases
        C @8
          reference: <testLibraryFragment>::@typeAlias::C
          element: <testLibrary>::@typeAlias::C
          typeParameters
            T @10
              element: T@10
        D @50
          reference: <testLibraryFragment>::@typeAlias::D
          element: <testLibrary>::@typeAlias::D
          typeParameters
            T @52
              element: T@52
  typeAliases
    notSimplyBounded C
      firstFragment: <testLibraryFragment>::@typeAlias::C
      typeParameters
        T
          bound: dynamic
      aliasedType: void Function()
    notSimplyBounded D
      firstFragment: <testLibraryFragment>::@typeAlias::D
      typeParameters
        T
          bound: dynamic
      aliasedType: void Function()
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_interfaceType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = List<T>;
typedef D<T extends C> = List<T>;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      typeAliases
        C @8
          reference: <testLibraryFragment>::@typeAlias::C
          element: <testLibrary>::@typeAlias::C
          typeParameters
            T @10
              element: T@10
        D @42
          reference: <testLibraryFragment>::@typeAlias::D
          element: <testLibrary>::@typeAlias::D
          typeParameters
            T @44
              element: T@44
  typeAliases
    notSimplyBounded C
      firstFragment: <testLibraryFragment>::@typeAlias::C
      typeParameters
        T
          bound: dynamic
      aliasedType: List<T>
    notSimplyBounded D
      firstFragment: <testLibraryFragment>::@typeAlias::D
      typeParameters
        T
          bound: dynamic
      aliasedType: List<T>
''');
  }

  test_class_notSimplyBounded_complex_by_reference_to_cycle() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends D> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: T@32
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    notSimplyBounded class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_use_of_parameter() async {
    var library = await buildLibrary('''
class C<T extends D<T>> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @33
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @35
              element: T@35
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<T>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_dependency_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // non-simply-bounded type D, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends D<dynamic>> {}
class D<T extends D<T>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @39
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @41
              element: T@41
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    notSimplyBounded class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: D<T>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_parameter_type() async {
    var library = await buildLibrary('''
class C<T extends void Function(T)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function(T)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_return_type() async {
    var library = await buildLibrary('''
class C<T extends T Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: T Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_simple() async {
    var library = await buildLibrary('''
class C<T extends void Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_refers_to_circular_typedef() async {
    // C's type parameter T has a bound of F, which is a circular typedef.  This
    // is illegal in Dart, but we need to make sure it doesn't lead to a crash
    // or infinite loop.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(G value);
typedef G(F value);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        F @32
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
        G @52
          reference: <testLibraryFragment>::@typeAlias::G
          element: <testLibrary>::@typeAlias::G
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      aliasedType: dynamic Function(dynamic)
    notSimplyBounded G
      firstFragment: <testLibraryFragment>::@typeAlias::G
      aliasedType: dynamic Function(dynamic)
''');
  }

  test_class_notSimplyBounded_self() async {
    var library = await buildLibrary('''
class C<T extends C> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_because_non_generic() async {
    // If no type parameters are specified, then the class is simply bounded, so
    // there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_by_lack_of_cycles() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: T@32
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_by_syntax() async {
    // If no bounds are specified, then the class is simply bounded by syntax
    // alone, so there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_operator() async {
    var library =
        await buildLibrary('class C { C operator+(C other) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            + @20
              reference: <testLibraryFragment>::@class::C::@method::+
              element: <testLibraryFragment>::@class::C::@method::+#element
              formalParameters
                other @24
                  element: <testLibraryFragment>::@class::C::@method::+::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        +
          firstFragment: <testLibraryFragment>::@class::C::@method::+
          formalParameters
            requiredPositional other
              type: C
          returnType: C
''');
  }

  test_class_operator_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator==(Object other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            == @25
              reference: <testLibraryFragment>::@class::C::@method::==
              element: <testLibraryFragment>::@class::C::@method::==#element
              formalParameters
                other @35
                  element: <testLibraryFragment>::@class::C::@method::==::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        ==
          firstFragment: <testLibraryFragment>::@class::C::@method::==
          formalParameters
            requiredPositional other
              type: Object
          returnType: bool
''');
  }

  test_class_operator_external() async {
    var library =
        await buildLibrary('class C { external C operator+(C other); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            + @29
              reference: <testLibraryFragment>::@class::C::@method::+
              element: <testLibraryFragment>::@class::C::@method::+#element
              formalParameters
                other @33
                  element: <testLibraryFragment>::@class::C::@method::+::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external +
          firstFragment: <testLibraryFragment>::@class::C::@method::+
          formalParameters
            requiredPositional other
              type: C
          returnType: C
''');
  }

  test_class_operator_greater_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator>=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            >= @25
              reference: <testLibraryFragment>::@class::C::@method::>=
              element: <testLibraryFragment>::@class::C::@method::>=#element
              formalParameters
                other @30
                  element: <testLibraryFragment>::@class::C::@method::>=::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        >=
          firstFragment: <testLibraryFragment>::@class::C::@method::>=
          formalParameters
            requiredPositional other
              type: C
          returnType: bool
''');
  }

  test_class_operator_index() async {
    var library =
        await buildLibrary('class C { bool operator[](int i) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            [] @23
              reference: <testLibraryFragment>::@class::C::@method::[]
              element: <testLibraryFragment>::@class::C::@method::[]#element
              formalParameters
                i @30
                  element: <testLibraryFragment>::@class::C::@method::[]::@parameter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        []
          firstFragment: <testLibraryFragment>::@class::C::@method::[]
          formalParameters
            requiredPositional i
              type: int
          returnType: bool
''');
  }

  test_class_operator_index_set() async {
    var library = await buildLibrary('''
class C {
  void operator[]=(int i, bool v) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            []= @25
              reference: <testLibraryFragment>::@class::C::@method::[]=
              element: <testLibraryFragment>::@class::C::@method::[]=#element
              formalParameters
                i @33
                  element: <testLibraryFragment>::@class::C::@method::[]=::@parameter::i#element
                v @41
                  element: <testLibraryFragment>::@class::C::@method::[]=::@parameter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        []=
          firstFragment: <testLibraryFragment>::@class::C::@method::[]=
          formalParameters
            requiredPositional i
              type: int
            requiredPositional v
              type: bool
          returnType: void
''');
  }

  test_class_operator_less_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator<=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            <= @25
              reference: <testLibraryFragment>::@class::C::@method::<=
              element: <testLibraryFragment>::@class::C::@method::<=#element
              formalParameters
                other @30
                  element: <testLibraryFragment>::@class::C::@method::<=::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        <=
          firstFragment: <testLibraryFragment>::@class::C::@method::<=
          formalParameters
            requiredPositional other
              type: C
          returnType: bool
''');
  }

  test_class_operator_minus() async {
    var library = await buildLibrary('''
class A {
  int operator -(int other) => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            - @25
              reference: <testLibraryFragment>::@class::A::@method::-
              element: <testLibraryFragment>::@class::A::@method::-#element
              formalParameters
                other @31
                  element: <testLibraryFragment>::@class::A::@method::-::@parameter::other#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        -
          firstFragment: <testLibraryFragment>::@class::A::@method::-
          formalParameters
            requiredPositional other
              type: int
          returnType: int
''');
  }

  test_class_operator_minus_unary() async {
    var library = await buildLibrary('''
class A {
  int operator -() => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            - @25
              reference: <testLibraryFragment>::@class::A::@method::unary-
              element: <testLibraryFragment>::@class::A::@method::unary-#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        unary-
          firstFragment: <testLibraryFragment>::@class::A::@method::unary-
          returnType: int
''');
  }

  test_class_ref_nullability_none() async {
    var library = await buildLibrary('''
class C {}
C c;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      topLevelVariables
        c @13
          reference: <testLibraryFragment>::@topLevelVariable::c
          element: <testLibrary>::@topLevelVariable::c
          getter2: <testLibraryFragment>::@getter::c
          setter2: <testLibraryFragment>::@setter::c
      getters
        synthetic get c
          reference: <testLibraryFragment>::@getter::c
          element: <testLibraryFragment>::@getter::c#element
      setters
        synthetic set c
          reference: <testLibraryFragment>::@setter::c
          element: <testLibraryFragment>::@setter::c#element
          formalParameters
            _c
              element: <testLibraryFragment>::@setter::c::@parameter::_c#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  topLevelVariables
    c
      reference: <testLibrary>::@topLevelVariable::c
      firstFragment: <testLibraryFragment>::@topLevelVariable::c
      type: C
      getter: <testLibraryFragment>::@getter::c#element
      setter: <testLibraryFragment>::@setter::c#element
  getters
    synthetic static get c
      firstFragment: <testLibraryFragment>::@getter::c
      returnType: C
  setters
    synthetic static set c
      firstFragment: <testLibraryFragment>::@setter::c
      formalParameters
        requiredPositional _c
          type: C
      returnType: void
''');
  }

  test_class_ref_nullability_question() async {
    var library = await buildLibrary('''
class C {}
C? c;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      topLevelVariables
        c @14
          reference: <testLibraryFragment>::@topLevelVariable::c
          element: <testLibrary>::@topLevelVariable::c
          getter2: <testLibraryFragment>::@getter::c
          setter2: <testLibraryFragment>::@setter::c
      getters
        synthetic get c
          reference: <testLibraryFragment>::@getter::c
          element: <testLibraryFragment>::@getter::c#element
      setters
        synthetic set c
          reference: <testLibraryFragment>::@setter::c
          element: <testLibraryFragment>::@setter::c#element
          formalParameters
            _c
              element: <testLibraryFragment>::@setter::c::@parameter::_c#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  topLevelVariables
    c
      reference: <testLibrary>::@topLevelVariable::c
      firstFragment: <testLibraryFragment>::@topLevelVariable::c
      type: C?
      getter: <testLibraryFragment>::@getter::c#element
      setter: <testLibraryFragment>::@setter::c#element
  getters
    synthetic static get c
      firstFragment: <testLibraryFragment>::@getter::c
      returnType: C?
  setters
    synthetic static set c
      firstFragment: <testLibraryFragment>::@setter::c
      formalParameters
        requiredPositional _c
          type: C?
      returnType: void
''');
  }

  test_class_sealed() async {
    var library = await buildLibrary('sealed class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @13
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    abstract sealed class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_sealed_induced_base_extends_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @11
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @29
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    base class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_base_implements_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @11
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @29
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    base class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_sealed_induced_base_implements_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @12
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @30
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    final class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_sealed_induced_final_extends_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @12
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @30
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    final class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed final class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_final_with_base_mixin() async {
    var library = await buildLibrary('''
base mixin A {}
interface class B {}
sealed class C extends B with A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @32
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @50
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin A @11
          reference: <testLibraryFragment>::@mixin::A
          element: <testLibrary>::@mixin::A
  classes
    interface class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    abstract sealed final class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      mixins
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
  mixins
    base mixin A
      reference: <testLibrary>::@mixin::A
      firstFragment: <testLibraryFragment>::@mixin::A
      superclassConstraints
        Object
''');
  }

  test_class_sealed_induced_interface_extends_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @16
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    interface class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed interface class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_none_implements_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @16
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    interface class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_setter_abstract() async {
    var library =
        await buildLibrary('abstract class C { void set x(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @34
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        abstract set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_covariant() async {
    var library =
        await buildLibrary('class C { void set x(covariant int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @35
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        abstract set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional covariant value
              type: int
          returnType: void
''');
  }

  test_class_setter_external() async {
    var library =
        await buildLibrary('class C { external void set x(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @34
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        external set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_implicit_param_type() async {
    var library = await buildLibrary('class C { void set x(value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @21
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional hasImplicitType value
              type: dynamic
          returnType: void
''');
  }

  test_class_setter_implicit_return_type() async {
    var library = await buildLibrary('class C { set x(int value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @14
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @20
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_inferred_type_conflictingInheritance() async {
    var library = await buildLibrary('''
class A {
  int t;
}
class B extends A {
  double t;
}
class C extends A implements B {
}
class D extends C {
  void set t(p) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            t @16
              reference: <testLibraryFragment>::@class::A::@field::t
              element: <testLibraryFragment>::@class::A::@field::t#element
              getter2: <testLibraryFragment>::@class::A::@getter::t
              setter2: <testLibraryFragment>::@class::A::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get t
              reference: <testLibraryFragment>::@class::A::@getter::t
              element: <testLibraryFragment>::@class::A::@getter::t#element
          setters
            synthetic set t
              reference: <testLibraryFragment>::@class::A::@setter::t
              element: <testLibraryFragment>::@class::A::@setter::t#element
              formalParameters
                _t
                  element: <testLibraryFragment>::@class::A::@setter::t::@parameter::_t#element
        class B @27
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            t @50
              reference: <testLibraryFragment>::@class::B::@field::t
              element: <testLibraryFragment>::@class::B::@field::t#element
              getter2: <testLibraryFragment>::@class::B::@getter::t
              setter2: <testLibraryFragment>::@class::B::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
          getters
            synthetic get t
              reference: <testLibraryFragment>::@class::B::@getter::t
              element: <testLibraryFragment>::@class::B::@getter::t#element
          setters
            synthetic set t
              reference: <testLibraryFragment>::@class::B::@setter::t
              element: <testLibraryFragment>::@class::B::@setter::t#element
              formalParameters
                _t
                  element: <testLibraryFragment>::@class::B::@setter::t::@parameter::_t#element
        class C @61
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @96
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic t
              reference: <testLibraryFragment>::@class::D::@field::t
              element: <testLibraryFragment>::@class::D::@field::t#element
              setter2: <testLibraryFragment>::@class::D::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          setters
            set t @121
              reference: <testLibraryFragment>::@class::D::@setter::t
              element: <testLibraryFragment>::@class::D::@setter::t#element
              formalParameters
                p @123
                  element: <testLibraryFragment>::@class::D::@setter::t::@parameter::p#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        t
          firstFragment: <testLibraryFragment>::@class::A::@field::t
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::t#element
          setter: <testLibraryFragment>::@class::A::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get t
          firstFragment: <testLibraryFragment>::@class::A::@getter::t
          returnType: int
      setters
        synthetic set t
          firstFragment: <testLibraryFragment>::@class::A::@setter::t
          formalParameters
            requiredPositional _t
              type: int
          returnType: void
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        t
          firstFragment: <testLibraryFragment>::@class::B::@field::t
          type: double
          getter: <testLibraryFragment>::@class::B::@getter::t#element
          setter: <testLibraryFragment>::@class::B::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get t
          firstFragment: <testLibraryFragment>::@class::B::@getter::t
          returnType: double
      setters
        synthetic set t
          firstFragment: <testLibraryFragment>::@class::B::@setter::t
          formalParameters
            requiredPositional _t
              type: double
          returnType: void
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      fields
        synthetic t
          firstFragment: <testLibraryFragment>::@class::D::@field::t
          type: dynamic
          setter: <testLibraryFragment>::@class::D::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
      setters
        set t
          firstFragment: <testLibraryFragment>::@class::D::@setter::t
          formalParameters
            requiredPositional hasImplicitType p
              type: dynamic
          returnType: void
''');
  }

  test_class_setter_inferred_type_nonStatic_implicit_param() async {
    var library =
        await buildLibrary('class C extends D { void set f(value) {} }'
            ' abstract class D { void set f(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set f @29
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                value @31
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::value#element
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::D::@field::f
              element: <testLibraryFragment>::@class::D::@field::f#element
              setter2: <testLibraryFragment>::@class::D::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          setters
            set f @71
              reference: <testLibraryFragment>::@class::D::@setter::f
              element: <testLibraryFragment>::@class::D::@setter::f#element
              formalParameters
                value @77
                  element: <testLibraryFragment>::@class::D::@setter::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      setters
        set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional hasImplicitType value
              type: int
          returnType: void
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::D::@field::f
          type: int
          setter: <testLibraryFragment>::@class::D::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      setters
        abstract set f
          firstFragment: <testLibraryFragment>::@class::D::@setter::f
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_inferred_type_static_implicit_return() async {
    var library = await buildLibrary('''
class C {
  static set f(int value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set f @23
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                value @29
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        static set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_invalid_named_parameter() async {
    var library = await buildLibrary('class C { void set x({a}) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                default a @22
                  reference: <testLibraryFragment>::@class::C::@setter::x::@parameter::a
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            optionalNamed hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::C::@setter::x::@parameter::a
              type: dynamic
          returnType: void
''');
  }

  test_class_setter_invalid_no_parameter() async {
    var library = await buildLibrary('class C { void set x() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          returnType: void
''');
  }

  test_class_setter_invalid_optional_parameter() async {
    var library = await buildLibrary('class C { void set x([a]) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                default a @22
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            optionalPositional hasImplicitType a
              type: dynamic
          returnType: void
''');
  }

  test_class_setter_invalid_too_many_parameters() async {
    var library = await buildLibrary('class C { void set x(a, b) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                a @21
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
                b @24
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            requiredPositional hasImplicitType b
              type: dynamic
          returnType: void
''');
  }

  test_class_setter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @16
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @24
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
          returnType: void
''');
  }

  test_class_setter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @16
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @24
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
          returnType: void
''');
  }

  test_class_setter_missingName() async {
    var library = await buildLibrary('''
class A {
  set (int _) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            set @12
              reference: <testLibraryFragment>::@class::A::@method::set
              element: <testLibraryFragment>::@class::A::@method::set#element
              formalParameters
                _ @21
                  element: <testLibraryFragment>::@class::A::@method::set::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        set
          firstFragment: <testLibraryFragment>::@class::A::@method::set
          formalParameters
            requiredPositional _
              type: int
          returnType: dynamic
''');
  }

  test_class_setter_native() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        external set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setter_ofGeneric_refEnclosingTypeParameter_false() async {
    var library = await buildLibrary('''
class C<T> {
  set foo(int _) {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          setters
            set foo @19
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _ @27
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _
              type: int
          returnType: void
''');
  }

  test_class_setter_ofGeneric_refEnclosingTypeParameter_true() async {
    var library = await buildLibrary('''
class C<T> {
  set foo(T _) {}
}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          setters
            set foo @19
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _ @25
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          hasEnclosingTypeParameterReference: true
          type: T
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          hasEnclosingTypeParameterReference: true
          formalParameters
            requiredPositional _
              type: T
          returnType: void
''');
  }

  test_class_setter_static() async {
    var library =
        await buildLibrary('class C { static void set x(int value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @26
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @32
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        static set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
''');
  }

  test_class_setters() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  set y(value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
            synthetic y
              reference: <testLibraryFragment>::@class::C::@field::y
              element: <testLibraryFragment>::@class::C::@field::y#element
              setter2: <testLibraryFragment>::@class::C::@setter::y
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
            set y @43
              reference: <testLibraryFragment>::@class::C::@setter::y
              element: <testLibraryFragment>::@class::C::@setter::y#element
              formalParameters
                value @45
                  element: <testLibraryFragment>::@class::C::@setter::y::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
        synthetic y
          firstFragment: <testLibraryFragment>::@class::C::@field::y
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::y#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
          returnType: void
        set y
          firstFragment: <testLibraryFragment>::@class::C::@setter::y
          formalParameters
            requiredPositional hasImplicitType value
              type: dynamic
          returnType: void
''');
  }

  test_class_supertype() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
''');
  }

  test_class_supertype_dynamic() async {
    var library = await buildLibrary('''
class A extends dynamic {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_supertype_extensionType() async {
    var library = await buildLibrary('''
extension type A(int it) {}
class B extends A {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
      extensionTypes
        extension type A @15
          reference: <testLibraryFragment>::@extensionType::A
          element: <testLibrary>::@extensionType::A
          fields
            it @21
              reference: <testLibraryFragment>::@extensionType::A::@field::it
              element: <testLibraryFragment>::@extensionType::A::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::A::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::A::@getter::it
              element: <testLibraryFragment>::@extensionType::A::@getter::it#element
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
  extensionTypes
    extension type A
      reference: <testLibrary>::@extensionType::A
      firstFragment: <testLibraryFragment>::@extensionType::A
      representation: <testLibraryFragment>::@extensionType::A::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::A::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::A::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::A::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::A::@getter::it
          returnType: int
''');
  }

  test_class_supertype_genericClass() async {
    var library = await buildLibrary('''
class C extends D<int, double> {}
class D<T1, T2> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @40
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T1 @42
              element: T1@42
            T2 @46
              element: T2@46
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D<int, double>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T1
        T2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_supertype_genericClass_tooManyArguments() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<int, String> {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
        class B @20
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<dynamic>
''');
  }

  test_class_supertype_typeArguments_self() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<B> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @20
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<B>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_supertype_typeParameter() async {
    var library = await buildLibrary('''
class A<T> extends T<int> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_supertype_unresolved() async {
    var library = await buildLibrary('class C extends D {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @11
              element: U@11
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_bound() async {
    var library = await buildLibrary('''
class C<T extends Object, U extends D> {}
class D {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @26
              element: U@26
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @48
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: Object
        U
          bound: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_typeParameters_cycle_1of1() async {
    var library = await buildLibrary('class C<T extends T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_cycle_2of3() async {
    var library = await buildLibrary(r'''
class C<T extends V, U, V extends T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @21
              element: U@21
            V @24
              element: V@24
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
        U
        V
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType() async {
    var library = await buildLibrary(r'''
class A<T extends void Function(A)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
          bound: void Function(A<dynamic>)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType2() async {
    var library = await buildLibrary(r'''
class C<T extends void Function<U extends C>()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function<U extends C<dynamic>>()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_contravariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = void Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @40
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @42
              element: X@42
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: void Function(X)
            alias: <testLibrary>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: void Function(X)
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_covariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function();

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @38
              element: X@38
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
            alias: <testLibrary>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function()
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @37
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @39
              element: X@39
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
            alias: <testLibrary>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function(X)
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant_legacy() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @37
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @39
              element: X@39
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
            alias: <testLibrary>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function(X)
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_both() async {
    var library = await buildLibrary(r'''
class A<X extends X Function(X)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: X@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_contravariant() async {
    var library = await buildLibrary(r'''
class A<X extends void Function(X)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: X@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: void Function(X)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: X@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant_legacy() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: X@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    notSimplyBounded class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_contravariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = List<void Function(X)>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @46
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            X @48
              element: X@48
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        X
          bound: List<void Function(X)>
            alias: <testLibrary>::@typeAlias::A
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        X
      aliasedType: List<void Function(X)>
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_covariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = Map<X, int>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @35
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            X @37
              element: X@37
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            X @10
              element: X@10
  classes
    notSimplyBounded class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        X
          bound: Map<X, int>
            alias: <testLibrary>::@typeAlias::A
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        X
      aliasedType: Map<X, int>
''');
  }

  test_class_typeParameters_f_bound_complex() async {
    var library = await buildLibrary('class C<T extends List<U>, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @27
              element: U@27
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: List<U>
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_f_bound_simple() async {
    var library = await buildLibrary('class C<T extends U, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
            U @21
              element: U@21
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    notSimplyBounded class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: U
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_missingName() async {
    var library = await buildLibrary(r'''
class A<T,> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
            <null-name> (offset=10)
              element: null@null
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
        <null-name>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_variance_contravariant() async {
    var library = await buildLibrary('class C<in T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @11
              element: T@11
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_covariant() async {
    var library = await buildLibrary('class C<out T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @12
              element: T@12
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_invariant() async {
    var library = await buildLibrary('class C<inout T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @14
              element: T@14
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_multiple() async {
    var library = await buildLibrary('class C<inout T, in U, out V> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @14
              element: T@14
            U @20
              element: U@20
            V @27
              element: V@27
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
        V
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_classAlias() async {
    var library = await buildLibrary('''
class C = D with E, F, G;
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @32
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @43
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
        class F @54
          reference: <testLibraryFragment>::@class::F
          element: <testLibrary>::@class::F
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::F::@constructor::new
              element: <testLibraryFragment>::@class::F::@constructor::new#element
              typeName: F
        class G @65
          reference: <testLibraryFragment>::@class::G
          element: <testLibrary>::@class::G
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::G::@constructor::new
              element: <testLibraryFragment>::@class::G::@constructor::new#element
              typeName: G
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
        F
        G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
    class F
      reference: <testLibrary>::@class::F
      firstFragment: <testLibraryFragment>::@class::F
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::F::@constructor::new
    class G
      reference: <testLibrary>::@class::G
      firstFragment: <testLibraryFragment>::@class::G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::G::@constructor::new
''');
  }

  test_classAlias_abstract() async {
    var library = await buildLibrary('''
abstract class C = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @35
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @46
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    abstract class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_base() async {
    var library = await buildLibrary('''
base class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @11
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin M @36
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    base class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_constructors_beforeOtherProperties() async {
    // https://github.com/dart-lang/sdk/issues/57035
    var library = await buildLibrary('''
abstract mixin class A {}
mixin M {}
class X = A with M;
''');

    var X = library.getClass2('X')!;
    expect(X.constructors2, hasLength(1));
  }

  test_classAlias_constructors_default() async {
    var library = await buildLibrary('''
class A {}
mixin class M {}
class X = A with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class M @23
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class X @34
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    mixin class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      supertype: A
      mixins
        M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_classAlias_constructors_dependencies() async {
    var library = await buildLibrary('''
class A {
  A(int i);
}
mixin class M1 {}
mixin class M2 {}

class C2 = C1 with M2;
class C1 = A with M1;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                i @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::i#element
        class M1 @36
          reference: <testLibraryFragment>::@class::M1
          element: <testLibrary>::@class::M1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M1::@constructor::new
              element: <testLibraryFragment>::@class::M1::@constructor::new#element
              typeName: M1
        class M2 @54
          reference: <testLibraryFragment>::@class::M2
          element: <testLibrary>::@class::M2
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M2::@constructor::new
              element: <testLibraryFragment>::@class::M2::@constructor::new#element
              typeName: M2
        class C2 @67
          reference: <testLibraryFragment>::@class::C2
          element: <testLibrary>::@class::C2
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C2::@constructor::new
              element: <testLibraryFragment>::@class::C2::@constructor::new#element
              typeName: C2
              formalParameters
                i (offset=-1)
                  element: <testLibraryFragment>::@class::C2::@constructor::new::@parameter::i#element
        class C1 @90
          reference: <testLibraryFragment>::@class::C1
          element: <testLibrary>::@class::C1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C1::@constructor::new
              element: <testLibraryFragment>::@class::C1::@constructor::new#element
              typeName: C1
              formalParameters
                i (offset=-1)
                  element: <testLibraryFragment>::@class::C1::@constructor::new::@parameter::i#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional i
              type: int
    mixin class M1
      reference: <testLibrary>::@class::M1
      firstFragment: <testLibraryFragment>::@class::M1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M1::@constructor::new
    mixin class M2
      reference: <testLibrary>::@class::M2
      firstFragment: <testLibraryFragment>::@class::M2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M2::@constructor::new
    class alias C2
      reference: <testLibrary>::@class::C2
      firstFragment: <testLibraryFragment>::@class::C2
      supertype: C1
      mixins
        M2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C2::@constructor::new
          formalParameters
            requiredPositional i
              type: int
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: i @-1
                    element: <testLibraryFragment>::@class::C2::@constructor::new::@parameter::i#element
                    staticType: int
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::C1::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::C1::@constructor::new#element
    class alias C1
      reference: <testLibrary>::@class::C1
      firstFragment: <testLibraryFragment>::@class::C1
      supertype: A
      mixins
        M1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C1::@constructor::new
          formalParameters
            requiredPositional i
              type: int
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: i @-1
                    element: <testLibraryFragment>::@class::C1::@constructor::new::@parameter::i#element
                    staticType: int
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_classAlias_constructors_optionalParameters() async {
    var library = await buildLibrary('''
class A {
  A.c1(int a);
  A.c2(int a, [int? b, int c = 0]);
  A.c3(int a, {int? b, int c = 0});
}

mixin M {}

class C = A with M;
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            c1 @14
              reference: <testLibraryFragment>::@class::A::@constructor::c1
              element: <testLibraryFragment>::@class::A::@constructor::c1#element
              typeName: A
              typeNameOffset: 12
              periodOffset: 13
              formalParameters
                a @21
                  element: <testLibraryFragment>::@class::A::@constructor::c1::@parameter::a#element
            c2 @29
              reference: <testLibraryFragment>::@class::A::@constructor::c2
              element: <testLibraryFragment>::@class::A::@constructor::c2#element
              typeName: A
              typeNameOffset: 27
              periodOffset: 28
              formalParameters
                a @36
                  element: <testLibraryFragment>::@class::A::@constructor::c2::@parameter::a#element
                default b @45
                  element: <testLibraryFragment>::@class::A::@constructor::c2::@parameter::b#element
                default c @52
                  element: <testLibraryFragment>::@class::A::@constructor::c2::@parameter::c#element
                  initializer: expression_0
                    IntegerLiteral
                      literal: 0 @56
                      staticType: int
            c3 @65
              reference: <testLibraryFragment>::@class::A::@constructor::c3
              element: <testLibraryFragment>::@class::A::@constructor::c3#element
              typeName: A
              typeNameOffset: 63
              periodOffset: 64
              formalParameters
                a @72
                  element: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::a#element
                default b @81
                  reference: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::b#element
                default c @88
                  reference: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::c
                  element: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::c#element
                  initializer: expression_1
                    IntegerLiteral
                      literal: 0 @92
                      staticType: int
        class C @118
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic c1
              reference: <testLibraryFragment>::@class::C::@constructor::c1
              element: <testLibraryFragment>::@class::C::@constructor::c1#element
              typeName: C
              formalParameters
                a (offset=-1)
                  element: <testLibraryFragment>::@class::C::@constructor::c1::@parameter::a#element
            synthetic c2
              reference: <testLibraryFragment>::@class::C::@constructor::c2
              element: <testLibraryFragment>::@class::C::@constructor::c2#element
              typeName: C
              formalParameters
                a (offset=-1)
                  element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::a#element
                default b (offset=-1)
                  element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::b#element
                default c (offset=-1)
                  element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::c#element
                  initializer: expression_0
            synthetic c3
              reference: <testLibraryFragment>::@class::C::@constructor::c3
              element: <testLibraryFragment>::@class::C::@constructor::c3#element
              typeName: C
              formalParameters
                a (offset=-1)
                  element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::a#element
                default b (offset=-1)
                  reference: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::b
                  element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::b#element
                default c (offset=-1)
                  reference: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::c
                  element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::c#element
                  initializer: expression_1
      mixins
        mixin M @106
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        c1
          firstFragment: <testLibraryFragment>::@class::A::@constructor::c1
          formalParameters
            requiredPositional a
              type: int
        c2
          firstFragment: <testLibraryFragment>::@class::A::@constructor::c2
          formalParameters
            requiredPositional a
              type: int
            optionalPositional b
              type: int?
            optionalPositional c
              type: int
              constantInitializer
                expression: expression_0
        c3
          firstFragment: <testLibraryFragment>::@class::A::@constructor::c3
          formalParameters
            requiredPositional a
              type: int
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::b
              type: int?
            optionalNamed c
              firstFragment: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::c
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::A::@constructor::c3::@parameter::c
                expression: expression_1
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      mixins
        M
      constructors
        synthetic c1
          firstFragment: <testLibraryFragment>::@class::C::@constructor::c1
          formalParameters
            requiredPositional a
              type: int
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: c1 @-1
                element: <testLibraryFragment>::@class::A::@constructor::c1#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: a @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c1::@parameter::a#element
                    staticType: int
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::c1#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::c1#element
        synthetic c2
          firstFragment: <testLibraryFragment>::@class::C::@constructor::c2
          formalParameters
            requiredPositional a
              type: int
            optionalPositional b
              type: int?
            optionalPositional c
              type: int
              constantInitializer
                expression: expression_0
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: c2 @-1
                element: <testLibraryFragment>::@class::A::@constructor::c2#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: a @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::a#element
                    staticType: int
                  SimpleIdentifier
                    token: b @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::b#element
                    staticType: int?
                  SimpleIdentifier
                    token: c @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c2::@parameter::c#element
                    staticType: int
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::c2#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::c2#element
        synthetic c3
          firstFragment: <testLibraryFragment>::@class::C::@constructor::c3
          formalParameters
            requiredPositional a
              type: int
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::b
              type: int?
            optionalNamed c
              firstFragment: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::c
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::c
                expression: expression_1
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: c3 @-1
                element: <testLibraryFragment>::@class::A::@constructor::c3#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: a @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::a#element
                    staticType: int
                  SimpleIdentifier
                    token: b @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::b#element
                    staticType: int?
                  SimpleIdentifier
                    token: c @-1
                    element: <testLibraryFragment>::@class::C::@constructor::c3::@parameter::c#element
                    staticType: int
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::c3#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::c3#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_constructors_requiredParameters() async {
    var library = await buildLibrary('''
class A<T extends num> {
  A(T x, T y);
}

mixin M {}

class B<E extends num> = A<E> with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: T@8
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 27
              formalParameters
                x @31
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::x#element
                y @36
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::y#element
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            E @63
              element: E@63
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              formalParameters
                x (offset=-1)
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::x#element
                y (offset=-1)
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::y#element
      mixins
        mixin M @49
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
          bound: num
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional x
              type: T
            requiredPositional y
              type: T
    class alias B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        E
          bound: num
      supertype: A<E>
      mixins
        M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional x
              type: E
            requiredPositional y
              type: E
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::x#element
                    staticType: E
                  SimpleIdentifier
                    token: y @-1
                    element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::y#element
                    staticType: E
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_documented() async {
    var library = await buildLibrary('''
/**
 * Docs
 */
class C = D with E;

class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @22
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @43
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @54
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// aaa
/// b
/// cc
class C = D with E;

class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @48
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @59
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// aaa\n/// b\n/// cc
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_documented_withLeadingNonDocumentation() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
class C = D with E;

class D {}
class E {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @66
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @87
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @98
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_final() async {
    var library = await buildLibrary('''
final class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin M @37
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    final class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_generic() async {
    var library = await buildLibrary('''
class Z = A with B<int>, C<double>;
class A {}
class B<B1> {}
class C<C1> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Z @6
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
        class A @42
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @53
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            B1 @55
              element: B1@55
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @68
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            C1 @70
              element: C1@70
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class alias Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      supertype: A
      mixins
        B<int>
        C<double>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::A::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        B1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        C1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_classAlias_interface() async {
    var library = await buildLibrary('''
interface class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @16
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin M @41
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    interface class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_invalid_extendsEnum() async {
    newFile('$testPackageLibPath/a.dart', r'''
enum E { v }
mixin M {}
''');

    var library = await buildLibrary('''
import 'a.dart';
class A = E with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class A @23
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class alias A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
''');
  }

  test_classAlias_invalid_extendsMixin() async {
    var library = await buildLibrary('''
mixin M1 {}
mixin M2 {}
class A = M1 with M2;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @30
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      mixins
        mixin M1 @6
          reference: <testLibraryFragment>::@mixin::M1
          element: <testLibrary>::@mixin::M1
        mixin M2 @18
          reference: <testLibraryFragment>::@mixin::M2
          element: <testLibrary>::@mixin::M2
  classes
    class alias A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      supertype: Object
      mixins
        M2
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M1
      reference: <testLibrary>::@mixin::M1
      firstFragment: <testLibraryFragment>::@mixin::M1
      superclassConstraints
        Object
    mixin M2
      reference: <testLibrary>::@mixin::M2
      firstFragment: <testLibraryFragment>::@mixin::M2
      superclassConstraints
        Object
''');
  }

  test_classAlias_mixin_class() async {
    var library = await buildLibrary('''
mixin class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin M @37
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    mixin class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_notSimplyBounded_self() async {
    var library = await buildLibrary('''
class C<T extends C> = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @39
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @50
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    notSimplyBounded class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: C<dynamic>
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_notSimplyBounded_simple_no_type_parameter_bound() async {
    // If no bounds are specified, then the class is simply bounded by syntax
    // alone, so there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C<T> = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @29
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @40
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_notSimplyBounded_simple_non_generic() async {
    // If no type parameters are specified, then the class is simply bounded, so
    // there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @26
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @37
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_sealed() async {
    var library = await buildLibrary('''
sealed class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @13
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      mixins
        mixin M @38
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    abstract sealed class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: dart:core::<fragment>::@class::Object::@constructor::new#element
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_with_const_constructors() async {
    newFile('$testPackageLibPath/a.dart', r'''
class Base {
  const Base._priv();
  const Base();
  const Base.named();
}
''');
    var library = await buildLibrary('''
import "a.dart";
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class M @23
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class MixinApp @34
          reference: <testLibraryFragment>::@class::MixinApp
          element: <testLibrary>::@class::MixinApp
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::new
              element: <testLibraryFragment>::@class::MixinApp::@constructor::new#element
              typeName: MixinApp
            synthetic const named
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::named
              element: <testLibraryFragment>::@class::MixinApp::@constructor::named#element
              typeName: MixinApp
  classes
    class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias MixinApp
      reference: <testLibrary>::@class::MixinApp
      firstFragment: <testLibraryFragment>::@class::MixinApp
      supertype: Base
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::new#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::new#element
        synthetic const named
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::named
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: named @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::named#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::named#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::named#element
''');
  }

  test_classAlias_with_forwarding_constructors() async {
    newFile('$testPackageLibPath/a.dart', r'''
class Base {
  bool x = true;
  Base._priv();
  Base();
  Base.noArgs();
  Base.requiredArg(x);
  Base.positionalArg([bool x = true]);
  Base.positionalArg2([this.x = true]);
  Base.namedArg({int x = 42});
  Base.namedArg2({this.x = true});
  factory Base.fact() => Base();
  factory Base.fact2() = Base.noArgs;
}
''');
    var library = await buildLibrary('''
import "a.dart";
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class M @23
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class MixinApp @34
          reference: <testLibraryFragment>::@class::MixinApp
          element: <testLibrary>::@class::MixinApp
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::new
              element: <testLibraryFragment>::@class::MixinApp::@constructor::new#element
              typeName: MixinApp
            synthetic noArgs
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::noArgs
              element: <testLibraryFragment>::@class::MixinApp::@constructor::noArgs#element
              typeName: MixinApp
            synthetic requiredArg
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::requiredArg
              element: <testLibraryFragment>::@class::MixinApp::@constructor::requiredArg#element
              typeName: MixinApp
              formalParameters
                x (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::requiredArg::@parameter::x#element
            synthetic positionalArg
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg
              element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg#element
              typeName: MixinApp
              formalParameters
                default x (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg::@parameter::x#element
                  initializer: expression_0
                    BooleanLiteral
                      literal: true @127
                      staticType: bool
            synthetic positionalArg2
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg2
              element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg2#element
              typeName: MixinApp
              formalParameters
                default x (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg2::@parameter::x#element
                  initializer: expression_1
                    BooleanLiteral
                      literal: true @167
                      staticType: bool
            synthetic namedArg
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg
              element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg#element
              typeName: MixinApp
              formalParameters
                default x (offset=-1)
                  reference: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg::@parameter::x
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg::@parameter::x#element
                  initializer: expression_2
                    IntegerLiteral
                      literal: 42 @200
                      staticType: int
            synthetic namedArg2
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2
              element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2#element
              typeName: MixinApp
              formalParameters
                default x (offset=-1)
                  reference: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2::@parameter::x
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2::@parameter::x#element
                  initializer: expression_3
                    BooleanLiteral
                      literal: true @233
                      staticType: bool
  classes
    class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias MixinApp
      reference: <testLibrary>::@class::MixinApp
      firstFragment: <testLibraryFragment>::@class::MixinApp
      supertype: Base
      mixins
        M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::new#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::new#element
        synthetic noArgs
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::noArgs
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: noArgs @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::noArgs#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::noArgs#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::noArgs#element
        synthetic requiredArg
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::requiredArg
          formalParameters
            requiredPositional x
              type: dynamic
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: requiredArg @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::requiredArg#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::requiredArg::@parameter::x#element
                    staticType: dynamic
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::requiredArg#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::requiredArg#element
        synthetic positionalArg
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg
          formalParameters
            optionalPositional x
              type: bool
              constantInitializer
                expression: expression_0
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: positionalArg @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg::@parameter::x#element
                    staticType: bool
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg#element
        synthetic positionalArg2
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg2
          formalParameters
            optionalPositional final x
              type: bool
              constantInitializer
                expression: expression_1
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: positionalArg2 @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg2#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::positionalArg2::@parameter::x#element
                    staticType: bool
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg2#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::positionalArg2#element
        synthetic namedArg
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg
          formalParameters
            optionalNamed x
              firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg::@parameter::x
              type: int
              constantInitializer
                fragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg::@parameter::x
                expression: expression_2
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: namedArg @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg::@parameter::x#element
                    staticType: int
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg#element
        synthetic namedArg2
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2
          formalParameters
            optionalNamed final x
              firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2::@parameter::x
              type: bool
              constantInitializer
                fragment: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2::@parameter::x
                expression: expression_3
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: namedArg2 @-1
                element: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg2#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: x @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::namedArg2::@parameter::x#element
                    staticType: bool
                rightParenthesis: ) @0
              element: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg2#element
          superConstructor: package:test/a.dart::<fragment>::@class::Base::@constructor::namedArg2#element
''');
  }

  test_classAlias_with_forwarding_constructors_type_substitution() async {
    var library = await buildLibrary('''
class Base<T> {
  Base.ctor(T t, List<T> l);
}
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Base @6
          reference: <testLibraryFragment>::@class::Base
          element: <testLibrary>::@class::Base
          typeParameters
            T @11
              element: T@11
          constructors
            ctor @23
              reference: <testLibraryFragment>::@class::Base::@constructor::ctor
              element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
              typeName: Base
              typeNameOffset: 18
              periodOffset: 22
              formalParameters
                t @30
                  element: <testLibraryFragment>::@class::Base::@constructor::ctor::@parameter::t#element
                l @41
                  element: <testLibraryFragment>::@class::Base::@constructor::ctor::@parameter::l#element
        class M @53
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class MixinApp @64
          reference: <testLibraryFragment>::@class::MixinApp
          element: <testLibrary>::@class::MixinApp
          constructors
            synthetic ctor
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::ctor
              element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor#element
              typeName: MixinApp
              formalParameters
                t (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::t#element
                l (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::l#element
  classes
    class Base
      reference: <testLibrary>::@class::Base
      firstFragment: <testLibraryFragment>::@class::Base
      typeParameters
        T
      constructors
        ctor
          firstFragment: <testLibraryFragment>::@class::Base::@constructor::ctor
          formalParameters
            requiredPositional t
              type: T
            requiredPositional l
              type: List<T>
    class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias MixinApp
      reference: <testLibrary>::@class::MixinApp
      firstFragment: <testLibraryFragment>::@class::MixinApp
      supertype: Base<dynamic>
      mixins
        M
      constructors
        synthetic ctor
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::ctor
          formalParameters
            requiredPositional t
              type: dynamic
            requiredPositional l
              type: List<dynamic>
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: ctor @-1
                element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: t @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::t#element
                    staticType: dynamic
                  SimpleIdentifier
                    token: l @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::l#element
                    staticType: List<dynamic>
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
          superConstructor: <testLibraryFragment>::@class::Base::@constructor::ctor#element
''');
  }

  test_classAlias_with_forwarding_constructors_type_substitution_complex() async {
    var library = await buildLibrary('''
class Base<T> {
  Base.ctor(T t, List<T> l);
}
class M {}
class MixinApp<U> = Base<List<U>> with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Base @6
          reference: <testLibraryFragment>::@class::Base
          element: <testLibrary>::@class::Base
          typeParameters
            T @11
              element: T@11
          constructors
            ctor @23
              reference: <testLibraryFragment>::@class::Base::@constructor::ctor
              element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
              typeName: Base
              typeNameOffset: 18
              periodOffset: 22
              formalParameters
                t @30
                  element: <testLibraryFragment>::@class::Base::@constructor::ctor::@parameter::t#element
                l @41
                  element: <testLibraryFragment>::@class::Base::@constructor::ctor::@parameter::l#element
        class M @53
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class MixinApp @64
          reference: <testLibraryFragment>::@class::MixinApp
          element: <testLibrary>::@class::MixinApp
          typeParameters
            U @73
              element: U@73
          constructors
            synthetic ctor
              reference: <testLibraryFragment>::@class::MixinApp::@constructor::ctor
              element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor#element
              typeName: MixinApp
              formalParameters
                t (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::t#element
                l (offset=-1)
                  element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::l#element
  classes
    class Base
      reference: <testLibrary>::@class::Base
      firstFragment: <testLibraryFragment>::@class::Base
      typeParameters
        T
      constructors
        ctor
          firstFragment: <testLibraryFragment>::@class::Base::@constructor::ctor
          formalParameters
            requiredPositional t
              type: T
            requiredPositional l
              type: List<T>
    class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias MixinApp
      reference: <testLibrary>::@class::MixinApp
      firstFragment: <testLibraryFragment>::@class::MixinApp
      typeParameters
        U
      supertype: Base<List<U>>
      mixins
        M
      constructors
        synthetic ctor
          firstFragment: <testLibraryFragment>::@class::MixinApp::@constructor::ctor
          formalParameters
            requiredPositional t
              type: List<U>
            requiredPositional l
              type: List<List<U>>
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              period: . @0
              constructorName: SimpleIdentifier
                token: ctor @-1
                element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
                staticType: null
              argumentList: ArgumentList
                leftParenthesis: ( @0
                arguments
                  SimpleIdentifier
                    token: t @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::t#element
                    staticType: List<U>
                  SimpleIdentifier
                    token: l @-1
                    element: <testLibraryFragment>::@class::MixinApp::@constructor::ctor::@parameter::l#element
                    staticType: List<List<U>>
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::Base::@constructor::ctor#element
          superConstructor: <testLibraryFragment>::@class::Base::@constructor::ctor#element
''');
  }

  test_classAlias_with_mixin_members() async {
    var library = await buildLibrary('''
class C = D with E;
class D {}
class E {
  int get a => null;
  void set b(int i) {}
  void f() {}
  int x;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @26
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @37
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          fields
            x @105
              reference: <testLibraryFragment>::@class::E::@field::x
              element: <testLibraryFragment>::@class::E::@field::x#element
              getter2: <testLibraryFragment>::@class::E::@getter::x
              setter2: <testLibraryFragment>::@class::E::@setter::x
            synthetic a
              reference: <testLibraryFragment>::@class::E::@field::a
              element: <testLibraryFragment>::@class::E::@field::a#element
              getter2: <testLibraryFragment>::@class::E::@getter::a
            synthetic b
              reference: <testLibraryFragment>::@class::E::@field::b
              element: <testLibraryFragment>::@class::E::@field::b#element
              setter2: <testLibraryFragment>::@class::E::@setter::b
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::E::@getter::x
              element: <testLibraryFragment>::@class::E::@getter::x#element
            get a @51
              reference: <testLibraryFragment>::@class::E::@getter::a
              element: <testLibraryFragment>::@class::E::@getter::a#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::E::@setter::x
              element: <testLibraryFragment>::@class::E::@setter::x#element
              formalParameters
                _x
                  element: <testLibraryFragment>::@class::E::@setter::x::@parameter::_x#element
            set b @73
              reference: <testLibraryFragment>::@class::E::@setter::b
              element: <testLibraryFragment>::@class::E::@setter::b#element
              formalParameters
                i @79
                  element: <testLibraryFragment>::@class::E::@setter::b::@parameter::i#element
          methods
            f @92
              reference: <testLibraryFragment>::@class::E::@method::f
              element: <testLibraryFragment>::@class::E::@method::f#element
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          constantInitializers
            SuperConstructorInvocation
              superKeyword: super @0
              argumentList: ArgumentList
                leftParenthesis: ( @0
                rightParenthesis: ) @0
              element: <testLibraryFragment>::@class::D::@constructor::new#element
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      fields
        x
          firstFragment: <testLibraryFragment>::@class::E::@field::x
          type: int
          getter: <testLibraryFragment>::@class::E::@getter::x#element
          setter: <testLibraryFragment>::@class::E::@setter::x#element
        synthetic a
          firstFragment: <testLibraryFragment>::@class::E::@field::a
          type: int
          getter: <testLibraryFragment>::@class::E::@getter::a#element
        synthetic b
          firstFragment: <testLibraryFragment>::@class::E::@field::b
          type: int
          setter: <testLibraryFragment>::@class::E::@setter::b#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::E::@getter::x
          returnType: int
        get a
          firstFragment: <testLibraryFragment>::@class::E::@getter::a
          returnType: int
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::E::@setter::x
          formalParameters
            requiredPositional _x
              type: int
          returnType: void
        set b
          firstFragment: <testLibraryFragment>::@class::E::@setter::b
          formalParameters
            requiredPositional i
              type: int
          returnType: void
      methods
        f
          firstFragment: <testLibraryFragment>::@class::E::@method::f
          returnType: void
''');
  }

  test_classes() async {
    var library = await buildLibrary('class C {} class D {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @17
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_implicitConstructor_named_const() async {
    var library = await buildLibrary('''
class C {
  final Object x;
  const C.named(this.x);
}
const x = C.named(42);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const named @38
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 36
              periodOffset: 37
              formalParameters
                this.x @49
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
      topLevelVariables
        hasInitializer x @61
          reference: <testLibraryFragment>::@topLevelVariable::x
          element: <testLibrary>::@topLevelVariable::x
          initializer: expression_0
            InstanceCreationExpression
              constructorName: ConstructorName
                type: NamedType
                  name: C @65
                  element2: <testLibrary>::@class::C
                  type: C
                period: . @66
                name: SimpleIdentifier
                  token: named @67
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
                  staticType: null
                element: <testLibraryFragment>::@class::C::@constructor::named#element
              argumentList: ArgumentList
                leftParenthesis: ( @72
                arguments
                  IntegerLiteral
                    literal: 42 @73
                    staticType: int
                rightParenthesis: ) @75
              staticType: C
          getter2: <testLibraryFragment>::@getter::x
      getters
        synthetic get x
          reference: <testLibraryFragment>::@getter::x
          element: <testLibraryFragment>::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          formalParameters
            requiredPositional final hasImplicitType x
              type: Object
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
          returnType: Object
  topLevelVariables
    const hasInitializer x
      reference: <testLibrary>::@topLevelVariable::x
      firstFragment: <testLibraryFragment>::@topLevelVariable::x
      type: C
      constantInitializer
        fragment: <testLibraryFragment>::@topLevelVariable::x
        expression: expression_0
      getter: <testLibraryFragment>::@getter::x#element
  getters
    synthetic static get x
      firstFragment: <testLibraryFragment>::@getter::x
      returnType: C
''');
  }

  test_invalid_setterParameter_fieldFormalParameter() async {
    var library = await buildLibrary('''
class C {
  int foo;
  void set bar(this.foo) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
            synthetic bar
              reference: <testLibraryFragment>::@class::C::@field::bar
              element: <testLibraryFragment>::@class::C::@field::bar#element
              setter2: <testLibraryFragment>::@class::C::@setter::bar
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
            set bar @32
              reference: <testLibraryFragment>::@class::C::@setter::bar
              element: <testLibraryFragment>::@class::C::@setter::bar#element
              formalParameters
                this.foo @41
                  element: <testLibraryFragment>::@class::C::@setter::bar::@parameter::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
        synthetic bar
          firstFragment: <testLibraryFragment>::@class::C::@field::bar
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::bar#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
          returnType: int
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
          returnType: void
        set bar
          firstFragment: <testLibraryFragment>::@class::C::@setter::bar
          formalParameters
            requiredPositional final hasImplicitType foo
              type: dynamic
          returnType: void
''');
  }

  test_invalid_setterParameter_fieldFormalParameter_self() async {
    var library = await buildLibrary('''
class C {
  set x(this.x) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @16
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                this.x @23
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
          returnType: void
''');
  }

  test_unused_type_parameter() async {
    var library = await buildLibrary('''
class C<T> {
  void f() {}
}
C<int> c;
var v = c.f;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: T@8
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @20
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
      topLevelVariables
        c @36
          reference: <testLibraryFragment>::@topLevelVariable::c
          element: <testLibrary>::@topLevelVariable::c
          getter2: <testLibraryFragment>::@getter::c
          setter2: <testLibraryFragment>::@setter::c
        hasInitializer v @43
          reference: <testLibraryFragment>::@topLevelVariable::v
          element: <testLibrary>::@topLevelVariable::v
          getter2: <testLibraryFragment>::@getter::v
          setter2: <testLibraryFragment>::@setter::v
      getters
        synthetic get c
          reference: <testLibraryFragment>::@getter::c
          element: <testLibraryFragment>::@getter::c#element
        synthetic get v
          reference: <testLibraryFragment>::@getter::v
          element: <testLibraryFragment>::@getter::v#element
      setters
        synthetic set c
          reference: <testLibraryFragment>::@setter::c
          element: <testLibraryFragment>::@setter::c#element
          formalParameters
            _c
              element: <testLibraryFragment>::@setter::c::@parameter::_c#element
        synthetic set v
          reference: <testLibraryFragment>::@setter::v
          element: <testLibraryFragment>::@setter::v#element
          formalParameters
            _v
              element: <testLibraryFragment>::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          returnType: void
  topLevelVariables
    c
      reference: <testLibrary>::@topLevelVariable::c
      firstFragment: <testLibraryFragment>::@topLevelVariable::c
      type: C<int>
      getter: <testLibraryFragment>::@getter::c#element
      setter: <testLibraryFragment>::@setter::c#element
    hasInitializer v
      reference: <testLibrary>::@topLevelVariable::v
      firstFragment: <testLibraryFragment>::@topLevelVariable::v
      type: void Function()
      getter: <testLibraryFragment>::@getter::v#element
      setter: <testLibraryFragment>::@setter::v#element
  getters
    synthetic static get c
      firstFragment: <testLibraryFragment>::@getter::c
      returnType: C<int>
    synthetic static get v
      firstFragment: <testLibraryFragment>::@getter::v
      returnType: void Function()
  setters
    synthetic static set c
      firstFragment: <testLibraryFragment>::@setter::c
      formalParameters
        requiredPositional _c
          type: C<int>
      returnType: void
    synthetic static set v
      firstFragment: <testLibraryFragment>::@setter::v
      formalParameters
        requiredPositional _v
          type: void Function()
      returnType: void
''');
  }
}

abstract class ClassElementTest_augmentation extends ElementsBaseTest {
  test_add_augment() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';

class A {
  void foo() {}
}

augment class A {
  void bar() {}
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        class A @28
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            foo @39
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::A
              returnType: void
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
            methods
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::bar
              <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo
        augment class A @65
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            bar @76
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::bar
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @28
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @39
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo#element
        class A @65
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            bar @76
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::bar
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::bar#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo
        bar
          reference: <testLibrary>::@class::A::@method::bar
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::bar
''');
  }

  test_augmentation_constField_hasConstConstructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  static const int foo = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  const A();
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const @33
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            static const foo @58
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @64
                  staticType: int
          accessors
            synthetic static get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 33
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            hasInitializer foo @58
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @64
                  staticType: int
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        static const hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          constantInitializer
            fragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            expression: expression_0
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic static get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmentation_constField_noConstConstructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  static const int foo = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            static const foo @58
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @64
                  staticType: int
          accessors
            synthetic static get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            hasInitializer foo @58
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @64
                  staticType: int
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        static const hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          constantInitializer
            fragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            expression: expression_0
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic static get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmentation_finalField_hasConstConstructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  final int foo = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  const A();
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const @33
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            final foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @57
                  staticType: int
          accessors
            synthetic get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 33
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            hasInitializer foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @57
                  staticType: int
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          constantInitializer
            fragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            expression: expression_0
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmentation_finalField_noConstConstructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  final int foo = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            final foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
          accessors
            synthetic get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            hasInitializer foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmentationTarget() async {
    newFile('$testPackageLibPath/a1.dart', r'''
part of 'test.dart';
part 'a11.dart';
part 'a12.dart';
augment class A {}
''');

    newFile('$testPackageLibPath/a11.dart', r'''
part of 'a1.dart';
augment class A {}
''');

    newFile('$testPackageLibPath/a12.dart', r'''
part of 'a1.dart';
augment class A {}
''');

    newFile('$testPackageLibPath/a2.dart', r'''
part of 'test.dart';
part 'a21.dart';
part 'a22.dart';
augment class A {}
''');

    newFile('$testPackageLibPath/a21.dart', r'''
part of 'a2.dart';
augment class A {}
''');

    newFile('$testPackageLibPath/a22.dart', r'''
part of 'a2.dart';
augment class A {}
''');

    configuration.withExportScope = true;
    var library = await buildLibrary(r'''
part 'a1.dart';
part 'a2.dart';
class A {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a1.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a1.dart
        part_1
          uri: package:test/a2.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a2.dart
      classes
        class A @38
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
    <testLibrary>::@fragment::package:test/a1.dart
      enclosingElement3: <testLibraryFragment>
      parts
        part_2
          uri: package:test/a11.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a1.dart
          unit: <testLibrary>::@fragment::package:test/a11.dart
        part_3
          uri: package:test/a12.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a1.dart
          unit: <testLibrary>::@fragment::package:test/a12.dart
      classes
        augment class A @69
          reference: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a1.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a11.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a1.dart
      classes
        augment class A @33
          reference: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a11.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          augmentation: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a12.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a1.dart
      classes
        augment class A @33
          reference: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a12.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
          augmentation: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a2.dart
      enclosingElement3: <testLibraryFragment>
      parts
        part_4
          uri: package:test/a21.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a2.dart
          unit: <testLibrary>::@fragment::package:test/a21.dart
        part_5
          uri: package:test/a22.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a2.dart
          unit: <testLibrary>::@fragment::package:test/a22.dart
      classes
        augment class A @69
          reference: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a2.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
          augmentation: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a21.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a2.dart
      classes
        augment class A @33
          reference: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a21.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
          augmentation: <testLibrary>::@fragment::package:test/a22.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a22.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a2.dart
      classes
        augment class A @33
          reference: <testLibrary>::@fragment::package:test/a22.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a22.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
  exportedReferences
    declared <testLibraryFragment>::@class::A
  exportNamespace
    A: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a1.dart
      classes
        class A @38
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
    <testLibrary>::@fragment::package:test/a1.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/a11.dart
      classes
        class A @69
          reference: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a11.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a1.dart
      previousFragment: <testLibrary>::@fragment::package:test/a1.dart
      nextFragment: <testLibrary>::@fragment::package:test/a12.dart
      classes
        class A @33
          reference: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a1.dart::@classAugmentation::A
          nextFragment: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a12.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a1.dart
      previousFragment: <testLibrary>::@fragment::package:test/a11.dart
      nextFragment: <testLibrary>::@fragment::package:test/a2.dart
      classes
        class A @33
          reference: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a11.dart::@classAugmentation::A
          nextFragment: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a2.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a12.dart
      nextFragment: <testLibrary>::@fragment::package:test/a21.dart
      classes
        class A @69
          reference: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a12.dart::@classAugmentation::A
          nextFragment: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a21.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a2.dart
      previousFragment: <testLibrary>::@fragment::package:test/a2.dart
      nextFragment: <testLibrary>::@fragment::package:test/a22.dart
      classes
        class A @33
          reference: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a2.dart::@classAugmentation::A
          nextFragment: <testLibrary>::@fragment::package:test/a22.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a22.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a2.dart
      previousFragment: <testLibrary>::@fragment::package:test/a21.dart
      classes
        class A @33
          reference: <testLibrary>::@fragment::package:test/a22.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a21.dart::@classAugmentation::A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  exportedReferences
    declared <testLibraryFragment>::@class::A
  exportNamespace
    A: <testLibraryFragment>::@class::A
''');
  }

  test_augmentationTarget_augmentationThenDeclaration() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';

augment class A {
  void foo1() {}
}

class A {
  void foo2() {}
}

augment class A {
  void foo3() {}
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @36
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0
          methods
            foo1 @47
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@method::foo1
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0
              returnType: void
        class A @66
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            foo2 @77
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::A
              returnType: void
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
            methods
              <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo2
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1::@method::foo3
        augment class A @104
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            foo3 @115
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1::@method::foo3
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @36
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0
          element: <testLibrary>::@class::A::@def::0
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@constructor::new#element
              typeName: A
          methods
            foo1 @47
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@method::foo1
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@method::foo1#element
        class A @66
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::A
          element: <testLibrary>::@class::A::@def::1
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new#element
              typeName: A
          methods
            foo2 @77
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo2#element
        class A @104
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1
          element: <testLibrary>::@class::A::@def::1
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A
          methods
            foo3 @115
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1::@method::foo3
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1::@method::foo3#element
  classes
    class A
      reference: <testLibrary>::@class::A::@def::0
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@constructor::new
      methods
        foo1
          reference: <testLibrary>::@class::A::@def::0::@method::foo1
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::0::@method::foo1
    class A
      reference: <testLibrary>::@class::A::@def::1
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A::@constructor::new
      methods
        foo2
          reference: <testLibrary>::@class::A::@def::1::@method::foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::A::@method::foo2
        foo3
          reference: <testLibrary>::@class::A::@def::1::@method::foo3
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@def::1::@method::foo3
''');
  }

  test_augmentationTarget_no2() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
part 'b.dart';
augment class A {
  void foo1() {}
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
augment class A {
  void foo2() {}
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class B {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class B @21
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      parts
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        augment class A @50
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          methods
            foo1 @61
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::foo1
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: void
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
            methods
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::foo1
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@method::foo2
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
      classes
        augment class A @32
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          methods
            foo2 @43
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@method::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class B @21
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @50
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new#element
              typeName: A
          methods
            foo1 @61
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::foo1
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::foo1#element
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a.dart
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @32
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          methods
            foo2 @43
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@method::foo2
              element: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@method::foo2#element
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
      methods
        foo1
          reference: <testLibrary>::@class::A::@method::foo1
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@method::foo1
        foo2
          reference: <testLibrary>::@class::A::@method::foo2
          firstFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@method::foo2
''');
  }

  test_augmented_constructor_augment_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment A.foo();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 50
              nameEnd: 54
              augmentationTargetAny: <testLibraryFragment>::@class::A::@getter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo#element
              typeName: A
              typeNameOffset: 49
              periodOffset: 50
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_constructor_augment_getter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment A.foo();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int get foo => 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          accessors
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 50
              nameEnd: 54
              augmentationTargetAny: <testLibraryFragment>::@class::A::@getter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          getters
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo#element
              typeName: A
              typeNameOffset: 49
              periodOffset: 50
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_augmented_constructor_augment_method() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment A.foo();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  void foo() {}
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: void
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
            methods
              <testLibraryFragment>::@class::A::@method::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 50
              nameEnd: 54
              augmentationTargetAny: <testLibraryFragment>::@class::A::@method::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo#element
              typeName: A
              typeNameOffset: 49
              periodOffset: 50
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
''');
  }

  test_augmented_constructor_augment_setter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment A.foo();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  set foo(int _) {}
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          accessors
            set foo= @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @39
                  type: int
              returnType: void
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
            accessors
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 50
              nameEnd: 54
              augmentationTargetAny: <testLibraryFragment>::@class::A::@setter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          setters
            set foo @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @39
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            augment foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo#element
              typeName: A
              typeNameOffset: 49
              periodOffset: 50
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructorAugmentation::foo
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_augmented_constructors_add_named() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  A.named();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 42
              nameEnd: 48
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named#element
              typeName: A
              typeNameOffset: 41
              periodOffset: 42
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        named
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
''');
  }

  test_augmented_constructors_add_named_generic() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A<T2> {
  A.named(T2 a);
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A<T1> {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T1 @23
              defaultType: dynamic
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          augmented
            constructors
              ConstructorMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
                augmentationSubstitution: {T2: T1}
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          typeParameters
            covariant T2 @37
              defaultType: dynamic
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            named @47
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 46
              nameEnd: 52
              parameters
                requiredPositional a @56
                  type: T2
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          typeParameters
            T1 @23
              element: <not-implemented>
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          typeParameters
            T2 @37
              element: <not-implemented>
          constructors
            named @47
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named#element
              typeName: A
              typeNameOffset: 45
              periodOffset: 46
              formalParameters
                a @56
                  element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named::@parameter::a#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T1
      constructors
        named
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
          formalParameters
            requiredPositional a
              type: T2
''');
  }

  test_augmented_constructors_add_named_hasUnnamed() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  A.named();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  A();
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            @27
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 42
              nameEnd: 48
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 27
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named#element
              typeName: A
              typeNameOffset: 41
              periodOffset: 42
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
        named
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
''');
  }

  test_augmented_constructors_add_unnamed() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  A();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            @41
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            new
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new#element
              typeName: A
              typeNameOffset: 41
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
''');
  }

  test_augmented_constructors_add_unnamed_hasNamed() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  A();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  A.named();
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            named @29
              reference: <testLibraryFragment>::@class::A::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 28
              nameEnd: 34
          augmented
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              <testLibraryFragment>::@class::A::@constructor::named
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            @41
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            named @29
              reference: <testLibraryFragment>::@class::A::@constructor::named
              element: <testLibraryFragment>::@class::A::@constructor::named#element
              typeName: A
              typeNameOffset: 27
              periodOffset: 28
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            new
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new#element
              typeName: A
              typeNameOffset: 41
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::A::@constructor::named
        new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::new
''');
  }

  test_augmented_constructors_add_useFieldFormal() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  A.named(this.f);
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int f;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            final f @37
              reference: <testLibraryFragment>::@class::A::@field::f
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::A::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::f
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
            accessors
              <testLibraryFragment>::@class::A::@getter::f
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 42
              nameEnd: 48
              parameters
                requiredPositional final hasImplicitType this.f @54
                  type: int
                  field: <testLibraryFragment>::@class::A::@field::f
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            f @37
              reference: <testLibraryFragment>::@class::A::@field::f
              element: <testLibraryFragment>::@class::A::@field::f#element
              getter2: <testLibraryFragment>::@class::A::@getter::f
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::A::@getter::f
              element: <testLibraryFragment>::@class::A::@getter::f#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            named @43
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named#element
              typeName: A
              typeNameOffset: 41
              periodOffset: 42
              formalParameters
                this.f @54
                  element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named::@parameter::f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final f
          firstFragment: <testLibraryFragment>::@class::A::@field::f
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::f#element
      constructors
        named
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
          formalParameters
            requiredPositional final hasImplicitType f
              type: int
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::A::@getter::f
''');
  }

  test_augmented_constructors_add_useFieldInitializer() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  const A.named() : f = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int f;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            final f @37
              reference: <testLibraryFragment>::@class::A::@field::f
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::A::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::f
            constructors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
            accessors
              <testLibraryFragment>::@class::A::@getter::f
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          constructors
            const named @49
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              periodOffset: 48
              nameEnd: 54
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: f @59
                    staticElement: <testLibraryFragment>::@class::A::@field::f
                    element: <testLibraryFragment>::@class::A::@field::f#element
                    staticType: null
                  equals: = @61
                  expression: IntegerLiteral
                    literal: 0 @63
                    staticType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            f @37
              reference: <testLibraryFragment>::@class::A::@field::f
              element: <testLibraryFragment>::@class::A::@field::f#element
              getter2: <testLibraryFragment>::@class::A::@getter::f
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::A::@getter::f
              element: <testLibraryFragment>::@class::A::@getter::f#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          constructors
            const named @49
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named#element
              typeName: A
              typeNameOffset: 47
              periodOffset: 48
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: f @59
                    staticElement: <testLibraryFragment>::@class::A::@field::f
                    element: <testLibraryFragment>::@class::A::@field::f#element
                    staticType: null
                  equals: = @61
                  expression: IntegerLiteral
                    literal: 0 @63
                    staticType: int
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final f
          firstFragment: <testLibraryFragment>::@class::A::@field::f
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::f#element
      constructors
        const named
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@constructor::named
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::A::@getter::f
''');
  }

  test_augmented_field_augment_constructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  A.foo();
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            foo @29
              reference: <testLibraryFragment>::@class::A::@constructor::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 28
              nameEnd: 32
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              augmentationTargetAny: <testLibraryFragment>::@class::A::@constructor::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            foo @29
              reference: <testLibraryFragment>::@class::A::@constructor::foo
              element: <testLibraryFragment>::@class::A::@constructor::foo#element
              typeName: A
              typeNameOffset: 27
              periodOffset: 28
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          type: int
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::A::@constructor::foo
''');
  }

  test_augmented_field_augment_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_field_augment_field2() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
part 'b.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
              augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_2
              augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
              nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_field_augment_field_afterGetter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 1;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
part 'b.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_field_augment_field_afterSetter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment set foo(int _) {}
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
part 'b.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo
          augmented
            fields
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          accessors
            augment set foo= @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              parameters
                requiredPositional _ @61
                  type: int
              returnType: void
              id: setter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          setters
            augment set foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @61
                  element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setterAugmentation::foo::@parameter::_#element
              previousFragment: <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_field_augment_field_augmentedInvocation() async {
    // This is invalid code, but it should not crash.
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {;
  augment static const int foo = augmented();
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  static const int foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            static const foo @44
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @50
                  staticType: int
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic static get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment static const foo @67
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                AugmentedInvocation
                  augmentedKeyword: augmented @73
                  arguments: ArgumentList
                    leftParenthesis: ( @82
                    rightParenthesis: ) @83
                  element: <null>
                  fragment: <null>
                  staticType: InvalidType
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @44
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @50
                  staticType: int
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @67
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              initializer: expression_1
                AugmentedInvocation
                  augmentedKeyword: augmented @73
                  arguments: ArgumentList
                    leftParenthesis: ( @82
                    rightParenthesis: ) @83
                  element: <null>
                  fragment: <null>
                  staticType: InvalidType
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        static const hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          constantInitializer
            fragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            expression: expression_1
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic static get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_augmented_field_augment_field_differentTypes() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment double foo = 1.2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @56
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: double
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @56
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_field_augment_field_plus() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment final int foo = augmented + 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int foo = 0;
  const A();
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            final foo @37
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @43
                  staticType: int
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            const @54
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment final foo @59
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                BinaryExpression
                  leftOperand: AugmentedExpression
                    augmentedKeyword: augmented @65
                    element: <testLibraryFragment>::@class::A::@field::foo
                    fragment: <testLibraryFragment>::@class::A::@field::foo
                    staticType: int
                  operator: + @75
                  rightOperand: IntegerLiteral
                    literal: 1 @77
                    staticType: int
                  staticElement: dart:core::<fragment>::@class::num::@method::+
                  element: dart:core::<fragment>::@class::num::@method::+#element
                  staticInvokeType: num Function(num)
                  staticType: int
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @37
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              initializer: expression_0
                IntegerLiteral
                  literal: 0 @43
                  staticType: int
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 54
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @59
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              initializer: expression_1
                BinaryExpression
                  leftOperand: AugmentedExpression
                    augmentedKeyword: augmented @65
                    element: <testLibraryFragment>::@class::A::@field::foo
                    fragment: <testLibraryFragment>::@class::A::@field::foo
                    staticType: int
                  operator: + @75
                  rightOperand: IntegerLiteral
                    literal: 1 @77
                    staticType: int
                  staticElement: dart:core::<fragment>::@class::num::@method::+
                  element: dart:core::<fragment>::@class::num::@method::+#element
                  staticInvokeType: num Function(num)
                  staticType: int
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          constantInitializer
            fragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            expression: expression_1
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  /// This is not allowed by the specification, but allowed syntactically,
  /// so we need a way to handle it.
  test_augmented_field_augment_getter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int get foo => 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_augmented_field_augment_method() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  void foo() {}
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: void
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            methods
              <testLibraryFragment>::@class::A::@method::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              augmentationTargetAny: <testLibraryFragment>::@class::A::@method::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          type: int
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
''');
  }

  /// This is not allowed by the specification, but allowed syntactically,
  /// so we need a way to handle it.
  test_augmented_field_augment_setter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int foo = 1;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  set foo(int _) {}
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              setter: setter_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            set foo= @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @39
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            augment foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              augmentationTarget: <testLibraryFragment>::@class::A::@field::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @39
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            augment hasInitializer foo @53
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@fieldAugmentation::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@field::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_augmented_fields_add() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  int foo2 = 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int foo1 = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo1 @31
              reference: <testLibraryFragment>::@class::A::@field::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo1 @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo1= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo1 @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo1
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo1
              <testLibraryFragment>::@class::A::@setter::foo1
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            foo2 @45
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_1
              getter: getter_1
              setter: setter_1
          accessors
            synthetic get foo2 @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: field_1
            synthetic set foo2= @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              parameters
                requiredPositional _foo2 @-1
                  type: int
              returnType: void
              id: setter_1
              variable: field_1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo1 @31
              reference: <testLibraryFragment>::@class::A::@field::foo1
              element: <testLibraryFragment>::@class::A::@field::foo1#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo1
              setter2: <testLibraryFragment>::@class::A::@setter::foo1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo1
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
          setters
            synthetic set foo1
              reference: <testLibraryFragment>::@class::A::@setter::foo1
              element: <testLibraryFragment>::@class::A::@setter::foo1#element
              formalParameters
                _foo1
                  element: <testLibraryFragment>::@class::A::@setter::foo1::@parameter::_foo1#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            hasInitializer foo2 @45
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              setter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
          getters
            synthetic get foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
          setters
            synthetic set foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2#element
              formalParameters
                _foo2
                  element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2::@parameter::_foo2#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo1
          firstFragment: <testLibraryFragment>::@class::A::@field::foo1
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo1#element
          setter: <testLibraryFragment>::@class::A::@setter::foo1#element
        hasInitializer foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
          type: int
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
          setter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo1
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo1
        synthetic get foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
      setters
        synthetic set foo1
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo1
          formalParameters
            requiredPositional _foo1
              type: int
        synthetic set foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
          formalParameters
            requiredPositional _foo2
              type: int
''');
  }

  test_augmented_fields_add_generic() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A<T1> {
  T1 foo2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A<T1> {
  T1 foo1;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T1 @23
              defaultType: dynamic
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo1 @34
              reference: <testLibraryFragment>::@class::A::@field::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              type: T1
              id: field_0
              getter: getter_0
              setter: setter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo1 @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: T1
              id: getter_0
              variable: field_0
            synthetic set foo1= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo1 @-1
                  type: T1
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo1
              FieldMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
                augmentationSubstitution: {T1: T1}
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo1
              <testLibraryFragment>::@class::A::@setter::foo1
              GetterMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
                augmentationSubstitution: {T1: T1}
              SetterMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
                augmentationSubstitution: {T1: T1}
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          typeParameters
            covariant T1 @37
              defaultType: dynamic
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            foo2 @48
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: T1
              id: field_1
              getter: getter_1
              setter: setter_1
          accessors
            synthetic get foo2 @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: T1
              id: getter_1
              variable: field_1
            synthetic set foo2= @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              parameters
                requiredPositional _foo2 @-1
                  type: T1
              returnType: void
              id: setter_1
              variable: field_1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          typeParameters
            T1 @23
              element: <not-implemented>
          fields
            foo1 @34
              reference: <testLibraryFragment>::@class::A::@field::foo1
              element: <testLibraryFragment>::@class::A::@field::foo1#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo1
              setter2: <testLibraryFragment>::@class::A::@setter::foo1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo1
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
          setters
            synthetic set foo1
              reference: <testLibraryFragment>::@class::A::@setter::foo1
              element: <testLibraryFragment>::@class::A::@setter::foo1#element
              formalParameters
                _foo1
                  element: <testLibraryFragment>::@class::A::@setter::foo1::@parameter::_foo1#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          typeParameters
            T1 @37
              element: <not-implemented>
          fields
            foo2 @48
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              setter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
          getters
            synthetic get foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
          setters
            synthetic set foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2#element
              formalParameters
                _foo2
                  element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2::@parameter::_foo2#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T1
      fields
        foo1
          firstFragment: <testLibraryFragment>::@class::A::@field::foo1
          type: T1
          getter: <testLibraryFragment>::@class::A::@getter::foo1#element
          setter: <testLibraryFragment>::@class::A::@setter::foo1#element
        foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
          type: T1
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
          setter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo1
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo1
        synthetic get foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
      setters
        synthetic set foo1
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo1
          formalParameters
            requiredPositional _foo1
              type: T1
        synthetic set foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@setter::foo2
          formalParameters
            requiredPositional _foo2
              type: T1
''');
  }

  test_augmented_fields_add_useFieldFormal() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  final int foo;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  A(this.foo);
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            @27
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional final hasImplicitType this.foo @34
                  type: int
                  field: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            final foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
          accessors
            synthetic get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 27
              formalParameters
                this.foo @34
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType foo
              type: int
      getters
        synthetic get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmented_fields_add_useFieldInitializer() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  final int foo;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  const A() : foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const @33
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: foo @39
                    staticElement: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
                    element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
                    staticType: null
                  equals: = @43
                  expression: IntegerLiteral
                    literal: 0 @45
                    staticType: int
          augmented
            fields
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            final foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
          accessors
            synthetic get foo @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 33
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: foo @39
                    staticElement: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
                    element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
                    staticType: null
                  equals: = @43
                  expression: IntegerLiteral
                    literal: 0 @45
                    staticType: int
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            foo @51
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
          getters
            synthetic get foo
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo
          type: int
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo
''');
  }

  test_augmented_getter_augments_constructor() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  A.foo();
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            foo @29
              reference: <testLibraryFragment>::@class::A::@constructor::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 28
              nameEnd: 32
          augmented
            constructors
              <testLibraryFragment>::@class::A::@constructor::foo
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_0
              variable: <null>
              augmentationTargetAny: <testLibraryFragment>::@class::A::@constructor::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            foo @29
              reference: <testLibraryFragment>::@class::A::@constructor::foo
              element: <testLibraryFragment>::@class::A::@constructor::foo#element
              typeName: A
              typeNameOffset: 27
              periodOffset: 28
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::A::@constructor::foo
      getters
        get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
''');
  }

  test_augmented_getter_augments_method() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  void foo() {}
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: void
          augmented
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
            methods
              <testLibraryFragment>::@class::A::@method::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_0
              variable: <null>
              augmentationTargetAny: <testLibraryFragment>::@class::A::@method::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @32
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
''');
  }

  test_augmented_getter_augments_setter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  set foo(int _) {}
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              setter: setter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            set foo= @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @39
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_0
              variable: <null>
              augmentationTargetAny: <testLibraryFragment>::@class::A::@setter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @31
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @39
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_augmented_getters_add() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  int get foo2 => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int get foo1 => 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo1 @-1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo1 @35
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo1
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo1
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            synthetic foo2 @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: int
              id: field_1
              getter: getter_1
          accessors
            get foo2 @49
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: field_1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              element: <testLibraryFragment>::@class::A::@field::foo1#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo1 @35
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          fields
            synthetic foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
          getters
            get foo2 @49
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo1
          firstFragment: <testLibraryFragment>::@class::A::@field::foo1
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo1#element
        synthetic foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
          type: int
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo1
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo1
        get foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
''');
  }

  test_augmented_getters_add_generic() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A<T1> {
  T1 get foo2;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A<T1> {
  T1 get foo1;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T1 @23
              defaultType: dynamic
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo1 @-1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              type: T1
              id: field_0
              getter: getter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            abstract get foo1 @38
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: T1
              id: getter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo1
              FieldMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
                augmentationSubstitution: {T1: T1}
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibraryFragment>::@class::A::@getter::foo1
              GetterMember
                base: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
                augmentationSubstitution: {T1: T1}
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          typeParameters
            covariant T1 @37
              defaultType: dynamic
          augmentationTarget: <testLibraryFragment>::@class::A
          fields
            synthetic foo2 @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              type: T1
              id: field_1
              getter: getter_1
          accessors
            abstract get foo2 @52
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: T1
              id: getter_1
              variable: field_1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          typeParameters
            T1 @23
              element: <not-implemented>
          fields
            synthetic foo1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              element: <testLibraryFragment>::@class::A::@field::foo1#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo1 @38
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          typeParameters
            T1 @37
              element: <not-implemented>
          fields
            synthetic foo2
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2#element
              getter2: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
          getters
            get foo2 @52
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T1
      fields
        synthetic foo1
          firstFragment: <testLibraryFragment>::@class::A::@field::foo1
          type: T1
          getter: <testLibraryFragment>::@class::A::@getter::foo1#element
        synthetic foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@field::foo2
          type: T1
          getter: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get foo1
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo1
        abstract get foo2
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getter::foo2
''');
  }

  test_augmented_getters_augment_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @31
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_getters_augment_field2() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
part 'b.dart';
class A {
  int foo = 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
              <testLibraryFragment>::@class::A::@setter::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo
              augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              returnType: int
              id: getter_2
              variable: <null>
              augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            hasInitializer foo @46
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _foo
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_foo#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo
              nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_augmented_getters_augment_getter() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo1 => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int get foo1 => 0;
  int get foo2 => 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo1 @-1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
            synthetic foo2 @-1
              reference: <testLibraryFragment>::@class::A::@field::foo2
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_1
              getter: getter_1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo1 @35
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo1
            get foo2 @56
              reference: <testLibraryFragment>::@class::A::@getter::foo2
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_1
              variable: field_1
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo1
              <testLibraryFragment>::@class::A::@field::foo2
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo1
              <testLibraryFragment>::@class::A::@getter::foo2
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo1 @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo1
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_2
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo1
              reference: <testLibraryFragment>::@class::A::@field::foo1
              element: <testLibraryFragment>::@class::A::@field::foo1#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo1
            synthetic foo2
              reference: <testLibraryFragment>::@class::A::@field::foo2
              element: <testLibraryFragment>::@class::A::@field::foo2#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo2
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo1 @35
              reference: <testLibraryFragment>::@class::A::@getter::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo1
            get foo2 @56
              reference: <testLibraryFragment>::@class::A::@getter::foo2
              element: <testLibraryFragment>::@class::A::@getter::foo2#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo1 @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo1
              element: <testLibraryFragment>::@class::A::@getter::foo1#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo1
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo1
          firstFragment: <testLibraryFragment>::@class::A::@field::foo1
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo1#element
        synthetic foo2
          firstFragment: <testLibraryFragment>::@class::A::@field::foo2
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo2#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo2
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo2
        get foo1
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo1
''');
  }

  test_augmented_getters_augment_getter2_oneLib_oneTop() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  int get foo => 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::1
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::1
            augment get foo @85
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::1
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_2
              variable: <null>
              augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @35
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::1
            augment get foo @85
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::1
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo::@def::0
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_augmented_getters_augment_getter2_twoLib() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
part 'b.dart';
class A {
  int get foo => 0;
}
''');

    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @50
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
              augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
          augmented
            fields
              <testLibraryFragment>::@class::A::@field::foo
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
            accessors
              <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_1
              variable: <null>
              augmentationTarget: <testLibraryFragment>::@class::A::@getter::foo
              augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
              returnType: int
              id: getter_2
              variable: <null>
              augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @50
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibraryFragment>::@class::A::@getter::foo
              nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
              previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_augmented_getters_augment_nothing() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A {
  augment int get foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');

    configuration
      ..withConstructors = false
      ..withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          augmented
            accessors
              <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          accessors
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
              returnType: int
              id: getter_0
              variable: <null>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          getters
            augment get foo @57
              reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
              element: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      getters
        get foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A::@getterAugmentation::foo
''');
  }

  test_augmented_interfaces() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A implements I2 {}
class I2 {}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A implements I1 {}
class I1 {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          interfaces
            I1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            interfaces
              I1
              I2
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
        class I1 @46
          reference: <testLibraryFragment>::@class::I1
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::I1
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          interfaces
            I2
        class I2 @60
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::I2
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class I1 @46
          reference: <testLibraryFragment>::@class::I1
          element: <testLibrary>::@class::I1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              element: <testLibraryFragment>::@class::I1::@constructor::new#element
              typeName: I1
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
        class I2 @60
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          element: <testLibrary>::@class::I2
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new#element
              typeName: I2
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      interfaces
        I1
        I2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class I1
      reference: <testLibrary>::@class::I1
      firstFragment: <testLibraryFragment>::@class::I1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::I1::@constructor::new
    class I2
      reference: <testLibrary>::@class::I2
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
''');
  }

  test_augmented_interfaces_chain() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
part 'b.dart';
augment class A implements I2 {}
class I2 {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
augment class A implements I3 {}
class I3 {}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A implements I1 {}
class I1 {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          interfaces
            I1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            interfaces
              I1
              I2
              I3
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
        class I1 @46
          reference: <testLibraryFragment>::@class::I1
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::I1
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      parts
        part_1
          uri: package:test/b.dart
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          unit: <testLibrary>::@fragment::package:test/b.dart
      classes
        augment class A @50
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          augmentationTarget: <testLibraryFragment>::@class::A
          augmentation: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          interfaces
            I2
        class I2 @75
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::I2
    <testLibrary>::@fragment::package:test/b.dart
      enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
      classes
        augment class A @32
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          augmentationTarget: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          interfaces
            I3
        class I3 @57
          reference: <testLibrary>::@fragment::package:test/b.dart::@class::I3
          enclosingElement3: <testLibrary>::@fragment::package:test/b.dart
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/b.dart::@class::I3::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/b.dart::@class::I3
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class I1 @46
          reference: <testLibraryFragment>::@class::I1
          element: <testLibrary>::@class::I1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              element: <testLibraryFragment>::@class::I1::@constructor::new#element
              typeName: I1
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      nextFragment: <testLibrary>::@fragment::package:test/b.dart
      classes
        class A @50
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
        class I2 @75
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          element: <testLibrary>::@class::I2
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new#element
              typeName: I2
    <testLibrary>::@fragment::package:test/b.dart
      element: <testLibrary>
      enclosingFragment: <testLibrary>::@fragment::package:test/a.dart
      previousFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @32
          reference: <testLibrary>::@fragment::package:test/b.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
        class I3 @57
          reference: <testLibrary>::@fragment::package:test/b.dart::@class::I3
          element: <testLibrary>::@class::I3
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/b.dart::@class::I3::@constructor::new
              element: <testLibrary>::@fragment::package:test/b.dart::@class::I3::@constructor::new#element
              typeName: I3
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      interfaces
        I1
        I2
        I3
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class I1
      reference: <testLibrary>::@class::I1
      firstFragment: <testLibraryFragment>::@class::I1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::I1::@constructor::new
    class I2
      reference: <testLibrary>::@class::I2
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
    class I3
      reference: <testLibrary>::@class::I3
      firstFragment: <testLibrary>::@fragment::package:test/b.dart::@class::I3
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/b.dart::@class::I3::@constructor::new
''');
  }

  test_augmented_interfaces_generic() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A<T2> implements I2<T2> {}
class I2<E> {}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A<T> implements I1 {}
class I1 {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @23
              defaultType: dynamic
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          interfaces
            I1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          augmented
            interfaces
              I1
              I2<T>
            constructors
              <testLibraryFragment>::@class::A::@constructor::new
        class I1 @49
          reference: <testLibraryFragment>::@class::I1
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::I1
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      classes
        augment class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          typeParameters
            covariant T2 @37
              defaultType: dynamic
          augmentationTarget: <testLibraryFragment>::@class::A
          interfaces
            I2<T2>
        class I2 @68
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          typeParameters
            covariant E @71
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              enclosingElement3: <testLibrary>::@fragment::package:test/a.dart::@class::I2
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          nextFragment: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          typeParameters
            T @23
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class I1 @49
          reference: <testLibraryFragment>::@class::I1
          element: <testLibrary>::@class::I1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::I1::@constructor::new
              element: <testLibraryFragment>::@class::I1::@constructor::new#element
              typeName: I1
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      classes
        class A @35
          reference: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          element: <testLibrary>::@class::A
          previousFragment: <testLibraryFragment>::@class::A
          typeParameters
            T2 @37
              element: <not-implemented>
        class I2 @68
          reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2
          element: <testLibrary>::@class::I2
          typeParameters
            E @71
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
              element: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new#element
              typeName: I2
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      interfaces
        I1
        I2<T>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class I1
      reference: <testLibrary>::@class::I1
      firstFragment: <testLibraryFragment>::@class::I1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::I1::@constructor::new
    class I2
      reference: <testLibrary>::@class::I2
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2
      typeParameters
        E
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::I2::@constructor::new
''');
  }

  test_augmented_interfaces_generic_mismatch() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
augment class A<T2, T3> implements I2<T2> {}
class I2<E> {}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A<T> implements I1 {}
class I1 {}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @23
              defaultType: dynamic
          augmentation: <testLibrary>::@fragment::package:test/a.dart::@classAugmentation::A
          interfaces
            I1
          constructors
  