import axios from "axios";
import { addError } from "../errors/errorsActions";

function addPasswordProfile(profile) {
  return {
    type: "ADD_PASSWORD_PROFILE",
    profile,
  };
}

function setPasswordProfiles(profiles) {
  return {
    type: "SET_PASSWORD_PROFILES",
    profiles,
  };
}

function removePasswordProfile(profile) {
  return {
    type: "REMOVE_PASSWORD_PROFILE",
    profile,
  };
}

export function getPasswordProfiles() {
  return (dispatch, getState) => {
    const { settings, auth } = getState();
    return axios
      .get(`${settings.baseURL}/passwords/`, {
        headers: { Authorization: `Bearer ${auth.accessToken}` },
      })
      .then((response) => {
        dispatch(setPasswordProfiles(response.data.results));
        return response;
      });
  };
}

export function savePasswordProfile(profile) {
  return (dispatch, getState) => {
    const { settings, auth } = getState();
    return axios
      .post(`${settings.baseURL}/passwords/`, profile, {
        headers: { Authorization: `Bearer ${auth.accessToken}` },
      })
      .then((response) => {
        dispatch(addPasswordProfile({ ...response.data }));
        return response;
      })
      .catch(() =>
        dispatch(
          addError(
            "We cannot save your password profile. Retry in a few minutes or contact us."
          )
        )
      );
  };
}

export function updatePasswordProfile(profile) {
  return (dispatch, getState) => {
    const { settings, auth } = getState();
    return axios
      .put(`${settings.baseURL}/passwords/${profile.id}/`, profile, {
        headers: { Authorization: `Bearer ${auth.accessToken}` },
      })
      .then((response) => {
        dispatch(addPasswordProfile({ ...response.data }));
        return response;
      })
      .catch(() =>
        dispatch(
          addError(
            "We cannot update your password profile. Retry in a few minutes or contact us."
          )
        )
      );
  };
}

export function deletePasswordProfile(profile) {
  return (dispatch, getState) => {
    const { settings, auth } = getState();
    return axios
      .delete(`${settings.baseURL}/passwords/${profile.id}/`, {
        headers: { Authorization: `Bearer ${auth.accessToken}` },
      })
      .then(() => dispatch(removePasswordProfile(profile)))
      .catch(() =>
        dispatch(
          addError(
            "We cannot delete your password profile. Retry in a few minutes or contact us."
          )
        )
      );
  };
}
