import * as React from 'react';
import type { StyleProp, ViewStyle, View } from 'react-native';

import color from 'color';

import { useInternalTheme } from '../../core/theming';
import { black } from '../../styles/themes/v2/colors';
import type { IconSource } from '../Icon';
import IconButton from '../IconButton/IconButton';

export type Props = React.ComponentPropsWithoutRef<typeof IconButton> & {
  /**
   *  Custom color for action icon.
   */
  color?: string;
  /**
   * Name of the icon to show.
   */
  icon: IconSource;
  /**
   * Optional icon size.
   */
  size?: number;
  /**
   * Whether the button is disabled. A disabled button is greyed out and `onPress` is not called on touch.
   */
  disabled?: boolean;
  /**
   * Accessibility label for the button. This is read by the screen reader when the user taps the button.
   */
  accessibilityLabel?: string;
  /**
   * Function to execute on press.
   */
  onPress?: () => void;
  /**
   * @supported Available in v5.x with theme version 3
   *
   * Whether it's the leading button.
   */
  isLeading?: boolean;
  style?: StyleProp<ViewStyle>;
  ref?: React.RefObject<View>;
};

/**
 * A component used to display an action item in the appbar.
 * <div class="screenshots">
 *   <img class="small" src="screenshots/appbar-action-android.png" />
 * </div>
 *
 * ## Usage
 * ```js
 * import * as React from 'react';
 * import { Appbar } from 'react-native-paper';
 * import { Platform } from 'react-native';
 *
 * const MORE_ICON = Platform.OS === 'ios' ? 'dots-horizontal' : 'dots-vertical';
 *
 * const MyComponent = () => (
 *     <Appbar.Header>
 *        <Appbar.Content title="Title" subtitle={'Subtitle'} />
 *         <Appbar.Action icon="magnify" onPress={() => {}} />
 *         <Appbar.Action icon={MORE_ICON} onPress={() => {}} />
 *     </Appbar.Header>
 * );
 *
 * export default MyComponent;
 * ```
 */
const AppbarAction = React.forwardRef<View, Props>(
  (
    {
      size = 24,
      color: iconColor,
      icon,
      disabled,
      onPress,
      accessibilityLabel,
      isLeading,
      ...rest
    }: Props,
    ref
  ) => {
    const theme = useInternalTheme();

    const actionIconColor = iconColor
      ? iconColor
      : theme.isV3
      ? isLeading
        ? theme.colors.onSurface
        : theme.colors.onSurfaceVariant
      : color(black).alpha(0.54).rgb().string();

    return (
      <IconButton
        size={size}
        onPress={onPress}
        iconColor={actionIconColor}
        icon={icon}
        disabled={disabled}
        accessibilityLabel={accessibilityLabel}
        animated
        ref={ref}
        {...rest}
      />
    );
  }
);

AppbarAction.displayName = 'Appbar.Action';

export default AppbarAction;

// @component-docs ignore-next-line
export { AppbarAction };
