"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash/"));

var _utilities = require("../utilities");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const schema = [];

const create = context => {
  const report = node => {
    context.report({
      loc: node.loc,
      message: 'Duplicate property.',
      node
    });
  };

  const analyzeElement = element => {
    const {
      type
    } = element;
    let value;

    switch (type) {
      case 'GenericTypeAnnotation':
        value = element.id.name;
        break;

      case 'ObjectTypeAnnotation':
        // eslint-disable-next-line no-use-before-define
        value = buildObjectStructure(element.properties);
        break;

      case 'TupleTypeAnnotation':
        // eslint-disable-next-line no-use-before-define
        value = buildArrayStructure(element.types);
        break;

      default:
        value = element.value;
        break;
    }

    return {
      type,
      value
    };
  };

  const buildArrayStructure = elements => _lodash.default.map(elements, element => analyzeElement(element));

  const buildObjectStructure = properties => _lodash.default.map(properties, property => {
    const element = analyzeElement(property.type === 'ObjectTypeSpreadProperty' ? property.argument : property.value);
    return { ...element,
      name: (0, _utilities.getParameterName)(property, context)
    };
  });

  const checkForDuplicates = node => {
    const haystack = []; // filter out complex object types, like ObjectTypeSpreadProperty

    const identifierNodes = _lodash.default.filter(node.properties, {
      type: 'ObjectTypeProperty'
    });

    for (const identifierNode of identifierNodes) {
      const needle = {
        name: (0, _utilities.getParameterName)(identifierNode, context)
      };

      if (identifierNode.value.type === 'FunctionTypeAnnotation') {
        needle.args = _lodash.default.map(identifierNode.value.params, param => analyzeElement(param.typeAnnotation));
      }

      const match = _lodash.default.some(haystack, existingNeedle => _lodash.default.isEqual(existingNeedle, needle));

      if (match) {
        report(identifierNode);
      } else {
        haystack.push(needle);
      }
    }
  };

  return {
    ObjectTypeAnnotation: checkForDuplicates
  };
};

var _default = {
  create,
  schema
};
exports.default = _default;
module.exports = exports.default;