from api import models
from api.models import LessPassUser

from django import forms
from django.contrib import admin
from django.contrib.auth.models import Group
from django.contrib.auth.admin import UserAdmin as BaseUserAdmin
from django.contrib.auth.forms import ReadOnlyPasswordHashField
from django.db.models import Count


class UserCreationForm(forms.ModelForm):
    password1 = forms.CharField(label="Password", widget=forms.PasswordInput)
    password2 = forms.CharField(
        label="Password confirmation", widget=forms.PasswordInput
    )

    class Meta:
        model = LessPassUser
        fields = ("email",)

    def clean_password2(self):
        password1 = self.cleaned_data.get("password1")
        password2 = self.cleaned_data.get("password2")
        if password1 and password2 and password1 != password2:
            raise forms.ValidationError("Passwords don't match")
        return password2

    def save(self, commit=True):
        user = super(UserCreationForm, self).save(commit=False)
        user.set_password(self.cleaned_data["password1"])
        if commit:
            user.save()
        return user


class UserChangeForm(forms.ModelForm):
    password = ReadOnlyPasswordHashField()

    class Meta:
        model = LessPassUser
        fields = ("email", "password", "is_active", "is_admin")

    def clean_password(self):
        return self.initial["password"]


class LessPassUserAdmin(BaseUserAdmin):
    form = UserChangeForm
    add_form = UserCreationForm

    list_display = ("email", "is_admin", "column_passwords_count")
    list_filter = ("is_admin", "is_active")
    fieldsets = (
        (None, {"fields": ("email", "password")}),
        ("Permissions", {"fields": ("is_admin",)}),
    )
    add_fieldsets = (
        (None, {"classes": ("wide",), "fields": ("email", "password1", "password2")}),
    )
    search_fields = ("email",)
    ordering = ("email",)
    filter_horizontal = ()

    def get_queryset(self, request):
        return models.LessPassUser.objects.annotate(passwords_count=Count("passwords"))

    def column_passwords_count(self, instance):
        return instance.passwords_count

    column_passwords_count.short_description = "Password count"
    column_passwords_count.admin_order_field = "passwords_count"


class PasswordAdmin(admin.ModelAdmin):
    list_display = (
        "id",
        "user",
    )
    search_fields = ("user__email",)
    ordering = ("user",)


admin.site.register(models.Password, PasswordAdmin)
admin.site.register(LessPassUser, LessPassUserAdmin)
admin.site.unregister(Group)
