## 数据库概览

NoWakeLock 应用使用 Room 数据库管理两种主要的数据实体：`Info`（统计信息）和 `InfoEvent`（事件记录）。

### 版本信息
- 当前数据库版本：**10**
- 主要升级：添加了 `eventKey` 作为 `InfoEvent` 实体的复合键

## 数据实体

### 1. Info（统计信息）
存储系统事件的统计汇总信息。

**文件位置**: `app/src/main/java/com/js/nowakelock/data/db/entity/Info.kt`

**字段**:
- `name`：事件名称
- `type`：事件类型（Wakelock/Alarm/Service）
- `packageName`：包名
- `userId`：用户ID
- `count`：事件计数
- `blockCount`：阻止次数
- `countTime`：累计时间（毫秒）

### 2. InfoEvent（事件记录）
存储详细的事件记录，包括开始时间、结束时间和阻止状态。

**文件位置**: `app/src/main/java/com/js/nowakelock/data/db/entity/InfoEvent.kt`

**字段**:
- `id`：主键ID
- `name`：事件名称
- `type`：事件类型
- `packageName`：包名
- `userId`：用户ID
- `startTime`：开始时间
- `endTime`：结束时间（可为空）
- `isBlocked`：是否被阻止
- `eventKey`：事件唯一键（新增，用于快速查找）

**关键方法**:
- `generateEventKey()`：生成用于唯一标识事件的键

## 数据访问对象（DAO）

### 1. InfoDao
处理统计信息的数据访问。

**文件位置**: `app/src/main/java/com/js/nowakelock/data/db/dao/InfoDao.kt`

**主要方法**:
- `loadInfos()`：加载统计信息，支持按类型/包名/用户ID筛选
- `loadInfo()`：加载单条统计信息
- `upCountPO()`：增加计数
- `upBlockCountPO()`：增加阻止计数
- `upCountTime()`：增加累计时间
- `rstAllCount()`：重置所有计数
- `rstAllCountTime()`：重置所有时间统计
- `clearAll()`：清除所有数据

### 2. InfoEventDao
处理事件记录的数据访问。

**文件位置**: `app/src/main/java/com/js/nowakelock/data/db/dao/InfoEventDao.kt`

**主要方法**:
- `loadAllEvents()`：加载所有事件
- `loadEvents()`：按类型/包名/用户ID筛选事件
- `loadEventsInTimeRange()`：在时间范围内筛选事件
- `loadEventByKey()`：通过事件键直接查找事件（性能优化）
- `clearAll()`：清除所有事件记录

## 内容提供者接口

**文件位置**: `app/src/main/java/com/js/nowakelock/data/provider/XProvider.kt`

**主要方法**:
- `RecordEvent`：记录事件开始或阻止，同时更新统计信息
- `EndEvent`：记录事件结束（主要用于 Wakelock）
- `LoadInfos`：加载统计汇总数据
- `LoadEvents`：加载详细事件记录
- `ClearData`：清除统计和事件数据
- `CheckHookActive`：检查钩子是否激活

## 接口调用示例

### 记录事件

```kotlin
// 记录普通事件
val bundle = XpRecord.addEvent(
    name = "事件名称",
    packageName = "com.example.app",
    type = Type.Wakelock,
    context = context
)
// 保存事件键以便后续使用
val eventKey = bundle?.getString("eventKey")

// 记录被阻止的事件
XpRecord.blockEvent(
    name = "事件名称",
    packageName = "com.example.app",
    type = Type.Alarm,
    context = context
)
```

### 记录事件结束

```kotlin
// 使用事件键结束事件
XpRecord.endEventWithKey(
    name = "事件名称",
    packageName = "com.example.app",
    context = context,
    eventKey = eventKey
)

// 或者通过重建键结束事件
XpRecord.endEvent(
    name = "事件名称",
    packageName = "com.example.app",
    context = context,
    startTime = startTimeValue
)
```

### 查询数据

```kotlin
// 查询统计信息
val contentResolver = context.contentResolver
val args = Bundle().apply {
    putString("packageName", "com.example.app")
    putString("type", Type.Wakelock.value)
}
val result = contentResolver.call(getURI(), "NoWakelock", ProviderMethod.LoadInfos.value, args)
val infos = result?.getSerializable("infos") as? Array<Info>

// 查询详细事件
val args = Bundle().apply {
    putString("packageName", "com.example.app")
    putLong("startTime", startTimeValue)
    putLong("endTime", endTimeValue)
}
val result = contentResolver.call(getURI(), "NoWakelock", ProviderMethod.LoadEvents.value, args)
val events = result?.getSerializable("events") as? Array<InfoEvent>
```

## 数据库设计亮点

1. **复合键优化**：添加 `eventKey` 实现高效查询，特别是在事件结束时定位对应事件
2. **灵活的查询支持**：提供多种筛选条件组合，满足UI不同场景需求
3. **性能考虑**：使用直接键查找替代复杂条件查询，显著提升性能
4. **进程恢复容错**：通过存储和重建事件键，支持在进程重启后仍能找到对应事件

此架构为UI层提供了丰富的数据访问能力，包括统计信息和详细事件记录，同时保持了良好的性能表现。
