# NoWakeLock 项目 - Wakelock详情页实现方案文档

**日期：** 2025年4月10日

## 目录

1. [项目背景](#项目背景)
2. [需求概述](#需求概述)
3. [方案讨论过程](#方案讨论过程)
4. [最终方案](#最终方案)
   - [数据模型与仓库设计](#数据模型与仓库设计)
   - [ViewModel设计](#viewmodel设计)
   - [UI设计与实现](#ui设计与实现)
   - [导航配置](#导航配置)
5. [代码实现](#代码实现)
   - [数据模型部分](#数据模型部分)
   - [仓库接口与实现](#仓库接口与实现)
   - [ViewModel实现](#viewmodel实现)
   - [UI组件实现](#ui组件实现)
6. [待解决问题](#待解决问题)
7. [总结](#总结)

## 项目背景

NoWakeLock是一个Android应用程序，用于监控和管理设备上的唤醒锁（Wakelock）。该应用允许用户查看设备上的各种唤醒锁，并提供一种机制来阻止可能导致电池耗尽的唤醒锁。项目基于Jetpack Compose构建，遵循Material Design 3设计规范。

## 需求概述

本次开发任务是实现一个Wakelock详情页，该页面需要显示特定Wakelock的详细信息，包括：
- 基本信息（名称、包名等）
- 统计数据（触发次数、总时间等）
- 详细描述和推荐信息
- 阻止设置（是否阻止、条件设置等）
- 活动时间轴（显示过去24小时的活动情况）
- 最近的活动记录

页面需要遵循Material Design 3规范，并与应用程序现有的架构和设计风格保持一致。

## 方案讨论过程

### 初步架构设计

最初，我们提出了一个包含以下组件的方案：
- 数据模型实现：`DAInfoEntry.kt`
- 数据管理器：`DAInfoManager.kt`
- 页面ViewModel：`DADetailViewModel.kt`
- UI界面：`DADetailScreen.kt`及其相关组件

### 调整数据管理方式

在讨论过程中，根据用户提供的现有代码库结构，我们发现原来设计的`DAInfoManager`需要调整为符合仓库模式的实现。因此，我们修改了方案，采用Repository Pattern：
- 创建`DAInfoRepository`接口
- 实现`DAInfoRepositoryImpl`类
- 适配现有的数据获取逻辑

### ViewModel设计调整

最初ViewModel设计中，参数绑定在初始化时进行。根据用户反馈，我们调整了设计，使ViewModel更加灵活可重用：
- 添加`loadDADetail`方法接收参数，而不是在初始化时绑定
- 使用`MutableStateFlow`管理UI和设置状态
- 添加安全检查机制确保标识符有效性

### UI组件规划

UI部分采用了模块化设计，将详情页分解为多个独立组件：
- 头部区域：显示Wakelock名称和包名
- 统计卡片：展示各种统计数据
- 信息卡片：显示详细描述和建议
- 设置卡片：提供阻止和条件设置的控制
- 时间轴卡片：可视化24小时活动
- 近期活动卡片：列出最近的事件

## 最终方案

### 数据模型与仓库设计

1. **DAInfoEntry**：
   - 数据类，包含Wakelock的详细信息
   - 字段：id, name, type, packageName, safeToBlock, description, recommendation, warning

2. **DAInfoRepository**：
   - 接口，定义获取Wakelock详细信息的方法
   - 方法：getDAInfo, hasInfoFor

3. **DAInfoRepositoryImpl**：
   - DAInfoRepository的实现
   - 从JSON文件懒加载数据
   - 实现查找匹配项的逻辑

4. **DARepository**：
   - 接口，定义获取Wakelock数据和事件的方法
   - 方法：getDAItem, getRecentEvents, getEventsInTimeRange, updateBlockingSetting等

5. **DARepositoryImpl**：
   - DARepository的实现
   - 通过DAO访问数据库
   - 实现各种数据查询和更新操作

### ViewModel设计

**DADetailViewModel**：
- 使用`MutableStateFlow`管理UI状态
- 提供`loadDADetail`方法接收参数
- 方法：updateBlockingSetting, updateConditionSettings, updateTimeInterval
- 实现统计数据计算和时间轴数据生成

### UI设计与实现

**主要组件**：
- `DADetailScreen`：主界面组件，基于状态显示内容
- `DAHeaderSection`：显示Wakelock名称和包名
- `StatisticsCard`：显示统计数据
- `InfoCard`：显示详细信息
- `SettingsCard`：提供设置控制
- `TimelineCard`：显示活动时间轴
- `RecentActivitiesCard`：显示最近活动

### 导航配置

1. **NavRoutes**：
   - 添加`DA_DETAIL`路由，包含参数：daId, packageName

2. **NavGraph**：
   - 添加详情页composable
   - 配置参数传递

## 代码实现

### 数据模型部分

**DAInfoEntry.kt**
```kotlin
package com.js.nowakelock.data.model

/**
 * 表示设备自动化项目的详细信息
 */
data class DAInfoEntry(
    val id: String,                // 唯一标识符
    val name: String,              // 显示名称
    val type: String,              // 类型（如wakelock, alarm等）
    val packageName: String? = null, // 包名（如果有）
    val safeToBlock: Boolean = false, // 是否安全阻止
    val description: String? = null,  // 详细描述
    val recommendation: String? = null, // 推荐处理方式
    val warning: String? = null       // 阻止警告
)
```

### 仓库接口与实现

**DAInfoRepository.kt**
```kotlin
package com.js.nowakelock.data.repository.dainfo

import com.js.nowakelock.data.model.DAInfoEntry

/**
 * 设备自动化项目详细信息的存储库接口
 */
interface DAInfoRepository {
    /**
     * 获取特定DA项目的详细信息
     * @param daId DA项目ID
     * @param packageName 包名（可选）
     * @return 详细信息或null（如果不存在）
     */
    suspend fun getDAInfo(daId: String, packageName: String? = null): DAInfoEntry?
    
    /**
     * 检查是否存在特定DA项目的详细信息
     * @param daId DA项目ID
     * @param packageName 包名（可选）
     * @return 如果存在信息则为true
     */
    suspend fun hasInfoFor(daId: String, packageName: String? = null): Boolean
}
```

**DAInfoRepositoryImpl.kt**
```kotlin
package com.js.nowakelock.data.repository.dainfo

import com.js.nowakelock.data.model.DAInfoEntry
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.Json
import kotlinx.serialization.decodeFromString
import android.content.Context
import kotlinx.serialization.builtins.ListSerializer
import java.io.IOException
import timber.log.Timber

/**
 * DAInfoRepository的实现
 * 从JSON资源文件加载数据
 */
class DAInfoRepositoryImpl(
    private val context: Context,
    private val json: Json = Json { ignoreUnknownKeys = true }
) : DAInfoRepository {
    
    private val infoList: List<DAInfoEntry> by lazy {
        loadDAInfoList()
    }
    
    private fun loadDAInfoList(): List<DAInfoEntry> {
        return try {
            val jsonString = context.assets.open("da_info.json").bufferedReader().use { it.readText() }
            json.decodeFromString(ListSerializer(DAInfoEntry.serializer()), jsonString)
        } catch (e: IOException) {
            Timber.e(e, "Failed to load DA info from assets")
            emptyList()
        }
    }
    
    override suspend fun getDAInfo(daId: String, packageName: String?): DAInfoEntry? = withContext(Dispatchers.IO) {
        infoList.find { entry -> 
            entry.id == daId && (packageName == null || packageName == entry.packageName)
        }
    }
    
    override suspend fun hasInfoFor(daId: String, packageName: String?): Boolean = withContext(Dispatchers.IO) {
        infoList.any { entry -> 
            entry.id == daId && (packageName == null || packageName == entry.packageName)
        }
    }
}
```

**DARepository.kt**
```kotlin
package com.js.nowakelock.data.repository.daitem

import com.js.nowakelock.data.db.entity.InfoEvent
import com.js.nowakelock.data.db.entity.St
import com.js.nowakelock.data.model.DAItem
import kotlinx.coroutines.flow.Flow

interface DARepository {
    /**
     * 按名称排序检索所有DAItem
     */
    suspend fun getDAItemsSortedByName(
        packageName: String = "",
        userId: Int = -1
    ): Flow<List<DAItem>>

    /**
     * 按计数排序检索所有DAItem
     */
    suspend fun getDAItemsSortedByCount(
        packageName: String = "",
        userId: Int = -1
    ): Flow<List<DAItem>>

    /**
     * 按时间排序检索所有DAItem
     */
    suspend fun getDAItemsSortedByTime(
        packageName: String = "",
        userId: Int = -1
    ): Flow<List<DAItem>>

    /**
     * 通过ID、包名和用户ID检索DAItem
     */
    suspend fun getDAItem(daId: String, packageName: String?, userId: Int): DAItem?

    /**
     * 获取特定DA项自指定时间以来的最近事件
     * @param daId DA项ID
     * @param packageName 包名（可选）
     * @param userId 用户ID
     * @param startTime 开始时间（毫秒）
     * @return 事件列表
     */
    suspend fun getRecentEvents(
        daId: String, 
        packageName: String?, 
        userId: Int, 
        startTime: Long
    ): List<InfoEvent>

    /**
     * 获取特定DA项在给定时间范围内的事件
     * @param daId DA项ID
     * @param packageName 包名（可选）
     * @param userId 用户ID
     * @param startTime 开始时间（毫秒）
     * @param endTime 结束时间（毫秒）
     * @return 事件列表
     */
    suspend fun getEventsInTimeRange(
        daId: String,
        packageName: String?,
        userId: Int,
        startTime: Long,
        endTime: Long
    ): List<InfoEvent>

    /**
     * 更新DA项的阻止设置
     */
    suspend fun updateBlockingSetting(
        daId: String,
        packageName: String?,
        userId: Int,
        enabled: Boolean
    )

    /**
     * 更新DA项的条件设置
     */
    suspend fun updateConditionSettings(
        daId: String,
        packageName: String?,
        userId: Int,
        sleepOnly: Boolean,
        screenOffOnly: Boolean
    )

    /**
     * 更新DA项的时间间隔设置
     */
    suspend fun updateTimeInterval(
        daId: String,
        packageName: String?,
        userId: Int,
        seconds: Int
    )

    /**
     * 更新DAItem设置（阻止状态和时间窗口）
     */
    suspend fun updateDAItemSettings(
        setting: St
    )

    /**
     * 与系统同步wakelock数据
     */
    suspend fun syncDB(packageName: String = "", userId: Int = -1)

    /**
     * 将InfoEvent数据从XProvider同步到AppDatabase
     */
    suspend fun syncInfoEvents(
        packageName: String = "",
        userId: Int = -1,
        startTime: Long = 0,
        endTime: Long = 0
    )
}
```

**DARepositoryImpl.kt**（部分实现）
```kotlin
package com.js.nowakelock.data.repository.daitem

import com.js.nowakelock.data.db.dao.DADao
import com.js.nowakelock.data.db.dao.InfoEventDao
import com.js.nowakelock.data.db.dao.StDao
import com.js.nowakelock.data.db.entity.InfoEvent
import com.js.nowakelock.data.db.entity.St
import com.js.nowakelock.data.model.DAItem
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.withContext
import timber.log.Timber

class DARepositoryImpl(
    private val daDao: DADao,
    private val infoEventDao: InfoEventDao,
    private val stDao: StDao
) : DARepository {

    override suspend fun getDAItem(
        daId: String,
        packageName: String?,
        userId: Int
    ): DAItem? = withContext(Dispatchers.IO) {
        try {
            val item = if (packageName.isNullOrEmpty()) {
                daDao.getDaItemById(daId, userId)
            } else {
                daDao.getDaItemByIdAndPackage(daId, packageName, userId)
            }
            
            return@withContext item
        } catch (e: Exception) {
            Timber.e(e, "Error getting DA item: $daId")
            return@withContext null
        }
    }

    override suspend fun getRecentEvents(
        daId: String,
        packageName: String?,
        userId: Int,
        startTime: Long
    ): List<InfoEvent> = withContext(Dispatchers.IO) {
        try {
            if (packageName.isNullOrEmpty()) {
                infoEventDao.getEventsSinceTime(daId, userId, startTime)
            } else {
                infoEventDao.getEventsSinceTimeByPackage(daId, packageName, userId, startTime)
            }
        } catch (e: Exception) {
            Timber.e(e, "Error getting recent events for: $daId")
            emptyList()
        }
    }

    override suspend fun getEventsInTimeRange(
        daId: String,
        packageName: String?,
        userId: Int,
        startTime: Long,
        endTime: Long
    ): List<InfoEvent> = withContext(Dispatchers.IO) {
        try {
            if (packageName.isNullOrEmpty()) {
                infoEventDao.getEventsInTimeRange(daId, userId, startTime, endTime)
            } else {
                infoEventDao.getEventsInTimeRangeByPackage(daId, packageName, userId, startTime, endTime)
            }
        } catch (e: Exception) {
            Timber.e(e, "Error getting events in time range for: $daId")
            emptyList()
        }
    }

    // 更多实现...
}
```

### ViewModel实现

**DADetailViewModel.kt**
```kotlin
package com.js.nowakelock.ui.screens.das

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.js.nowakelock.data.db.entity.InfoEvent
import com.js.nowakelock.data.model.DAInfoEntry
import com.js.nowakelock.data.model.DAItem
import com.js.nowakelock.data.repository.dainfo.DAInfoRepository
import com.js.nowakelock.data.repository.daitem.DARepository
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch
import timber.log.Timber
import java.util.Calendar
import java.util.concurrent.TimeUnit

/**
 * Wakelock详情页的统计信息
 */
data class DAStatistics(
    val totalCount: Int = 0,
    val blockedCount: Int = 0,
    val totalDuration: Long = 0,   // 毫秒
    val averageDuration: Long = 0, // 毫秒
    val percentage: Float = 0f     // 阻止百分比
)

/**
 * Wakelock详情页的状态
 */
sealed class DADetailState {
    object Loading : DADetailState()
    data class Success(
        val daItem: DAItem,
        val daInfo: DAInfoEntry? = null,
        val statistics: DAStatistics = DAStatistics(),
        val timelineData: List<Pair<Long, Pair<Int, Int>>> = emptyList(), // 时间戳, (总数, 阻止数)
        val recentEvents: List<InfoEvent> = emptyList(),
        val isBlocked: Boolean = false,
        val sleepOnly: Boolean = false,
        val screenOffOnly: Boolean = false,
        val timeIntervalSeconds: Int = 0
    ) : DADetailState()
    data class Error(val message: String) : DADetailState()
}

/**
 * Wakelock详情页的ViewModel
 */
class DADetailViewModel(
    private val daRepository: DARepository,
    private val daInfoRepository: DAInfoRepository
) : ViewModel() {
    
    private val _uiState = MutableStateFlow<DADetailState>(DADetailState.Loading)
    val uiState: StateFlow<DADetailState> = _uiState.asStateFlow()
    
    // 当前加载的项目标识符
    private var currentDaId: String? = null
    private var currentPackageName: String? = null
    private var currentUserId: Int = -1
    
    /**
     * 加载Wakelock详情
     */
    fun loadDADetail(daId: String, packageName: String?, userId: Int) {
        if (daId.isEmpty()) {
            _uiState.value = DADetailState.Error("Invalid DA ID")
            return
        }
        
        currentDaId = daId
        currentPackageName = packageName
        currentUserId = userId
        
        viewModelScope.launch {
            _uiState.value = DADetailState.Loading
            
            try {
                // 获取DA项目基本信息
                val daItem = daRepository.getDAItem(daId, packageName, userId)
                if (daItem == null) {
                    _uiState.value = DADetailState.Error("DA Item not found")
                    return@launch
                }
                
                // 获取详细信息
                val daInfo = daInfoRepository.getDAInfo(daId, packageName)
                
                // 获取24小时内的事件用于统计和时间轴
                val endTime = System.currentTimeMillis()
                val startTime = endTime - TimeUnit.HOURS.toMillis(24)
                val events = daRepository.getEventsInTimeRange(daId, packageName, userId, startTime, endTime)
                
                // 获取最近的10个事件
                val recentStartTime = endTime - TimeUnit.DAYS.toMillis(7) // 最近7天
                val recentEvents = daRepository.getRecentEvents(daId, packageName, userId, recentStartTime)
                    .take(10) // 最多10个事件
                
                // 计算统计数据
                val statistics = calculateStatistics(events)
                
                // 生成时间轴数据
                val timelineData = generateTimelineData(events, startTime, endTime)
                
                // 成功加载，更新UI状态
                _uiState.value = DADetailState.Success(
                    daItem = daItem,
                    daInfo = daInfo,
                    statistics = statistics,
                    timelineData = timelineData,
                    recentEvents = recentEvents,
                    isBlocked = daItem.fullBlocked,
                    sleepOnly = daItem.dozeMode,
                    screenOffOnly = daItem.screenOffBlock,
                    timeIntervalSeconds = daItem.timeWindowSec
                )
                
            } catch (e: Exception) {
                Timber.e(e, "Error loading DA detail")
                _uiState.value = DADetailState.Error("Failed to load data: ${e.message}")
            }
        }
    }
    
    /**
     * 更新阻止设置
     */
    fun updateBlockingSetting(enabled: Boolean) {
        val currentId = currentDaId
        val currentPkg = currentPackageName
        val currentUser = currentUserId
        
        if (currentId == null || currentUser < 0) {
            return
        }
        
        viewModelScope.launch {
            try {
                daRepository.updateBlockingSetting(currentId, currentPkg, currentUser, enabled)
                
                // 更新UI状态
                updateSettingInUiState { state -> 
                    state.copy(isBlocked = enabled) 
                }
            } catch (e: Exception) {
                Timber.e(e, "Error updating blocking setting")
            }
        }
    }
    
    /**
     * 更新条件设置
     */
    fun updateConditionSettings(sleepOnly: Boolean, screenOffOnly: Boolean) {
        val currentId = currentDaId
        val currentPkg = currentPackageName
        val currentUser = currentUserId
        
        if (currentId == null || currentUser < 0) {
            return
        }
        
        viewModelScope.launch {
            try {
                daRepository.updateConditionSettings(currentId, currentPkg, currentUser, sleepOnly, screenOffOnly)
                
                // 更新UI状态
                updateSettingInUiState { state -> 
                    state.copy(sleepOnly = sleepOnly, screenOffOnly = screenOffOnly) 
                }
            } catch (e: Exception) {
                Timber.e(e, "Error updating condition settings")
            }
        }
    }
    
    /**
     * 更新时间间隔设置
     */
    fun updateTimeInterval(seconds: Int) {
        val currentId = currentDaId
        val currentPkg = currentPackageName
        val currentUser = currentUserId
        
        if (currentId == null || currentUser < 0) {
            return
        }
        
        viewModelScope.launch {
            try {
                daRepository.updateTimeInterval(currentId, currentPkg, currentUser, seconds)
                
                // 更新UI状态
                updateSettingInUiState { state -> 
                    state.copy(timeIntervalSeconds = seconds) 
                }
            } catch (e: Exception) {
                Timber.e(e, "Error updating time interval")
            }
        }
    }
    
    /**
     * 计算统计数据
     */
    private fun calculateStatistics(events: List<InfoEvent>): DAStatistics {
        if (events.isEmpty()) {
            return DAStatistics()
        }
        
        val totalCount = events.size
        val blockedCount = events.count { it.blocked }
        val totalDuration = events.sumOf { it.durationMs }
        val averageDuration = if (totalCount > 0) totalDuration / totalCount else 0
        val percentage = if (totalCount > 0) blockedCount.toFloat() / totalCount else 0f
        
        return DAStatistics(
            totalCount = totalCount,
            blockedCount = blockedCount,
            totalDuration = totalDuration,
            averageDuration = averageDuration,
            percentage = percentage
        )
    }
    
    /**
     * 生成时间轴数据
     * 返回24个小时段的数据点
     */
    private fun generateTimelineData(
        events: List<InfoEvent>,
        startTime: Long,
        endTime: Long
    ): List<Pair<Long, Pair<Int, Int>>> {
        val hourlyData = mutableListOf<Pair<Long, Pair<Int, Int>>>()
        val hourInMs = TimeUnit.HOURS.toMillis(1)
        
        // 创建24个小时的时间段
        for (i in 0 until 24) {
            val segmentStart = startTime + (i * hourInMs)
            val segmentEnd = segmentStart + hourInMs
            
            // 计算该时间段内的事件总数和阻止数
            val segmentEvents = events.filter { it.timestamp in segmentStart until segmentEnd }
            val totalCount = segmentEvents.size
            val blockedCount = segmentEvents.count { it.blocked }
            
            hourlyData.add(Pair(segmentStart, Pair(totalCount, blockedCount)))
        }
        
        return hourlyData
    }
    
    /**
     * 辅助方法：更新UI状态中的设置
     */
    private fun updateSettingInUiState(update: (DADetailState.Success) -> DADetailState.Success) {
        val currentState = _uiState.value
        if (currentState is DADetailState.Success) {
            _uiState.value = update(currentState)
        }
    }
}
```

### UI组件实现

**DADetailScreen.kt**（部分实现，UI组件）
```kotlin
package com.js.nowakelock.ui.screens.das

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import com.js.nowakelock.R
import org.koin.androidx.compose.getViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DADetailScreen(
    daId: String,
    packageName: String?,
    viewModel: DADetailViewModel = getViewModel(),
    userId: Int = 0
) {
    // 触发数据加载
    LaunchedEffect(daId, packageName, userId) {
        viewModel.loadDADetail(daId, packageName, userId)
    }
    
    // 观察UI状态
    val uiState by viewModel.uiState.collectAsState()
    
    Scaffold { paddingValues ->
        Box(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
        ) {
            when (val state = uiState) {
                is DADetailState.Loading -> LoadingIndicator()
                is DADetailState.Error -> ErrorMessage(state.message)
                is DADetailState.Success -> DADetailContent(
                    state = state,
                    onBlockingChanged = viewModel::updateBlockingSetting,
                    onConditionSettingsChanged = viewModel::updateConditionSettings,
                    onTimeIntervalChanged = viewModel::updateTimeInterval
                )
            }
        }
    }
}

@Composable
private fun LoadingIndicator() {
    Box(
        modifier = Modifier.fillMaxSize(),
        contentAlignment = Alignment.Center
    ) {
        CircularProgressIndicator()
    }
}

@Composable
private fun ErrorMessage(message: String) {
    Box(
        modifier = Modifier.fillMaxSize(),
        contentAlignment = Alignment.Center
    ) {
        Text(
            text = stringResource(R.string.error_loading_data, message),
            style = MaterialTheme.typography.bodyLarge,
            textAlign = TextAlign.Center,
            modifier = Modifier.padding(16.dp)
        )
    }
}

@Composable
private fun DADetailContent(
    state: DADetailState.Success,
    onBlockingChanged: (Boolean) -> Unit,
    onConditionSettingsChanged: (Boolean, Boolean) -> Unit,
    onTimeIntervalChanged: (Int) -> Unit
) {
    LazyColumn(
        modifier = Modifier
            .fillMaxSize()
            .padding(16.dp),
        verticalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        item {
            DAHeaderSection(
                name = state.daItem.name,
                packageName = state.daItem.packageName
            )
        }
        
        item {
            StatisticsCard(statistics = state.statistics)
        }
        
        item {
            if (state.daInfo != null) {
                InfoCard(daInfo = state.daInfo)
            }
        }
        
        item {
            SettingsCard(
                isBlocked = state.isBlocked,
                sleepOnly = state.sleepOnly,
                screenOffOnly = state.screenOffOnly,
                timeIntervalSeconds = state.timeIntervalSeconds,
                onBlockingChanged = onBlockingChanged,
                onConditionSettingsChanged = onConditionSettingsChanged,
                onTimeIntervalChanged = onTimeIntervalChanged
            )
        }
        
        item {
            TimelineCard(timelineData = state.timelineData)
        }
        
        item {
            Re
继续文档内容：

```kotlin
        item {
            RecentActivitiesCard(recentEvents = state.recentEvents)
        }
    }
}

@Composable
private fun DAHeaderSection(
    name: String,
    packageName: String
) {
    Column(
        modifier = Modifier.fillMaxWidth(),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        // Wakelock图标
        Surface(
            modifier = Modifier.size(72.dp),
            shape = MaterialTheme.shapes.extraLarge,
            color = MaterialTheme.colorScheme.primaryContainer
        ) {
            Box(contentAlignment = Alignment.Center) {
                Text(
                    text = name.take(1).uppercase(),
                    style = MaterialTheme.typography.headlineLarge,
                    color = MaterialTheme.colorScheme.onPrimaryContainer
                )
            }
        }
        
        Spacer(modifier = Modifier.height(8.dp))
        
        // Wakelock名称
        Text(
            text = name,
            style = MaterialTheme.typography.headlineSmall,
            color = MaterialTheme.colorScheme.onSurface
        )
        
        // 包名
        Text(
            text = packageName,
            style = MaterialTheme.typography.bodyMedium,
            color = MaterialTheme.colorScheme.onSurfaceVariant
        )
    }
}

@Composable
private fun StatisticsCard(statistics: DAStatistics) {
    ElevatedCard(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = stringResource(R.string.statistics),
                style = MaterialTheme.typography.titleLarge
            )
            
            Spacer(modifier = Modifier.height(16.dp))
            
            // 统计数据行
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceEvenly
            ) {
                StatItem(
                    value = statistics.totalCount.toString(),
                    label = stringResource(R.string.total_count)
                )
                
                VerticalDivider()
                
                StatItem(
                    value = formatDuration(statistics.totalDuration),
                    label = stringResource(R.string.total_time)
                )
                
                VerticalDivider()
                
                StatItem(
                    value = formatDuration(statistics.averageDuration),
                    label = stringResource(R.string.average_time)
                )
            }
            
            Spacer(modifier = Modifier.height(16.dp))
            
            // 阻止统计
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceEvenly
            ) {
                StatItem(
                    value = statistics.blockedCount.toString(),
                    label = stringResource(R.string.blocked_count)
                )
                
                VerticalDivider()
                
                StatItem(
                    value = String.format("%.1f%%", statistics.percentage * 100),
                    label = stringResource(R.string.blocked_percentage)
                )
            }
        }
    }
}

@Composable
private fun StatItem(value: String, label: String) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text(
            text = value,
            style = MaterialTheme.typography.headlineSmall,
            color = MaterialTheme.colorScheme.onSurface
        )
        Text(
            text = label,
            style = MaterialTheme.typography.bodySmall,
            color = MaterialTheme.colorScheme.onSurfaceVariant
        )
    }
}

@Composable
private fun VerticalDivider() {
    Divider(
        modifier = Modifier
            .height(48.dp)
            .width(1.dp),
        color = MaterialTheme.colorScheme.outlineVariant
    )
}

@Composable
private fun InfoCard(daInfo: DAInfoEntry) {
    ElevatedCard(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = stringResource(R.string.information),
                style = MaterialTheme.typography.titleLarge
            )
            
            Spacer(modifier = Modifier.height(16.dp))
            
            // 描述
            if (!daInfo.description.isNullOrEmpty()) {
                Text(
                    text = stringResource(R.string.description),
                    style = MaterialTheme.typography.titleMedium,
                    color = MaterialTheme.colorScheme.onSurface
                )
                
                Spacer(modifier = Modifier.height(4.dp))
                
                Text(
                    text = daInfo.description,
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                
                Spacer(modifier = Modifier.height(12.dp))
            }
            
            // 推荐
            if (!daInfo.recommendation.isNullOrEmpty()) {
                Text(
                    text = stringResource(R.string.recommendation),
                    style = MaterialTheme.typography.titleMedium,
                    color = MaterialTheme.colorScheme.onSurface
                )
                
                Spacer(modifier = Modifier.height(4.dp))
                
                Text(
                    text = daInfo.recommendation,
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                
                Spacer(modifier = Modifier.height(12.dp))
            }
            
            // 警告
            if (!daInfo.warning.isNullOrEmpty()) {
                Text(
                    text = stringResource(R.string.warning),
                    style = MaterialTheme.typography.titleMedium,
                    color = MaterialTheme.colorScheme.error
                )
                
                Spacer(modifier = Modifier.height(4.dp))
                
                Text(
                    text = daInfo.warning,
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.error
                )
            }
        }
    }
}

@Composable
private fun SettingsCard(
    isBlocked: Boolean,
    sleepOnly: Boolean,
    screenOffOnly: Boolean,
    timeIntervalSeconds: Int,
    onBlockingChanged: (Boolean) -> Unit,
    onConditionSettingsChanged: (Boolean, Boolean) -> Unit,
    onTimeIntervalChanged: (Int) -> Unit
) {
    ElevatedCard(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = stringResource(R.string.settings),
                style = MaterialTheme.typography.titleLarge
            )
            
            Spacer(modifier = Modifier.height(16.dp))
            
            // 阻止开关
            SwitchItem(
                title = stringResource(R.string.block_wakelock),
                subtitle = stringResource(R.string.block_wakelock_description),
                checked = isBlocked,
                onCheckedChange = onBlockingChanged
            )
            
            Divider(modifier = Modifier.padding(vertical = 8.dp))
            
            // 条件设置
            if (isBlocked) {
                Text(
                    text = stringResource(R.string.conditions),
                    style = MaterialTheme.typography.titleMedium
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                // 仅在Doze模式下阻止
                SwitchItem(
                    title = stringResource(R.string.only_in_doze),
                    subtitle = stringResource(R.string.only_in_doze_description),
                    checked = sleepOnly,
                    onCheckedChange = { newValue ->
                        onConditionSettingsChanged(newValue, screenOffOnly)
                    }
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                // 仅在屏幕关闭时阻止
                SwitchItem(
                    title = stringResource(R.string.only_when_screen_off),
                    subtitle = stringResource(R.string.only_when_screen_off_description),
                    checked = screenOffOnly,
                    onCheckedChange = { newValue ->
                        onConditionSettingsChanged(sleepOnly, newValue)
                    }
                )
                
                Divider(modifier = Modifier.padding(vertical = 8.dp))
                
                // 时间间隔设置
                Text(
                    text = stringResource(R.string.time_interval),
                    style = MaterialTheme.typography.titleMedium
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                Text(
                    text = stringResource(R.string.time_interval_description),
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                TimeIntervalInput(
                    seconds = timeIntervalSeconds,
                    onSecondsChanged = onTimeIntervalChanged
                )
            }
        }
    }
}

@Composable
private fun SwitchItem(
    title: String,
    subtitle: String? = null,
    checked: Boolean,
    onCheckedChange: (Boolean) -> Unit
) {
    Row(
        modifier = Modifier
            .fillMaxWidth()
            .padding(vertical = 4.dp),
        verticalAlignment = Alignment.CenterVertically
    ) {
        Column(
            modifier = Modifier.weight(1f)
        ) {
            Text(
                text = title,
                style = MaterialTheme.typography.bodyLarge
            )
            if (subtitle != null) {
                Text(
                    text = subtitle,
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }
        }
        
        Switch(
            checked = checked,
            onCheckedChange = onCheckedChange
        )
    }
}

@Composable
private fun TimeIntervalInput(
    seconds: Int,
    onSecondsChanged: (Int) -> Unit
) {
    // 用于编辑的状态
    var inputValue by remember { mutableStateOf(seconds.toString()) }
    
    Row(
        modifier = Modifier
            .fillMaxWidth()
            .padding(4.dp),
        verticalAlignment = Alignment.CenterVertically
    ) {
        OutlinedTextField(
            value = inputValue,
            onValueChange = { value ->
                // 只允许数字输入
                if (value.isEmpty() || value.all { it.isDigit() }) {
                    inputValue = value
                    value.toIntOrNull()?.let { onSecondsChanged(it) }
                }
            },
            modifier = Modifier.weight(1f),
            label = { Text(stringResource(R.string.seconds)) },
            singleLine = true,
            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number)
        )
        
        Spacer(modifier = Modifier.width(16.dp))
        
        // 预设时间按钮
        Column {
            PresetTimeButton(
                text = stringResource(R.string.minute, 1),
                onClick = {
                    val newValue = 60
                    inputValue = newValue.toString()
                    onSecondsChanged(newValue)
                }
            )
            
            Spacer(modifier = Modifier.height(8.dp))
            
            PresetTimeButton(
                text = stringResource(R.string.minutes, 5),
                onClick = {
                    val newValue = 300
                    inputValue = newValue.toString()
                    onSecondsChanged(newValue)
                }
            )
        }
    }
}

@Composable
private fun PresetTimeButton(
    text: String,
    onClick: () -> Unit
) {
    OutlinedButton(
        onClick = onClick,
        modifier = Modifier.width(100.dp)
    ) {
        Text(text)
    }
}

@Composable
private fun TimelineCard(timelineData: List<Pair<Long, Pair<Int, Int>>>) {
    ElevatedCard(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = stringResource(R.string.activity_timeline),
                style = MaterialTheme.typography.titleLarge
            )
            
            Spacer(modifier = Modifier.height(16.dp))
            
            if (timelineData.isEmpty() || timelineData.all { it.second.first == 0 }) {
                // 无数据显示
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(200.dp),
                    contentAlignment = Alignment.Center
                ) {
                    Text(
                        text = stringResource(R.string.no_activity_data),
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
            } else {
                // 时间轴图表
                TimelineChart(
                    timelineData = timelineData,
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(200.dp)
                )
            }
            
            Spacer(modifier = Modifier.height(8.dp))
            
            // 图例
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.Center
            ) {
                Legend(
                    color = MaterialTheme.colorScheme.primary,
                    text = stringResource(R.string.total)
                )
                
                Spacer(modifier = Modifier.width(24.dp))
                
                Legend(
                    color = MaterialTheme.colorScheme.error,
                    text = stringResource(R.string.blocked)
                )
            }
        }
    }
}

@Composable
private fun Legend(color: Color, text: String) {
    Row(
        verticalAlignment = Alignment.CenterVertically
    ) {
        Box(
            modifier = Modifier
                .size(12.dp)
                .background(color, CircleShape)
        )
        
        Spacer(modifier = Modifier.width(4.dp))
        
        Text(
            text = text,
            style = MaterialTheme.typography.bodySmall
        )
    }
}

@Composable
private fun TimelineChart(
    timelineData: List<Pair<Long, Pair<Int, Int>>>,
    modifier: Modifier = Modifier
) {
    val maxCount = timelineData.maxOfOrNull { it.second.first } ?: 0
    
    Canvas(modifier = modifier) {
        val chartWidth = size.width
        val chartHeight = size.height * 0.8f
        val barWidth = chartWidth / timelineData.size * 0.7f
        val spacing = chartWidth / timelineData.size * 0.3f
        
        // 绘制时间轴
        drawLine(
            color = MaterialTheme.colorScheme.outlineVariant,
            start = Offset(0f, chartHeight),
            end = Offset(chartWidth, chartHeight),
            strokeWidth = 1.dp.toPx()
        )
        
        // 绘制数据条形
        timelineData.forEachIndexed { index, (timestamp, counts) ->
            val x = index * (barWidth + spacing) + spacing / 2
            val totalHeight = if (maxCount > 0) (counts.first.toFloat() / maxCount) * chartHeight else 0f
            val blockedHeight = if (maxCount > 0) (counts.second.toFloat() / maxCount) * chartHeight else 0f
            
            // 总数条形
            drawRect(
                color = MaterialTheme.colorScheme.primary,
                topLeft = Offset(x, chartHeight - totalHeight),
                size = Size(barWidth, totalHeight)
            )
            
            // 阻止条形（叠加在总数上）
            if (blockedHeight > 0) {
                drawRect(
                    color = MaterialTheme.colorScheme.error,
                    topLeft = Offset(x, chartHeight - blockedHeight),
                    size = Size(barWidth, blockedHeight)
                )
            }
            
            // 每隔几小时显示一个时间标签
            if (index % 4 == 0) {
                val hour = SimpleDateFormat("HH", Locale.getDefault()).format(Date(timestamp))
                drawContext.canvas.nativeCanvas.drawText(
                    hour,
                    x + barWidth / 2,
                    chartHeight + 12.dp.toPx(),
                    Paint().apply {
                        color = android.graphics.Color.parseColor(MaterialTheme.colorScheme.onSurfaceVariant.toArgb().toString())
                        textSize = 10.dp.toPx()
                        textAlign = Paint.Align.CENTER
                    }
                )
            }
        }
    }
}

@Composable
private fun RecentActivitiesCard(recentEvents: List<InfoEvent>) {
    ElevatedCard(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = stringResource(R.string.recent_activities),
                style = MaterialTheme.typography.titleLarge
            )
            
            Spacer(modifier = Modifier.height(16.dp))
            
            if (recentEvents.isEmpty()) {
                // 无数据显示
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .height(100.dp),
                    contentAlignment = Alignment.Center
                ) {
                    Text(
                        text = stringResource(R.string.no_recent_activities),
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
            } else {
                // 活动列表
                Column(
                    verticalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    recentEvents.forEach { event ->
                        ActivityItem(event = event)
                    }
                }
            }
        }
    }
}

@Composable
private fun ActivityItem(event: InfoEvent) {
    Row(
        modifier = Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically
    ) {
        // 状态图标（阻止/允许）
        Icon(
            imageVector = if (event.blocked) Icons.Filled.Block else Icons.Filled.CheckCircle,
            contentDescription = null,
            tint = if (event.blocked) MaterialTheme.colorScheme.error else MaterialTheme.colorScheme.primary
        )
        
        Spacer(modifier = Modifier.width(8.dp))
        
        Column(
            modifier = Modifier.weight(1f)
        ) {
            // 时间
            Text(
                text = formatDateTime(event.timestamp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.onSurfaceVariant
            )
            
            // 持续时间
            Text(
                text = if (event.blocked) 
                    stringResource(R.string.blocked_event) 
                else 
                    stringResource(R.string.lasted_for, formatDuration(event.durationMs)),
                style = MaterialTheme.typography.bodyMedium
            )
        }
    }
}

/**
 * 格式化持续时间为可读字符串
 */
private fun formatDuration(durationMs: Long): String {
    if (durationMs < 1000) {
        return "${durationMs}ms"
    }
    
    val seconds = durationMs / 1000
    if (seconds < 60) {
        return "${seconds}s"
    }
    
    val minutes = seconds / 60
    val remainingSeconds = seconds % 60
    if (minutes < 60) {
        return "${minutes}m ${remainingSeconds}s"
    }
    
    val hours = minutes / 60
    val remainingMinutes = minutes % 60
    return "${hours}h ${remainingMinutes}m"
}

/**
 * 格式化时间戳为可读日期时间
 */
private fun formatDateTime(timestamp: Long): String {
    val date = Date(timestamp)
    return SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault()).format(date)
}
```

## 数据库相关DAO（Data Access Object）

**InfoEventDao.kt**
```kotlin
package com.js.nowakelock.data.db.dao

import androidx.room.Dao
import androidx.room.Delete
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import com.js.nowakelock.data.db.entity.InfoEvent

@Dao
interface InfoEventDao {
    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insert(infoEvent: InfoEvent): Long

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertAll(infoEvents: List<InfoEvent>)

    @Delete
    suspend fun delete(infoEvent: InfoEvent)

    @Query("SELECT * FROM info_events WHERE name = :name AND user_id = :userId ORDER BY timestamp DESC")
    suspend fun getEventsByName(name: String, userId: Int): List<InfoEvent>

    @Query("SELECT * FROM info_events WHERE name = :name AND package_name = :packageName AND user_id = :userId ORDER BY timestamp DESC")
    suspend fun getEventsByNameAndPackage(name: String, packageName: String, userId: Int): List<InfoEvent>

    @Query("SELECT * FROM info_events WHERE name = :name AND user_id = :userId AND timestamp >= :startTime ORDER BY timestamp DESC")
    suspend fun getEventsSinceTime(name: String, userId: Int, startTime: Long): List<InfoEvent>

    @Query("SELECT * FROM info_events WHERE name = :name AND package_name = :packageName AND user_id = :userId AND timestamp >= :startTime ORDER BY timestamp DESC")
    suspend fun getEventsSinceTimeByPackage(name: String, packageName: String, userId: Int, startTime: Long): List<InfoEvent>

    @Query("SELECT * FROM info_events WHERE name = :name AND user_id = :userId AND timestamp BETWEEN :startTime AND :endTime ORDER BY timestamp DESC")
    suspend fun getEventsInTimeRange(name: String, userId: Int, startTime: Long, endTime: Long): List<InfoEvent>

    @Query("SELECT * FROM info_events WHERE name = :name AND package_name = :packageName AND user_id = :userId AND timestamp BETWEEN :startTime AND :endTime ORDER BY timestamp DESC")
    suspend fun getEventsInTimeRangeByPackage(name: String, packageName: String, userId: Int, startTime: Long, endTime: Long): List<InfoEvent>

    @Query("DELETE FROM info_events WHERE timestamp < :beforeTime")
    suspend fun deleteEventsBefore(beforeTime: Long)
}
```

**StDao.kt**
```kotlin
package com.js.nowakelock.data.db.dao

import androidx.room.Dao
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import com.js.nowakelock.data.db.entity.St

@Dao
interface StDao {
    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insert(setting: St)

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertAll(settings: List<St>)

    @Query("SELECT * FROM settings WHERE name = :name AND package_name = :packageName AND user_id = :userId LIMIT 1")
    suspend fun getSetting(name: String, packageName: String, userId: Int): St?

    @Query("SELECT * FROM settings WHERE user_id = :userId")
    suspend fun getSettingsByUserId(userId: Int): List<St>

    @Query("SELECT * FROM settings WHERE package_name = :packageName AND user_id = :userId")
    suspend fun getSettingsByPackage(packageName: String, userId: Int): List<St>
}
```

## 导航配置

**NavRoutes.kt（更新）**
```kotlin:navigation/NavRoutes.kt
package com.js.nowakelock.navigation

/**
 * 应用导航路由
 */
object NavRoutes {
    const val APPS = "apps"
    const val DAS = "das"
    const val SETTINGS = "settings"
    const val APP_DETAIL = "app_detail/{packageName}"
    const val DA_DETAIL = "da_detail/{daId}?packageName={packageName}" // 新增路由

    // 用于生成路由路径的函数
    fun appDetail(packageName: String) = "app_detail/$packageName"
    
    // 新增：用于生成DA详情路由路径的函数
    fun daDetail(daId: String, packageName: String?) = 
        if (packageName.isNullOrEmpty()) "da_detail/$daId" 
        else "da_detail/$daId?packageName=$packageName"
}
```

**NavGraph.kt（更新部分）**
```kotlin:navigation/NavGraph.kt
// NavGraph.kt中需要添加的部分

navController.composable(
    route = NavRoutes.DA_DETAIL,
    arguments = listOf(
        navArgument("daId") { type = NavType.StringType },
        navArgument("packageName") {
            type = NavType.StringType
            nullable = true
            defaultValue = null
        }
    )
) { backStackEntry ->
    val daId = backStackEntry.arguments?.getString("daId") ?: ""
    val packageName = backStackEntry.arguments?.getString("packageName")
    
    DADetailScreen(
        daId = daId,
        packageName = packageName
    )
}
```

## 待解决问题

在实现过程中，我们发现了一些需要关注的问题：

1. **`getEventsInTimeRange`和`getRecentEvents`的区别和用途**：
   - `getRecentEvents`：用于获取从指定时间点到当前的所有事件，通常用于显示最近的活动。
   - `getEventsInTimeRange`：用于获取特定时间范围内的事件，更灵活，适合用于生成时间轴和统计数据。

2. **数据库访问和同步机制**：
   - `DARepository`中的`syncDB`和`syncInfoEvents`方法需要根据实际的同步机制进行实现。

## 总结

本文档总结了NoWakeLock项目中Wakelock详情页面的设计和实现过程。我们遵循了Material Design 3设计规范，基于MVVM架构模式，为详情页提供了完整的实现方案。

主要成果包括：

1. **数据模型和仓库设计**：
   - 定义了`DAInfoEntry`数据模型
   - 实现了`DAInfoRepository`和`DARepository`接口及其实现类
   - 设计了数据库访问对象（DAO）

2. **ViewModel设计**：
   - 设计了`DADetailViewModel`，管理UI状态和业务逻辑
   - 实现了数据加载、统计计算和设置更新等功能

3. **UI组件实现**：
   - 设计了`DADetailScreen`及其子组件
   - 实现了数据可视化和用户交互界面
   - 提供了加载状态和错误处理机制

4. **导航配置**：
   - 更新了导航路由和图表，支持带参数的导航

该实现方案确保了Wakelock详情页与应用程序其他部分的一致性，提供了良好的用户体验，并充分利用了Jetpack Compose和Material Design 3的特性。
