/*
 * Copyright (c) 2016-2019 Juan García Basilio
 *
 * This file is part of WaveUp.
 *
 * WaveUp is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * WaveUp is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WaveUp.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jarsilio.android.waveup.service

import android.annotation.TargetApi
import android.app.admin.DevicePolicyManager
import android.content.Context
import android.os.Build
import android.os.PowerManager
import android.os.VibrationEffect
import android.os.Vibrator
import android.view.accessibility.AccessibilityEvent
import android.view.accessibility.AccessibilityManager
import com.jarsilio.android.common.extensions.isOreoOrNewer
import com.jarsilio.android.common.extensions.isPieOrNewer
import com.jarsilio.android.common.utils.SingletonHolder
import com.jarsilio.android.waveup.R
import com.jarsilio.android.waveup.extensions.settings
import com.jarsilio.android.waveup.extensions.state
import eu.chainfire.libsuperuser.Shell
import timber.log.Timber

class ScreenHandler private constructor(context: Context) {
    private val applicationContext: Context = context.applicationContext

    private val powerManager: PowerManager = applicationContext.getSystemService(Context.POWER_SERVICE) as PowerManager
    private val wakeLock: PowerManager.WakeLock =
        powerManager.newWakeLock(
            PowerManager.ACQUIRE_CAUSES_WAKEUP or PowerManager.SCREEN_BRIGHT_WAKE_LOCK,
            "waveup:wakelock",
        )
    private val policyManager: DevicePolicyManager =
        applicationContext.getSystemService(
            Context.DEVICE_POLICY_SERVICE,
        ) as DevicePolicyManager

    private var turnOffScreenThread: Thread? = null
    private var turningOffScreen: Boolean = false

    var lastTimeScreenOnOrOff: Long = 0
        private set

    private fun turnOffScreenThread(delay: Long): Thread {
        return object : Thread() {
            override fun run() {
                if (shouldTurnOffScreen()) {
                    Timber.d("Creating a thread to turn off display if still covered in ${delay / 1000.0} seconds")
                    try {
                        vibrateIfSet()
                        sleep(delay)
                        if (shouldTurnOffScreen()) {
                            vibrateIfSet()
                            doTurnOffScreen()
                        }
                    } catch (e: InterruptedException) {
                        Timber.d("Interrupted thread: Turning off screen cancelled.")
                    }
                }
            }
        }
    }

    private fun shouldTurnOffScreen(): Boolean {
        if (!applicationContext.state.isScreenOn) {
            Timber.v("Should turn off screen? Screen is already off.")
        }
        if (applicationContext.state.isExcludedAppInForeground()) {
            Timber.v("Should turn off screen? Excluded app is running in foreground. Not turning off screen.")
        }
        return applicationContext.state.isScreenOn && !applicationContext.state.isExcludedAppInForeground()
    }

    private fun doTurnOffScreen() {
        turningOffScreen = true // Issue #68. Avoid interrupting the thread if screen is already being turned off.
        lastTimeScreenOnOrOff = System.currentTimeMillis()
        Timber.i("Switched from 'far' to 'near'.")
        if (isPieOrNewer) {
            val manager = applicationContext.getSystemService(Context.ACCESSIBILITY_SERVICE) as AccessibilityManager
            if (manager.isEnabled) {
                val event = AccessibilityEvent.obtain(AccessibilityEvent.TYPE_NOTIFICATION_STATE_CHANGED)
                event.packageName = applicationContext.packageName
                event.className = this.javaClass.name
                event.isEnabled = true
                event.text.add(applicationContext.getString(R.string.accessibility_service_text))
                Timber.v("Accessibility enabled. Firing AccessibilityEvent: $event")
                manager.sendAccessibilityEvent(event)
            } else {
                Timber.v("Accessibility disabled. Can't fire AccessibilityEvent")
            }
        } else if (applicationContext.settings.isLockScreenWithPowerButton) {
            Timber.i("Turning screen off simulating power button press.")
            Shell.SU.run("input keyevent 26")
        } else {
            Timber.i("Turning screen off.")
            try {
                policyManager.lockNow()
            } catch (e: IllegalStateException) {
                Timber.e(
                    "Failed to run lockNow() to turn off the screen. " +
                        "Probably due to an ongoing call. Exception: $e",
                )
            } catch (e: SecurityException) {
                Timber.e(
                    "Failed to run lockNow() to turn off the screen. " +
                        "Probably due to missing device admin rights, which I don't really understand... Exception: $e",
                )
            }
        }
        turningOffScreen = false
    }

    fun turnOffScreen(delay: Long = applicationContext.settings.sensorCoverTimeBeforeLockingScreen) {
        turnOffScreenThread = turnOffScreenThread(delay)
        turnOffScreenThread?.start()
    }

    fun cancelTurnOff() {
        if (turnOffScreenThread?.state != Thread.State.TERMINATED && !turningOffScreen) {
            Timber.d("Cancelling turning off of display")
            turnOffScreenThread?.interrupt()
        }
    }

    fun turnOnScreen() {
        if (!applicationContext.state.isScreenOn) {
            lastTimeScreenOnOrOff = System.currentTimeMillis()
            Timber.i("Switched from 'near' to 'far'. Turning screen on")
            if (wakeLock.isHeld) {
                wakeLock.release()
            }
            wakeLock.acquire(TIME_SCREEN_ON)
        }
    }

    private fun vibrateIfSet() {
        if (applicationContext.settings.vibrateWhileLockingTime > 0) {
            val vibrator = applicationContext.getSystemService(Context.VIBRATOR_SERVICE) as Vibrator?

            if (vibrator == null) {
                Timber.e("Failed to access vibrator on device.")
                return
            }

            @TargetApi(Build.VERSION_CODES.O)
            if (isOreoOrNewer) {
                vibrator.vibrate(
                    VibrationEffect.createOneShot(applicationContext.settings.vibrateWhileLockingTime, VibrationEffect.DEFAULT_AMPLITUDE),
                )
            } else {
                vibrator.vibrate(applicationContext.settings.vibrateWhileLockingTime)
            }
        }
    }

    companion object : SingletonHolder<ScreenHandler, Context>(::ScreenHandler) {
        private const val TIME_SCREEN_ON: Long = 5000
    }
}
