package com.jarsilio.android.pingoff.services

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import com.jarsilio.android.pingoff.extensions.foregroundApp
import com.jarsilio.android.pingoff.extensions.isAccessibilityServiceEnabled
import timber.log.Timber
import java.util.Timer
import java.util.TimerTask

object ScreenReceiver : BroadcastReceiver() {
    private var timer: Timer = Timer()

    override fun onReceive(
        context: Context,
        intent: Intent,
    ) {
        if (intent.action == Intent.ACTION_SCREEN_OFF) {
            Timber.d("Screen off. Disabling scheduled task to check running app (TimerTask)")
            unscheduleTimer()
            if (DoNotDisturbHandler.didWeEnableDnd()) {
                Timber.d("Disabling DND because we set it and screen went off")
                DoNotDisturbHandler.setDoNotDisturb(false) // disable DND while if display is off
            }
        } else if (intent.action == Intent.ACTION_SCREEN_ON) {
            Timber.d("Screen on. Re-enabling scheduled task to check running app (TimerTask)")
            scheduleTimer(context)
        }
    }

    fun unscheduleTimer() {
        timer.cancel()
    }

    fun scheduleTimer(context: Context) {
        if (context.isAccessibilityServiceEnabled) {
            Timber.d("Not scheduling recurring timer to poll currently running app because the AccessibilityService is enabled.")
            return
        }

        var lastForegroundApp = ""

        timer.cancel()
        timer = Timer()
        timer.schedule(
            object : TimerTask() {
                override fun run() {
                    val foregroundApp = context.foregroundApp

                    // TODO I think this is exactly the same logic as in AppLaunchDetectionService. Move somewhere else?
                    if (foregroundApp != lastForegroundApp) { // Only show the log once, and not every second
                        if (DoNotDisturbHandler.isInDoNotDisturbList(foregroundApp)) {
                            Timber.d("$foregroundApp is in the list for DND. Setting DND to true (if not already set)")
                            DoNotDisturbHandler.setDoNotDisturb(true)
                        } else {
                            Timber.d(
                                "$foregroundApp is *not* in the list for DND. Setting DND to false (if not already set)",
                            )
                            if (DoNotDisturbHandler.didWeEnableDnd()) {
                                Timber.d("App enabled DND earlier — now disabling it.")
                                DoNotDisturbHandler.setDoNotDisturb(false)
                            } else {
                                Timber.d("DND was set by user or schedule — respecting it.")
                            }
                        }
                    }
                    lastForegroundApp = foregroundApp
                }
            },
            500,
            1000,
        )
    }
}
