package com.jarsilio.android.pingoff.services

import android.annotation.SuppressLint
import android.app.NotificationManager
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import com.jarsilio.android.pingoff.applist.AppDatabase
import com.jarsilio.android.pingoff.requireApplicationContext
import timber.log.Timber

@SuppressLint("StaticFieldLeak") // TODO: Sure about this?
object DoNotDisturbHandler {
    private val context = requireApplicationContext()

    private val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

    private var wasSetByApp = false
    private var lastDndSetTimestamp: Long = 0L

    fun setDoNotDisturb(enabled: Boolean) {
        if (!notificationManager.isNotificationPolicyAccessGranted) {
            Timber.w("DND access not granted by user.")
            // TODO: prompt to go to settings?
            return
        }

        val currentInterruptionFilter = notificationManager.currentInterruptionFilter

        val desiredFilter =
            if (enabled) {
                // NotificationManager.INTERRUPTION_FILTER_NONE
                NotificationManager.INTERRUPTION_FILTER_PRIORITY
            } else {
                NotificationManager.INTERRUPTION_FILTER_ALL
            }

        if (currentInterruptionFilter != desiredFilter) {
            lastDndSetTimestamp = System.currentTimeMillis()
            wasSetByApp = enabled
            notificationManager.setInterruptionFilter(desiredFilter)
            Timber.d("Do Not Disturb mode set to: $enabled")
        } else {
            Timber.d("No change to DND filter.")
        }
    }

    fun isInDoNotDisturbList(packageName: String): Boolean {
        for (app in AppDatabase.getInstance(context).appsDao().autorotateApps) {
            if (app.packageName == packageName) {
                Timber.d("${app.name} (${app.packageName}) is running in foreground")
                return true
            }
        }
        return false
    }

    fun isDoNotDisturbedEnabled(): Boolean {
        return notificationManager.currentInterruptionFilter == NotificationManager.INTERRUPTION_FILTER_NONE
    }

    fun didWeEnableDnd(): Boolean {
        return wasSetByApp
    }

    fun registerDndChangeReceiver() {
        val filter = IntentFilter(NotificationManager.ACTION_INTERRUPTION_FILTER_CHANGED)
        context.registerReceiver(dndReceiver, filter)
    }

    fun unregisterDndChangeReceiver() {
        context.unregisterReceiver(dndReceiver)
    }

    private val dndReceiver =
        object : BroadcastReceiver() {
            override fun onReceive(
                context: Context,
                intent: Intent?,
            ) {
                if (intent?.action == NotificationManager.ACTION_INTERRUPTION_FILTER_CHANGED) {
                    val now = System.currentTimeMillis()
                    val timeSinceChange = now - lastDndSetTimestamp
                    val currentFilter = notificationManager.currentInterruptionFilter

                    Timber.d("Interruption filter changed: $currentFilter (after ${timeSinceChange}ms)")

                    if (timeSinceChange > 2000) {
                        Timber.d("Assuming DND change was NOT from us — clearing flag")
                        wasSetByApp = false
                    }
                }
            }
        }
}
