package com.jarsilio.android.pingoff.services

import android.accessibilityservice.AccessibilityService
import android.content.ComponentName
import android.content.Intent
import android.content.pm.PackageManager
import android.view.accessibility.AccessibilityEvent
import com.jarsilio.android.pingoff.extensions.lastForegroundApp
import com.jarsilio.android.pingoff.prefs.Prefs
import timber.log.Timber

class AppLaunchDetectionService : AccessibilityService() {
    // From https://stackoverflow.com/questions/3873659/android-how-can-i-get-the-current-foreground-activity-from-a-service#4753333 So cool, thank you!

    override fun onAccessibilityEvent(event: AccessibilityEvent) {
        if (!Prefs.isEnabled) {
            return
        }

        if (event.eventType == AccessibilityEvent.TYPE_WINDOW_STATE_CHANGED) {
            if (event.packageName != null && event.className != null) {
                val packageName = event.packageName.toString()
                val className = event.className.toString()

                Thread {
                    if (isActivity(ComponentName(packageName, className))) {
                        lastForegroundApp = packageName
                        if (DoNotDisturbHandler.isInDoNotDisturbList(packageName)) {
                            Timber.d("$packageName in foreground: enabling DND (if not already the case)")
                            DoNotDisturbHandler.setDoNotDisturb(true)
                        } else {
                            Timber.d("$packageName in foreground: disabling DND (if not already the case)")
                            if (DoNotDisturbHandler.didWeEnableDnd()) {
                                Timber.d("We enabled DND earlier — disabling it.")
                                DoNotDisturbHandler.setDoNotDisturb(false)
                            } else {
                                Timber.d("DND was set by user or schedule — respecting it.")
                            }
                        }
                    }
                }.start()
            }
        }
    }

    val currentLauncherPackage: String
        get() {
            val intent =
                Intent(Intent.ACTION_MAIN).apply {
                    addCategory(Intent.CATEGORY_HOME)
                }
            val resolveInfo = packageManager.resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY)
            return resolveInfo!!.activityInfo.packageName
        }

    private fun isActivity(componentName: ComponentName): Boolean {
        val activityInfo =
            try {
                packageManager.getActivityInfo(componentName, 0)
            } catch (e: PackageManager.NameNotFoundException) {
                null
            }
        return activityInfo != null
    }

    override fun onInterrupt() {}
}
