package com.jarsilio.android.drowser.services

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.jarsilio.android.drowser.models.AppsManager
import com.jarsilio.android.drowser.prefs.Prefs
import com.jarsilio.android.drowser.utils.Utils
import timber.log.Timber
import java.util.concurrent.TimeUnit

class ScreenReceiver : BroadcastReceiver() {
    var drowserThread: Thread? = null

    override fun onReceive(
        context: Context,
        intent: Intent,
    ) {
        if (intent.action == Intent.ACTION_SCREEN_OFF) {
            val prefs = Prefs.getInstance(context)

            Timber.d("Screen off")

            if (prefs.disableUntil < System.currentTimeMillis()) {
                Timber.d("Scheduling WorkManager to drowse apps in ${prefs.drowseDelay / 1000} seconds...")
                val drowseWorkRequest =
                    OneTimeWorkRequestBuilder<DrowserScreenOffDelayWorker>()
                        .setInitialDelay(prefs.drowseDelay, TimeUnit.MILLISECONDS)
                        .addTag("DrowserScreenOffDelayWorker")
                        .build()
                WorkManager.getInstance(context).enqueue(drowseWorkRequest)
            } else {
                Timber.d("Drowser temporarily disabled (until ${Utils.getReadableDate(prefs.disableUntil)}), not drowsing apps.")
            }
        } else if (intent.action == Intent.ACTION_SCREEN_ON) {
            Timber.d("Screen on")
            Timber.d("Canceling any pending drowse work if screen is turned on (WorkManager tagged 'DrowserScreenOffDelayWorker')")
            WorkManager.getInstance(context).cancelAllWorkByTag("DrowserScreenOffDelayWorker")
        }
    }
}

class DrowserScreenOffDelayWorker(context: Context, params: WorkerParameters) : Worker(context, params) {
    override fun doWork(): Result {
        val prefs = Prefs.getInstance(applicationContext)
        if (prefs.disableUntil < System.currentTimeMillis()) {
            AppsManager(applicationContext).forceStopApps()
            return Result.success()
        } else {
            return Result.failure()
        }
    }
}
