package com.jarsilio.android.drowser.services

import android.content.Context
import android.os.Build
import android.os.PowerManager
import androidx.work.WorkManager
import androidx.work.WorkRequest
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.jarsilio.android.drowser.models.AppsManager
import timber.log.Timber
import java.util.concurrent.TimeUnit
import androidx.work.OneTimeWorkRequestBuilder as OneTimeWorkRequestBuilder

class AppDrowserWorker(context: Context, workerParams: WorkerParameters) : Worker(context, workerParams) {
    override fun doWork(): Result {
        if (isScreenOn()) {
            Timber.d(
                "Not drowsing apps because screen is on. Letting the natural drowsing " +
                    "course take place (i.e. drowse when the screen goes off)",
            )
            return Result.success()
        } else {
            AppsManager(applicationContext).forceStopApps()
        }
        return Result.success()
    }

    private fun isScreenOn(): Boolean {
        val powerManager = applicationContext.getSystemService(Context.POWER_SERVICE) as PowerManager?
        if (powerManager == null) {
            // Assume screen is on to not kill any apps the user might be using
            // (this could be impolite, but I prefer to make sure the apps are drowsed)
            return true
        }

        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT_WATCH) {
            powerManager.isInteractive
        } else {
            powerManager.isScreenOn
        }
    }
}

object Scheduler {
    fun scheduleAlarm(
        context: Context,
        time: Long,
    ) {
        Timber.d("Canceling previously set alarm")
        cancelAlarm(context)

        Timber.d("Scheduling alarm to drowse apps after pause")
        val delay = time - System.currentTimeMillis()
        val drowsingRequest: WorkRequest =
            OneTimeWorkRequestBuilder<AppDrowserWorker>()
                .setInitialDelay(delay, TimeUnit.MILLISECONDS)
                .build()
        WorkManager.getInstance(context).enqueue(drowsingRequest)
    }

    private fun cancelAlarm(context: Context) {
        WorkManager.getInstance(context).cancelAllWork()
    }
}
