package com.inspiredandroid.kai

import android.content.Context
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.ui.draganddrop.DragAndDropEvent
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.security.crypto.EncryptedSharedPreferences
import androidx.security.crypto.MasterKey
import com.inspiredandroid.kai.data.AppSettings
import com.inspiredandroid.kai.network.tools.ParameterSchema
import com.inspiredandroid.kai.network.tools.Tool
import com.inspiredandroid.kai.network.tools.ToolInfo
import com.inspiredandroid.kai.network.tools.ToolSchema
import com.inspiredandroid.kai.tools.CalendarPermissionController
import com.inspiredandroid.kai.tools.CalendarRepository
import com.inspiredandroid.kai.tools.CalendarResult
import com.inspiredandroid.kai.tools.CommonTools
import com.inspiredandroid.kai.tools.NotificationHelper
import com.inspiredandroid.kai.tools.NotificationPermissionController
import com.inspiredandroid.kai.tools.NotificationResult
import com.inspiredandroid.kai.tools.SmsPermissionController
import com.inspiredandroid.kai.tools.SmsRepository
import com.inspiredandroid.kai.tools.SmsResult
import com.inspiredandroid.kai.tools.SmsSendResult
import com.russhwolf.settings.Settings
import com.russhwolf.settings.SharedPreferencesSettings
import io.github.vinceglb.filekit.PlatformFile
import io.ktor.client.HttpClient
import io.ktor.client.HttpClientConfig
import io.ktor.client.engine.android.Android
import kotlinx.coroutines.Dispatchers
import org.koin.java.KoinJavaComponent.inject
import kotlin.coroutines.CoroutineContext

actual fun httpClient(config: HttpClientConfig<*>.() -> Unit): HttpClient = HttpClient(Android) {
    config(this)
}

// Singleton SmsRepository to maintain conversation ID mapping across tool calls
private val smsRepositorySingleton: SmsRepository by lazy {
    val context: Context by inject(Context::class.java)
    val smsPermissionController: SmsPermissionController by inject(SmsPermissionController::class.java)
    SmsRepository(context, smsPermissionController)
}

actual fun getBackgroundDispatcher(): CoroutineContext = Dispatchers.IO

actual fun onDragAndDropEventDropped(event: DragAndDropEvent): PlatformFile? = null

actual val BackIcon: ImageVector = Icons.AutoMirrored.Filled.ArrowBack

actual val isMobilePlatform: Boolean = true

actual fun getAppFilesDirectory(): String {
    val context: Context by inject(Context::class.java)
    return context.filesDir.absolutePath
}

actual fun createSecureSettings(): Settings {
    val context: Context by inject(Context::class.java)
    val masterKey = MasterKey.Builder(context)
        .setKeyScheme(MasterKey.KeyScheme.AES256_GCM)
        .build()
    val encryptedPrefs = EncryptedSharedPreferences.create(
        context,
        "kai_secure_prefs",
        masterKey,
        EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
        EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM,
    )
    return SharedPreferencesSettings(encryptedPrefs)
}

actual fun createLegacySettings(): Settings? {
    val context: Context by inject(Context::class.java)
    val prefs = context.getSharedPreferences("com.inspiredandroid.kai_preferences", Context.MODE_PRIVATE)
    return SharedPreferencesSettings(prefs)
}

// Tool definitions for Android platform
actual fun getPlatformToolDefinitions(): List<ToolInfo> = listOf(
    CommonTools.localTimeToolInfo,
    CommonTools.ipLocationToolInfo,
    ToolInfo(
        id = "send_notification",
        name = "Send Notification",
        description = "Send a push notification to the device",
    ),
    ToolInfo(
        id = "create_calendar_event",
        name = "Create Calendar Event",
        description = "Create a calendar event on the user's device",
    ),
    ToolInfo(
        id = "get_recent_sms",
        name = "Check Recent SMS",
        description = "Get recent SMS conversations with the last few messages from each",
    ),
    ToolInfo(
        id = "send_sms",
        name = "Send SMS",
        description = "Reply to an existing SMS conversation",
    ),
)

actual fun getAvailableTools(): List<Tool> {
    val context: Context by inject(Context::class.java)
    val appSettings: AppSettings by inject(AppSettings::class.java)
    val calendarPermissionController: CalendarPermissionController by inject(CalendarPermissionController::class.java)
    val calendarRepository = CalendarRepository(context, calendarPermissionController)

    return buildList {
        if (appSettings.isToolEnabled(CommonTools.localTimeTool.schema.name)) {
            add(CommonTools.localTimeTool)
        }

        if (appSettings.isToolEnabled(CommonTools.ipLocationTool.schema.name)) {
            add(CommonTools.ipLocationTool)
        }

        if (appSettings.isToolEnabled("send_notification")) {
            val notificationPermissionController: NotificationPermissionController by inject(NotificationPermissionController::class.java)
            val notificationHelper = NotificationHelper(context, notificationPermissionController)

            add(
                object : Tool {
                    override val schema = ToolSchema(
                        "send_notification",
                        "Send a push notification to the device",
                        mapOf(
                            "title" to ParameterSchema("string", "Notification title", false),
                            "message" to ParameterSchema("string", "Notification content/body", true),
                        ),
                    )

                    override suspend fun execute(args: Map<String, Any>): Any {
                        val title = args["title"] as? String ?: "Kai"
                        val message = args["message"] as? String
                            ?: return mapOf("success" to false, "error" to "Message is required")

                        return when (val result = notificationHelper.sendNotification(title, message)) {
                            is NotificationResult.Success -> mapOf(
                                "success" to true,
                                "notification_id" to result.notificationId,
                                "message" to "Notification sent successfully",
                            )

                            is NotificationResult.Error -> mapOf(
                                "success" to false,
                                "error" to result.message,
                            )
                        }
                    }
                },
            )
        }

        if (appSettings.isToolEnabled("create_calendar_event")) {
            add(
                object : Tool {
                    override val schema = ToolSchema(
                        "create_calendar_event",
                        "Create a calendar event on the user's device",
                        mapOf(
                            "title" to ParameterSchema("string", "Event title", true),
                            "start_time" to ParameterSchema("string", "Start time in ISO 8601 format (e.g., 2024-03-15T14:30:00)", true),
                            "end_time" to ParameterSchema("string", "End time in ISO 8601 format. Defaults to 1 hour after start", false),
                            "description" to ParameterSchema("string", "Event notes or description", false),
                            "location" to ParameterSchema("string", "Event location", false),
                            "all_day" to ParameterSchema("boolean", "Whether this is an all-day event", false),
                            "reminder_minutes" to ParameterSchema("integer", "Minutes before event to send reminder (default: 15)", false),
                        ),
                    )

                    override suspend fun execute(args: Map<String, Any>): Any {
                        val title = args["title"] as? String
                            ?: return mapOf("success" to false, "error" to "Title is required")
                        val startTime = args["start_time"] as? String
                            ?: return mapOf("success" to false, "error" to "Start time is required")
                        val endTime = args["end_time"] as? String
                        val description = args["description"] as? String
                        val location = args["location"] as? String
                        val allDay = (args["all_day"] as? Boolean) ?: false
                        val reminderMinutes = (args["reminder_minutes"] as? Number)?.toInt() ?: 15

                        return when (
                            val result = calendarRepository.createEvent(
                                title = title,
                                startTimeIso = startTime,
                                endTimeIso = endTime,
                                description = description,
                                location = location,
                                allDay = allDay,
                                reminderMinutes = reminderMinutes,
                            )
                        ) {
                            is CalendarResult.Success -> mapOf(
                                "success" to true,
                                "event_id" to result.eventId,
                                "title" to result.title,
                                "scheduled_for" to result.startTime,
                                "message" to "Event '${result.title}' created successfully for ${result.startTime}",
                            )

                            is CalendarResult.Error -> mapOf(
                                "success" to false,
                                "error" to result.message,
                            )
                        }
                    }
                },
            )
        }

        // SMS tools share a singleton SmsRepository to maintain conversation ID mapping across tool calls
        if (appSettings.isToolEnabled("get_recent_sms")) {
            add(
                object : Tool {
                    override val schema = ToolSchema(
                        "get_recent_sms",
                        "Get recent SMS conversations with the last few messages from each. Returns conversation_id for each conversation which can be used with send_sms to reply.",
                        emptyMap(),
                    )

                    override suspend fun execute(args: Map<String, Any>): Any = when (val result = smsRepositorySingleton.getRecentConversations()) {
                        is SmsResult.Success -> mapOf(
                            "success" to true,
                            "conversations" to result.conversations.map { conversation ->
                                mapOf(
                                    "conversation_id" to conversation.conversationId,
                                    "contact_name" to (conversation.contactName ?: "Unknown"),
                                    "messages" to conversation.messages.map { message ->
                                        mapOf(
                                            "body" to message.body,
                                            "timestamp" to message.timestamp,
                                            "from_me" to message.isFromMe,
                                        )
                                    },
                                )
                            },
                        )

                        is SmsResult.Error -> mapOf(
                            "success" to false,
                            "error" to result.message,
                        )
                    }
                },
            )
        }

        if (appSettings.isToolEnabled("send_sms")) {
            add(
                object : Tool {
                    override val schema = ToolSchema(
                        "send_sms",
                        "Reply to an existing SMS conversation. Can only send to conversations retrieved from get_recent_sms.",
                        mapOf(
                            "conversation_id" to ParameterSchema("string", "Conversation ID from recent SMS list (required - can only reply to existing conversations)", true),
                            "message" to ParameterSchema("string", "Message text to send", true),
                        ),
                    )

                    override suspend fun execute(args: Map<String, Any>): Any {
                        val conversationId = args["conversation_id"] as? String
                            ?: return mapOf("success" to false, "error" to "Conversation ID is required")
                        val message = args["message"] as? String
                            ?: return mapOf("success" to false, "error" to "Message is required")

                        return when (val result = smsRepositorySingleton.sendSms(conversationId, message)) {
                            is SmsSendResult.Success -> mapOf(
                                "success" to true,
                                "message" to "SMS sent successfully to conversation ${result.conversationId}",
                            )

                            is SmsSendResult.Error -> mapOf(
                                "success" to false,
                                "error" to result.message,
                            )
                        }
                    }
                },
            )
        }
    }
}
