package com.hardcodecoder.pulsemusic.loaders;

import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hardcodecoder.pulsemusic.model.MusicModel;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

public abstract class BaseMasterListLoader implements Callable<List<MusicModel>> {

    private final WeakReference<Context> mContextReference;
    private final String mSortOrder;
    private final String mSelection;
    private final String[] mSelectionArgs;

    protected BaseMasterListLoader(@NonNull Context context,
                                   @Nullable SortOrder sortOrder,
                                   @Nullable String selection,
                                   @Nullable String[] selectionArgs) {
        mContextReference = new WeakReference<>(context);
        mSortOrder = MediaStoreHelper.getSortOrderFor(sortOrder);
        mSelection = selection;
        mSelectionArgs = selectionArgs;
    }

    @Override
    @Nullable
    public List<MusicModel> call() {
        Context context = mContextReference.get();
        if (null == context) return null;

        final Uri uri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
        final int skdInt = Build.VERSION.SDK_INT;

        onReady(context);
        final Cursor cursor = context.getContentResolver().query(uri,
                MediaStoreHelper.getMediaAudioColumns(skdInt),
                getSelection(context, mSelection),
                getSelectionArgs(context, mSelectionArgs),
                mSortOrder);

        if (null == cursor || !cursor.moveToFirst()) {
            if (null != cursor) cursor.close();
            return null;
        }

        List<MusicModel> masterList = new ArrayList<>();
        final int idColIndex = cursor.getColumnIndexOrThrow(MediaStore.Audio.Media._ID);
        do {
            int id = cursor.getInt(idColIndex);
            if (excludeItem(id)) continue;
            masterList.add(MediaStoreHelper.getModelFromCursor(cursor, id, skdInt));
        } while (cursor.moveToNext());
        cursor.close();

        return masterList;
    }

    public void onReady(@NonNull Context context) {
    }

    public abstract boolean excludeItem(int id);

    @Nullable
    public abstract String getSelection(@NonNull Context context, @Nullable String baseSelection);

    @Nullable
    public abstract String[] getSelectionArgs(@NonNull Context context, @Nullable String[] baseSelectionArgs);
}
