package com.hardcodecoder.pulsemusic.glide;

import android.content.ContentResolver;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.Uri;
import android.provider.MediaStore;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bumptech.glide.load.Options;
import com.bumptech.glide.load.model.ModelLoader;
import com.bumptech.glide.signature.ObjectKey;
import com.hardcodecoder.pulsemusic.Preferences;
import com.hardcodecoder.pulsemusic.utils.AppSettings;

import java.io.InputStream;

public class MediaArtworkLoader implements ModelLoader<MediaArtwork, InputStream>,
        SharedPreferences.OnSharedPreferenceChangeListener {

    private Context mContext;
    private boolean mIgnoreMediaStoreArtwork;

    public MediaArtworkLoader(@NonNull Context context) {
        mContext = context;
        mContext.getSharedPreferences(Preferences.PREF_GENERAL, Context.MODE_PRIVATE)
                .registerOnSharedPreferenceChangeListener(this);
        mIgnoreMediaStoreArtwork = AppSettings.ignoreMediaStoreArtwork(mContext);
    }

    @Nullable
    @Override
    public LoadData<InputStream> buildLoadData(@NonNull MediaArtwork mediaArtwork, int width, int height, @NonNull Options options) {
        return new LoadData<>(
                new ObjectKey(getCacheKey(mediaArtwork)),
                new MediaArtFetcher(mContext, mediaArtwork, mIgnoreMediaStoreArtwork));
    }

    @NonNull
    private String getCacheKey(@NonNull MediaArtwork artwork) {
        if (mIgnoreMediaStoreArtwork)
            return artwork.getMediaUri() + artwork.getLastModified();
        else return artwork.getArtworkUri() + artwork.getLastModified();
    }

    @Override
    public boolean handles(@NonNull MediaArtwork mediaArtwork) {
        if (mIgnoreMediaStoreArtwork) {
            Uri mediaUri = Uri.parse(mediaArtwork.getMediaUri());
            return ContentResolver.SCHEME_CONTENT.equals(mediaUri.getScheme())
                    && MediaStore.AUTHORITY.equals(mediaUri.getAuthority())
                    && mediaUri.toString().contains("audio");
        }
        Uri artworkUri = Uri.parse(mediaArtwork.getArtworkUri());
        return ContentResolver.SCHEME_CONTENT.equals(artworkUri.getScheme())
                && MediaStore.AUTHORITY.equals(artworkUri.getAuthority())
                && artworkUri.toString().contains("albumart");
    }

    @Override
    public void onSharedPreferenceChanged(@NonNull SharedPreferences sharedPreferences, @NonNull String key) {
        if (key.equals(Preferences.KEY_IGNORE_MEDIA_STORE_ARTWORK))
            mIgnoreMediaStoreArtwork = AppSettings.ignoreMediaStoreArtwork(mContext);
    }

    public void cleanup() {
        mContext.getSharedPreferences(Preferences.PREF_GENERAL, Context.MODE_PRIVATE)
                .unregisterOnSharedPreferenceChangeListener(this);
        mContext = null;
    }
}