@file:OptIn(ExperimentalPermissionsApi::class)

package com.grappim.hateitorrateit.feature.details.ui

import android.Manifest
import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.os.Build
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.PagerState
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalResources
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.PreviewParameter
import androidx.compose.ui.tooling.preview.PreviewParameterProvider
import androidx.compose.ui.unit.dp
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import coil.compose.rememberAsyncImagePainter
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.PermissionState
import com.google.accompanist.permissions.isGranted
import com.google.accompanist.permissions.rememberPermissionState
import com.google.accompanist.permissions.shouldShowRationale
import com.grappim.hateitorrateit.data.repoapi.models.HateRateType
import com.grappim.hateitorrateit.strings.RString
import com.grappim.hateitorrateit.uikit.icons.PlatoIconType
import com.grappim.hateitorrateit.uikit.theme.HateItOrRateItTheme
import com.grappim.hateitorrateit.uikit.utils.PreviewDarkLight
import com.grappim.hateitorrateit.uikit.utils.color
import com.grappim.hateitorrateit.uikit.utils.icon
import com.grappim.hateitorrateit.uikit.widgets.PlatoAlertDialog
import com.grappim.hateitorrateit.uikit.widgets.PlatoCard
import com.grappim.hateitorrateit.uikit.widgets.PlatoIcon
import com.grappim.hateitorrateit.uikit.widgets.PlatoIconButton
import com.grappim.hateitorrateit.uikit.widgets.PlatoImage
import com.grappim.hateitorrateit.uikit.widgets.PlatoPagerIndicator
import com.grappim.hateitorrateit.uikit.widgets.PlatoPlaceholderImage
import com.grappim.hateitorrateit.uikit.widgets.PlatoProgressIndicator
import com.grappim.hateitorrateit.uikit.widgets.PlatoTopBar
import com.grappim.hateitorrateit.uikit.widgets.topbar.LocalTopBarConfig
import com.grappim.hateitorrateit.uikit.widgets.topbar.TopBarConfig
import com.grappim.hateitorrateit.uikit.widgets.topbar.TopBarState
import com.grappim.hateitorrateit.utils.ui.NativeText
import com.grappim.hateitorrateit.utils.ui.ObserveAsEvents
import com.grappim.hateitorrateit.utils.ui.findActivity
import timber.log.Timber

const val DETAILS_SCREEN_CONTENT_TAG = "details_screen_content_tag"
const val DETAILS_TOP_APP_BAR_TAG = "details_top_app_bar_tag"
const val DETAILS_DEMONSTRATION_CONTENT_TAG = "details_demonstration_content_tag"

private const val TOP_APP_BAR_WEIGHT = 1.2f

@Composable
fun DetailsRoute(
    goBack: () -> Unit,
    onEditClick: (id: Long) -> Unit,
    onImageClick: (productId: Long, index: Int) -> Unit,
    isFromEdit: Boolean,
    showSnackbar: (NativeText, actionLabel: String?) -> Unit,
    viewModel: DetailsViewModel = hiltViewModel()
) {
    val state by viewModel.viewState.collectAsStateWithLifecycle()
    val context = LocalContext.current
    val resources = LocalResources.current
    val topBarController = LocalTopBarConfig.current

    LaunchedEffect(Unit) {
        topBarController.update(TopBarConfig(state = TopBarState.Hidden))
    }

    ObserveAsEvents(viewModel.snackBarMessage) { snackbarMessage ->
        if (snackbarMessage !is NativeText.Empty) {
            showSnackbar(snackbarMessage, resources.getString(RString.close))
        }
    }

    ObserveAsEvents(viewModel.viewEvents) { event ->
        when (event) {
            is DetailsEvents.SaveImageSuccess -> {
                state.setSnackbarMessage(NativeText.Resource(R.string.image_saved_in_gallery))
            }

            is DetailsEvents.SaveImageFailure -> {
                state.setSnackbarMessage(NativeText.Resource(R.string.image_saved_in_gallery_error))
            }
        }
    }

    ObserveAsEvents(viewModel.intentAction) { intent ->
        val activity = context.findActivity()
        try {
            activity.startActivity(intent)
        } catch (e: ActivityNotFoundException) {
            Timber.e(e)
            state.setSnackbarMessage(NativeText.Resource(R.string.share_image_error))
        }
    }

    DisposableEffect(Unit) {
        state.trackScreenStart
        onDispose {}
    }

    LaunchedEffect(state.productDeleted) {
        if (state.productDeleted) {
            goBack()
        }
    }

    LaunchedEffect(isFromEdit) {
        if (isFromEdit) {
            state.updateProduct()
        }
    }

    PlatoAlertDialog(
        text = stringResource(id = R.string.are_you_sure_to_delete_product),
        showAlertDialog = state.showAlertDialog,
        dismissButtonText = stringResource(id = R.string.no),
        onDismissRequest = {
            state.onShowAlertDialog(false)
        },
        onConfirmButtonClick = {
            state.onDeleteProductConfirm()
        },
        onDismissButtonClick = {
            state.onShowAlertDialog(false)
        }
    )

    if (state.isLoading.not()) {
        DetailsScreenContent(
            state = state,
            goBack = goBack,
            onImageClick = onImageClick,
            onEditClick = onEditClick
        )
    } else {
        PlatoProgressIndicator(true)
    }
}

@Composable
private fun DetailsScreenContent(
    state: DetailsViewState,
    goBack: () -> Unit,
    onImageClick: (productId: Long, index: Int) -> Unit,
    onEditClick: (id: Long) -> Unit
) {
    Column(
        modifier = Modifier
            .testTag(DETAILS_SCREEN_CONTENT_TAG),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Top
    ) {
        TopAppBarContent(
            modifier = Modifier
                .fillMaxWidth()
                .weight(TOP_APP_BAR_WEIGHT),
            state = state,
            onImageClick = onImageClick,
            goBack = goBack,
            onEditClick = onEditClick
        )

        val scrollState = rememberScrollState()
        val showBottomFade = scrollState.canScrollForward

        Box(
            modifier = Modifier
                .padding(top = 16.dp)
                .weight(1f)
                .fillMaxWidth()
        ) {
            DetailsDemonstrationContent(
                modifier = Modifier
                    .fillMaxSize()
                    .verticalScroll(scrollState),
                state = state
            )

            if (showBottomFade) {
                Box(
                    modifier = Modifier
                        .align(Alignment.BottomCenter)
                        .fillMaxWidth()
                        .height(48.dp)
                        .background(
                            brush = Brush.verticalGradient(
                                colors = listOf(
                                    Color.Transparent,
                                    MaterialTheme.colorScheme.surface
                                )
                            )
                        )
                )
            }
        }
    }
}

@Composable
private fun TopAppBarContent(
    state: DetailsViewState,
    onImageClick: (productId: Long, index: Int) -> Unit,
    goBack: () -> Unit,
    onEditClick: (id: Long) -> Unit,
    modifier: Modifier = Modifier
) {
    val pagerState = rememberPagerState {
        state.images.size
    }

    LaunchedEffect(pagerState) {
        snapshotFlow { pagerState.currentPage }.collect { page ->
            state.setCurrentDisplayedImageIndex(page)
        }
    }

    Box(
        modifier = modifier.testTag(DETAILS_TOP_APP_BAR_TAG)
    ) {
        AppBarImageContent(
            modifier = Modifier
                .fillMaxSize()
                .align(Alignment.TopCenter),
            state = state,
            pagerState = pagerState,
            onImageClick = onImageClick
        )

        PlatoPagerIndicator(
            modifier = Modifier.align(Alignment.BottomCenter),
            show = state.images.size > 1,
            size = state.images.size,
            pagerState = pagerState
        )

        AppBarTopButtonsContent(
            state = state,
            goBack = goBack,
            onEditClick = onEditClick
        )

        ImageInteractionsSection(state)
    }
}

@Composable
private fun BoxScope.ImageInteractionsSection(state: DetailsViewState) {
    val context = LocalContext.current
    val resources = LocalResources.current
    val permissionState = rememberPermissionState(
        permission = Manifest.permission.WRITE_EXTERNAL_STORAGE
    )

    if (state.currentImage != null) {
        PlatoAlertDialog(
            text = state.permissionsAlertDialogText,
            showAlertDialog = state.showProvidePermissionsAlertDialog,
            confirmButtonText = stringResource(id = R.string.ok),
            onDismissRequest = {
                state.onShowPermissionsAlertDialog(false, null)
            },
            onConfirmButtonClick = {
                val activity = context.findActivity()
                openAppSettings(activity, state)
                state.onShowPermissionsAlertDialog(false, null)
            },
            dismissButtonText = stringResource(id = R.string.cancel),
            onDismissButtonClick = {
                state.onShowPermissionsAlertDialog(false, null)
            }
        )

        PlatoIconButton(
            modifier = Modifier
                .align(Alignment.BottomEnd)
                .padding(bottom = 4.dp, end = 4.dp),
            icon = PlatoIconType.Share.imageVector,
            onButtonClick = {
                state.onShareImageClick(state.currentImage)
            }
        )

        PlatoIconButton(
            modifier = Modifier
                .align(Alignment.BottomStart)
                .padding(bottom = 4.dp, start = 4.dp),
            icon = PlatoIconType.Download.imageVector,
            onButtonClick = {
                onDownloadClicked(
                    state = state,
                    permissionState = permissionState,
                    text = resources.getString(R.string.provide_permission)
                )
            }
        )
    }
}

private fun openAppSettings(activity: Activity, state: DetailsViewState) {
    val intent = state.appSettingsIntent
    activity.startActivity(intent)
}

private fun onDownloadClicked(
    state: DetailsViewState,
    permissionState: PermissionState,
    text: String
) {
    val image = requireNotNull(state.currentImage)
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
        state.saveFileToGallery(image)
    } else {
        if (permissionState.status.isGranted) {
            state.saveFileToGallery(image)
        } else {
            if (permissionState.status.shouldShowRationale.not()) {
                state.onShowPermissionsAlertDialog(true, text)
            }
            permissionState.launchPermissionRequest()
        }
    }
}

@Composable
private fun AppBarTopButtonsContent(
    state: DetailsViewState,
    goBack: () -> Unit,
    onEditClick: (id: Long) -> Unit
) {
    PlatoTopBar(
        modifier = Modifier.padding(top = 2.dp),
        goBack = goBack,
        defaultBackButton = false,
        backgroundColor = Color.Transparent,
        actions = {
            PlatoIconButton(
                icon = PlatoIconType.Edit.imageVector,
                onButtonClick = {
                    state.trackEditButtonClicked()
                    onEditClick(state.productId)
                }
            )
            Spacer(modifier = Modifier.width(12.dp))
            PlatoIconButton(
                icon = PlatoIconType.Delete.imageVector,
                onButtonClick = state.onDeleteProduct
            )
        }
    )
}

@Composable
private fun AppBarImageContent(
    state: DetailsViewState,
    pagerState: PagerState,
    onImageClick: (productId: Long, index: Int) -> Unit,
    modifier: Modifier = Modifier
) {
    if (state.images.isNotEmpty()) {
        HorizontalPager(
            modifier = modifier,
            state = pagerState
        ) { index ->
            val productImage = state.images[index]

            PlatoCard(
                modifier = Modifier.fillMaxSize(),
                shape = RoundedCornerShape(
                    bottomEnd = 16.dp,
                    bottomStart = 16.dp
                ),
                onClick = {
                    onImageClick(
                        state.productId,
                        index
                    )
                }
            ) {
                PlatoImage(
                    modifier = Modifier.fillMaxWidth(),
                    painter = rememberAsyncImagePainter(productImage.uriString),
                    contentScale = ContentScale.Crop
                )
            }
        }
    } else {
        PlatoPlaceholderImage(
            modifier = modifier
        )
    }
}

@Composable
private fun DetailsDemonstrationContent(state: DetailsViewState, modifier: Modifier = Modifier) {
    Surface(
        modifier = modifier
            .padding(horizontal = 16.dp)
            .testTag(DETAILS_DEMONSTRATION_CONTENT_TAG)
    ) {
        Column(
            horizontalAlignment = Alignment.Start,
            verticalArrangement = Arrangement.Top
        ) {
            val type = requireNotNull(state.type)

            Row(
                modifier = Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                Text(
                    modifier = Modifier.weight(1f),
                    text = state.name,
                    style = MaterialTheme.typography.headlineMedium,
                    overflow = TextOverflow.Ellipsis
                )
                PlatoIcon(
                    imageVector = type.icon(),
                    tint = type.color()
                )
            }

            if (state.shop.isNotEmpty()) {
                Spacer(modifier = Modifier.height(16.dp))
                Text(
                    text = stringResource(id = RString.shop),
                    style = MaterialTheme.typography.labelMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                Text(
                    text = state.shop,
                    style = MaterialTheme.typography.titleMedium
                )
            }

            if (state.createdDate.isNotEmpty()) {
                Spacer(modifier = Modifier.height(16.dp))
                Text(
                    text = stringResource(id = RString.date),
                    style = MaterialTheme.typography.labelMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                Text(
                    text = state.createdDate,
                    style = MaterialTheme.typography.titleMedium
                )
            }

            if (state.description.isNotEmpty()) {
                Spacer(modifier = Modifier.height(16.dp))
                Text(
                    text = stringResource(id = RString.description),
                    style = MaterialTheme.typography.labelMedium,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
                Text(
                    text = state.description,
                    style = MaterialTheme.typography.bodyLarge
                )
            }
        }
    }
}

@[Composable PreviewDarkLight]
private fun DetailsScreenPreview() {
    HateItOrRateItTheme {
        DetailsScreenContent(
            state = DetailsViewState(
                productId = 1L,
                appSettingsIntent = Intent(),
                isLoading = false,
                type = HateRateType.HATE,
                name = "Darren Stanton fn89r qw089h890qwn qw9ej90qw qw90jeqwjn qwe9jqw90e",
                description = "altera 23f8230rj 3nr289r 2389hr2389r 2398hr29837r 283hr8923r932",
                shop = "pulvinar",
                createdDate = "ornare",
                productFolderName = "Estelle Duke"
            ),
            goBack = {},
            onImageClick = { _, _ -> },
            onEditClick = {}
        )
    }
}

@[Composable PreviewDarkLight]
private fun DetailsScreenWithLoadingPreview() {
    HateItOrRateItTheme {
        DetailsScreenContent(
            state = DetailsViewState(
                productId = 1L,
                appSettingsIntent = Intent(),
                isLoading = true,
                type = HateRateType.HATE
            ),
            goBack = {},
            onImageClick = { _, _ -> },
            onEditClick = {}
        )
    }
}

@[Composable PreviewDarkLight]
private fun TopAppBarContentPreview(
    @PreviewParameter(StateProvider::class) state: DetailsViewState
) {
    HateItOrRateItTheme {
        TopAppBarContent(state = state, onImageClick = { _, _ -> }, goBack = {}, onEditClick = {})
    }
}

@[Composable PreviewDarkLight]
private fun DetailsDemonstrationContentPreview(
    @PreviewParameter(StateProvider::class) state: DetailsViewState
) {
    HateItOrRateItTheme {
        DetailsDemonstrationContent(
            state = state
        )
    }
}

private class StateProvider : PreviewParameterProvider<DetailsViewState> {
    override val values: Sequence<DetailsViewState>
        get() = sequenceOf(
            DetailsViewState(
                productId = 1L,
                name = "Darren Stanton fn89r qw089h890qwn qw9ej90qw qw90jeqwjn qwe9jqw90e",
                description = "altera 23f8230rj 3nr289r 2389hr2389r 2398hr29837r 283hr8923r932",
                shop = "pulvinar",
                createdDate = "ornare",
                productFolderName = "Estelle Duke",
                images = listOf(),
                type = HateRateType.HATE,
                showAlertDialog = false,
                productDeleted = false,
                setCurrentDisplayedImageIndex = {},
                setSnackbarMessage = {},
                saveFileToGallery = { _ -> },
                onShareImageClick = { _ -> },
                onShowPermissionsAlertDialog = { _, _ -> },
                appSettingsIntent = Intent()
            )
        )
}
