/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.android.apps.mytracks.services.sensors;

import static com.google.android.apps.mytracks.MyTracksConstants.TAG;
import com.google.android.apps.mytracks.MyTracksSettings;
import com.google.android.apps.mytracks.content.Sensor;
import com.google.android.maps.mytracks.R;

import com.dsi.ant.AntDefine;
import com.dsi.ant.AntMesg;
import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

/**
 * A sensor manager to connect ANT+ sensors.
 * This can include heart rate monitors.
 *
 * @author Sandor Dornbush
 */
public class AntDirectSensorManager extends AntSensorManager {

  /*
   * These constants are defined by the ANT+ heart rate monitor spec.
   */
  public static final byte HRM_CHANNEL = 0;
  public static final byte NETWORK_NUMBER = 1;
  public static final byte HEART_RATE_DEVICE_TYPE = 120;
  public static final byte POWER_DEVICE_TYPE = 11;
  public static final byte MANUFACTURER_ID = 1;
  public static final short CHANNEL_PERIOD = 8070;
  public static final byte RF_FREQUENCY = 57;

  private short deviceNumberHRM;

  public AntDirectSensorManager(Context context) {
    super(context);
    
    deviceNumberHRM = WILDCARD;

    // First read the the device id that we will be pairing with.
    SharedPreferences prefs = context.getSharedPreferences(
        MyTracksSettings.SETTINGS_NAME, 0);
    if (prefs != null) {
      deviceNumberHRM = 
        (short) prefs.getInt(context.getString(R.string.ant_heart_rate_sensor_id_key), 0);
    }
    Log.i(TAG, "Pairing with heart rate monitor: " + deviceNumberHRM);
  }

  @Override
  public void handleMessage(byte[] antMessage) {
    int channel =
      antMessage[AntMesg.MESG_DATA_OFFSET] & AntDefine.CHANNEL_NUMBER_MASK;
    switch (channel) {
      case HRM_CHANNEL:
        antDecodeHRM(antMessage);
        break;
      default:
        Log.d(TAG, "Unhandled message: " + channel);
    }
  }

  /**
   * Decode an ant heart rate monitor message.
   * @param antMessage The byte array received from the heart rate monitor.
   */
  private void antDecodeHRM(byte[] antMessage) {
    // message ID == broadcast
    switch (antMessage[AntMesg.MESG_ID_OFFSET]) {
      case AntMesg.MESG_BROADCAST_DATA_ID:
        handleBroadcastData(antMessage);
        break;
      case AntMesg.MESG_RESPONSE_EVENT_ID:
        handleMessageResponse(antMessage);
        break;
      case AntMesg.MESG_CHANNEL_ID_ID:
        handleChannelId(antMessage);
        break;
      default:
        Log.e(TAG, "Unexpected message id: " + antMessage[3]);
    }
  }

  private void handleBroadcastData(byte[] antMessage) {
    if (deviceNumberHRM == WILDCARD) {
      getAntReceiver().ANTRequestMessage(HRM_CHANNEL,
          AntMesg.MESG_CHANNEL_ID_ID);
      Log.d(TAG, "Requesting channel id id.");
    }

    setSensorState(Sensor.SensorState.CONNECTED);
    int bpm = (int) antMessage[10] & 0xFF;
    Sensor.SensorData.Builder b = Sensor.SensorData.newBuilder()
      .setValue(bpm)
      .setState(Sensor.SensorState.SENDING);
    sensorData =
      Sensor.SensorDataSet.newBuilder()
      .setCreationTime(System.currentTimeMillis())
      .setHeartRate(b)
      .build();
  }

  void handleChannelId(byte[] antMessage) {
    // Store the device id.
    deviceNumberHRM =
        (short) (((int) antMessage[3] & 0xFF  |
                  ((int) (antMessage[4] & 0xFF) << 8)) & 0xFFFF);
    Log.i(TAG, "Found device id: " + deviceNumberHRM);

    SharedPreferences prefs = context.getSharedPreferences(
        MyTracksSettings.SETTINGS_NAME, Context.MODE_PRIVATE);
    SharedPreferences.Editor editor = prefs.edit();
    editor.putInt(context.getString(R.string.ant_heart_rate_sensor_id_key), deviceNumberHRM);
    editor.commit();
  }

  private void handleMessageResponse(byte[] antMessage) {
    if (antMessage[3] == AntMesg.MESG_EVENT_ID &&
        antMessage[4] == AntDefine.EVENT_RX_SEARCH_TIMEOUT) {
      // Search timeout
      Log.w(TAG, "Search timed out. Unassigning channel.");
      getAntReceiver().ANTUnassignChannel((byte) 0);
      setSensorState(Sensor.SensorState.DISCONNECTED);
    } else if (antMessage[3] == AntMesg.MESG_UNASSIGN_CHANNEL_ID) {
      setSensorState(Sensor.SensorState.DISCONNECTED);
      Log.i(TAG, "Disconnected from the sensor: " + getSensorState());
    }
  }

  /**
   * Open a channel to the SRM head unit.
   */
  @Override
  public void setupChannel() {
    super.setupChannel();
    antChannelSetup(NETWORK_NUMBER,
        HRM_CHANNEL,
        deviceNumberHRM,
        HEART_RATE_DEVICE_TYPE,
        (byte) 0x01,
        CHANNEL_PERIOD,
        RF_FREQUENCY,
        (byte) 0);
    setSensorState(Sensor.SensorState.CONNECTING);
  }

  public short getDeviceNumberHRM() {
    return deviceNumberHRM;
  }

  void setDeviceNumberHRM(short deviceNumberHRM) {
    this.deviceNumberHRM = deviceNumberHRM;
  }
}
