#############################################################################
#   NiDeau                                                                  #
#   ======                                                                  #
#                                                                           #
#   by schaerphix                                                           #
#   date:       04.02.2024                                                  #
#   version:    v2.1.0                                                     #
#                                                                           #
#############################################################################


#   Import
#   ------
from kivymd.app import MDApp
from kivy.core.window import Window
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.network.urlrequest import UrlRequest
from kivy.clock import Clock

from kivymd.uix.button import MDIconButton, MDRectangleFlatIconButton
from kivymd.uix.label import MDLabel
from kivymd.uix.boxlayout import MDBoxLayout

from datetime import datetime
from bs4 import BeautifulSoup
from jnius import autoclass
import webbrowser

from nideau_init import *


#   classes
#   -------

class NiDeau:
    """
    class for the water temperature of switzerland.
    data src:    https://nidau-strandbad.wassertemperatur.live/
                 https://biel-strandbad.wassertemperatur.live/
                 https://api.existenz.ch/apiv1/hydro/latest?locations=2029
    """


    def __init__(self):
        self.version = 'v2.1.0'
        self.name = 'NiDeau'
        self.author = 'schaerphix'
        self.data_nidau = 'https://nidau-strandbad.wassertemperatur.live/'
        self.data_biel_sb = 'https://biel-strandbad.wassertemperatur.live/'
        self.data_aare_bruegg = 'https://api.existenz.ch/apiv1/hydro/latest?locations=2029'


    def get_water_live(self, data_source):
        result_class = self._url_request(data_source)

        if result_class.resp_status == 200:
            result = result_class.result
            html = BeautifulSoup(result, 'html.parser')
            water_live = html.find(class_="container").text.strip().split('\n')

            out_dictonary = {'name': water_live[0]}

            temp_list = []
            for i in range(1, len(water_live) - 1):
                temp_list.append(water_live[i])
                if len(temp_list) == 2:
                    out_dictonary[temp_list[0]] = self._edit_temperature(temp_list[1])
                    temp_list = []

            out_dictonary['time'] = (water_live[len(water_live) - 1][17:19] + '.' +
                                     water_live[len(water_live) - 1][14:16] + '.' +
                                     water_live[len(water_live) - 1][9:13] +
                                     water_live[len(water_live) - 1][19:]
                                     )
        else:
            out_dictonary = {'name': '',
                             'time': '',
                             'Pool': '--°C',
                             'See/Lac': '--°C'}
        return out_dictonary


    def get_aare_bafu(self, data_source, name):
        result_class = self._url_request(data_source)

        if result_class.resp_status == 200:
            result = result_class.result
            date_time = str(datetime.fromtimestamp(result['payload'][2]['timestamp']))
            date_time_temp = date_time[8:10] + '.' + date_time[5:7] + '.' + date_time[:4]
            date_time = date_time_temp + ' ' + date_time[11:19]

            out_dictonary = {'name': name,
                             'time': date_time,
                             'Fluss/See': self._edit_temperature(str(result['payload'][2]['val'])[
                                          :len(str(result['payload'][2]['val'])) - 1] + '°C')}
        else:
            out_dictonary = {'name': name,
                             'time': '',
                             'Fluss/See': '--°C'}
        return out_dictonary


    def _url_request(self, url):
        Clock.start_clock()
        req = UrlRequest(url)
        while not req.is_finished:
            Clock.tick()
        Clock.stop_clock()
        return req


    def _edit_temperature(self, value):
        if '°C' == value:
            return '--°C'
        elif '°C' in value:
            value = value.split('°C')[0]
        if '.' not in value:
            return value + '.0°C'
        elif len(value.split('.')) > 1:
            if len(value.split('.')[1]) > 0:
                return value.split('.')[0] + '.' + value.split('.')[1] + '°C'
            else:
                return value.split('.')[0] + '.0°C'
        else:
            return value


class NiDeauGUI(MDApp):

    def build(self):
        self.nidau = NiDeau()

        self.black = '#000000'
        self.big = 200          # mobile: 200,  pc: 100
        Window.clearcolor = (1, 1, 1, 1)

        self.__sys_language = self.__get_system_language()
        self.__language_code = self.__get_language_code(self.__sys_language)

        self.box_list = []
        #   for debug on pc (set text size to the half)
        #Window.size = (540, 1170)

        self.screen = MDBoxLayout(orientation='vertical')

        return self.main_page()


    #   main page
    #   ---------
    def main_page(self):
        self.box_list = self.get_data()

        self.screen.clear_widgets()
        main_layout = MDBoxLayout(orientation="vertical")

        #   header
        self._empty_line(main_layout)

        title_layout = MDBoxLayout(orientation="horizontal")

        title_layout.add_widget(MDIconButton(
            icon="information-outline",
            pos_hint={"center_x": 0.5, "center_y": 0.5},
            icon_size="35sp",
            theme_icon_color="Custom",
            icon_color=self.theme_cls.primary_color,
            on_press=self.info_button
            )
        )

        title_layout.add_widget(Image(source='NiDeau.png'))

        title_layout.add_widget(MDIconButton(
            icon="reload",
            pos_hint={"center_x": 0.5, "center_y": 0.5},
            icon_size="35sp",
            theme_icon_color="Custom",
            icon_color=self.theme_cls.primary_color,
            on_press=self.reload_button
            )
        )

        main_layout.add_widget(title_layout)

        self._empty_line(main_layout)

        #   show data
        for i in self.box_list:
            line_layout = MDBoxLayout(orientation='horizontal')
            self._empty_line(line_layout,(0.1, 1))
            self._lable_title(line_layout, i[2], (0.8, 1))
            main_layout.add_widget(line_layout)
            self._lable_temperature(main_layout, i[0])

            #       Add update date and time to main page,
            #self._lable_sub(main_layout, i[1])
            self._empty_line(main_layout)

        self.screen.add_widget(main_layout)

        return self.screen


    def get_data(self):
        data_a = self.nidau.get_water_live(self.nidau.data_nidau)
        output_c = [data_a['Pool'], data_a['time'], nideau_text['pool_text'][self.__language_code]]
        output_a = [data_a['See/Lac'], data_a['time'], nideau_text['lake_text'][self.__language_code]]

        data_b = self.nidau.get_water_live(self.nidau.data_biel_sb)
        output_d = [data_b['See/Lac'], data_b['time'], nideau_text['river_small_text'][self.__language_code]]

        data_c = self.nidau.get_aare_bafu(self.nidau.data_aare_bruegg, 'aare')
        output_b = [data_c['Fluss/See'], data_c['time'], nideau_text['river_text'][self.__language_code]]

        return [output_a, output_b, output_c, output_d]


    def reload_button(self, x):
        self.main_page()


    def info_button(self, x):
        self.go_info_page()


    def go_info_page(self):
        self.info_page()


    #   info Page
    #   ---------
    def info_page(self):
        self.screen.clear_widgets()
        main_layout = MDBoxLayout(orientation="vertical")

        #   title/logo
        self._empty_line(main_layout)
        main_layout.add_widget(Image(source='NiDeau.png'))
        self._empty_line(main_layout)

        #   last Update
        update_list = [nideau_text['lake_text'][self.__language_code] + ': ' + self.box_list[0][1],
                       nideau_text['river_text'][self.__language_code] + ': ' + self.box_list[1][1],
                       nideau_text['pool_text'][self.__language_code] + ': ' + self.box_list[2][1],
                       nideau_text['river_small_text'][self.__language_code] + ': ' + self.box_list[3][1],
                       ]
        self._text_block(main_layout, nideau_text['last_update_text'][self.__language_code], update_list)
        self._empty_line(main_layout)

        #   data source
        self._title_block(main_layout, nideau_text['data_source_text'][self.__language_code])
        self._text_body_block(main_layout, nideau_text['lake_and_pool_text'][self.__language_code] + ':')
        main_layout.add_widget(MDRectangleFlatIconButton(
            text='https://nidau-strandbad.wassertemperatur.live/',
            icon="web",
            line_color=(0, 0, 0, 0),
            pos_hint={"left_x": 0, "center_y": .5},
            #halign='right',
            on_press=self.link_nidau_strand
        ))

        self._text_body_block(main_layout, nideau_text['river_text'][self.__language_code] + ':')
        main_layout.add_widget(MDRectangleFlatIconButton(
            text='https://www.hydrodaten.admin.ch/de/seen-und-fluesse/\nstationen-und-daten/2029',
            icon="web",
            line_color=(0, 0, 0, 0),
            pos_hint={"left_x": 0, "center_y": .5},
            #halign='left',
            on_press=self.link_aare
        ))

        self._text_body_block(main_layout, nideau_text['river_small_text'][self.__language_code] + ':')
        main_layout.add_widget(MDRectangleFlatIconButton(
            text='https://biel-strandbad.wassertemperatur.live/',
            icon="web",
            line_color=(0, 0, 0, 0),
            pos_hint={"left_x": 0, "center_y": .5},
            #halign='left',
            on_press=self.link_zihl
        ))
        self._empty_line(main_layout)

        #   source code
        self._title_block(main_layout, nideau_text['source_code_text'][self.__language_code])
        main_layout.add_widget(MDRectangleFlatIconButton(
                    text='https://gitlab.com/schaerphix_tool/NiDeau',
                    icon="web",
                    line_color=(0, 0, 0, 0),
                    pos_hint={"left_x": 0, "center_y": .5},
                    on_press=self.link_source_code
                ))
        self._empty_line(main_layout)

        #   licence
        self._text_block(main_layout, nideau_text['licence_text'][self.__language_code],
                         ['GNU General Public License v3.0 '])
        self._empty_line(main_layout)

        #   About
        self._text_block(main_layout, nideau_text['about_text'][self.__language_code],
                         [nideau_text['version_text'][self.__language_code] + ': ' + (self.nidau.version),
                          nideau_text['author_text'][self.__language_code] + ': ' + self.nidau.author])
        self._empty_line(main_layout)

        #   bug report/contact
        self._title_block(main_layout, nideau_text['bugreport_text'][self.__language_code])
        main_layout.add_widget(MDRectangleFlatIconButton(
                text='https://gitlab.com/schaerphix_tool/NiDeau/-/issues/new',
                icon="web",
                line_color=(0, 0, 0, 0),
                pos_hint={"left_x": 0, "center_y": .5},
                #halign='left',
                on_press=self.link_issue
                ))

        self._text_body_block(main_layout,'schaerphix@gmail.com')
        self._empty_line(main_layout)

        #   reload button
        main_layout.add_widget(MDIconButton(
            icon="arrow-left",
            pos_hint={"center_x": 0.5, "center_y": 0.5},
            icon_size="35sp",
            theme_icon_color="Custom",
            icon_color=self.theme_cls.primary_color,
            on_press=self.reload_button
            )
        )
        self._empty_line(main_layout)

        self.screen.add_widget(main_layout)

        return self.screen


    def link_nidau_strand(self, x):
        self.open_web('https://nidau-strandbad.wassertemperatur.live/')


    def link_aare(self, x):
        self.open_web('https://www.hydrodaten.admin.ch/de/seen-und-fluesse/stationen-und-daten/2029')


    def link_zihl(self, x):
        self.open_web('https://biel-strandbad.wassertemperatur.live/')

    def link_source_code(self, x):
        self.open_web('https://gitlab.com/schaerphix_tool/NiDeau')


    def link_issue(self, x):
        self.open_web('https://gitlab.com/schaerphix_tool/NiDeau/-/issues/new')


    def open_web(self, web_link):
        webbrowser.open(web_link)

    #   lables
    #   ------
    def _lable_title(self, layout, text, size_hi=(1,1)):
        layout.add_widget(MDLabel(text=text,
                                  font_style='H4',
                                  pos_hint={'center_y': 0.6},
                                  size_hint=size_hi,
                                  halign='left',
                                  ))


    def _lable_temperature(self, layout, text):
        layout.add_widget(Label(text='[b]' + text + '[/b]',
                                markup=True,
                                color=self.black,
                                font_size=self.big,
                                pos_hint={'center_x': .6},
                                ))


    def _lable_sub(self, layout, text, size_hi=(1,1)):
        layout.add_widget(MDLabel(text=text,
                                  font_style='H6',
                                  pos_hint={'center_x': 0.5},
                                  size_hint=size_hi,
                                  halign='left',
                                  ))


    def _lable_text(self, layout, text, size_hi=(1,1)):
        layout.add_widget(MDLabel(text=text,
                                  font_style='Body2',
                                  pos_hint={'center_y': 0.4},
                                  size_hint=size_hi,
                                  halign='left',
                                  ))


    def _empty_line(self, layout, size_hi=(1,1)):
        self._lable_text(layout, '', size_hi)


    def _title_block(self, layout, title):
        line_layout = MDBoxLayout(orientation='horizontal')
        self._empty_line(line_layout, (0.1, 1))
        self._lable_sub(line_layout, title)
        layout.add_widget(line_layout)


    def _text_body_block(self, layout, text):
        line_layout = MDBoxLayout(orientation='horizontal')
        #self._lable_text(line_layout, '', (0.1, 1))
        self._empty_line(line_layout, (0.15, 1))
        self._lable_text(line_layout, text)
        layout.add_widget(line_layout)


    def _text_block(self, layout, title, text_list):
        self._title_block(layout, title)

        for i in text_list:
            self._text_body_block(layout, i)


    #   system infos
    #   ------------
    def __get_system_language(self):
        Locale = autoclass('java.util.Locale')
        return Locale.getDefault().getLanguage()


    def __get_language_code(self, sys_language):
        #   Language codes: 0: English      1: German       2: French       3: Italian
        if sys_language == 'en':
            return 0
        elif sys_language == 'de':
            return 1
        elif sys_language == 'fr':
            return 2
        elif sys_language == 'it':
            return 3
        else:
            return 0


#   main
#   ----
if __name__ == '__main__':
    app = NiDeauGUI()
    app.run()