#   NiDeau
#   ======
#
#   by schaerphix
#   date:   03.11.2023
#   v2.0.10

#   Import
#   ------
from kivy.app import App
from kivy.core.window import Window
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.uix.boxlayout import BoxLayout
from kivy.network.urlrequest import UrlRequest
from kivy.clock import Clock

from datetime import datetime
from bs4 import BeautifulSoup


#   classes
#   -------


class NiDeau:
    """
    class for the water temperature of switzerland.
    data src:    https://nidau-strandbad.wassertemperatur.live/
                 https://biel-strandbad.wassertemperatur.live/
                 https://api.existenz.ch/apiv1/hydro/latest?locations=2029
    """

    def __init__(self):
        self.version = 'v2.0.10'
        self.name = 'NiDeau'
        self.author = 'schaerphix'
        self.data_nidau = 'https://nidau-strandbad.wassertemperatur.live/'
        self.data_biel_sb = 'https://biel-strandbad.wassertemperatur.live/'
        self.data_aare_bruegg = 'https://api.existenz.ch/apiv1/hydro/latest?locations=2029'

    def get_water_live(self, data_source):
        result_class = self._url_request(data_source)

        if result_class.resp_status == 200:
            result = result_class.result
            html = BeautifulSoup(result, 'html.parser')
            water_live = html.find(class_="container").text.strip().split('\n')

            out_dictonary = {'name': water_live[0]}

            temp_list = []
            for i in range(1, len(water_live) - 1):
                temp_list.append(water_live[i])
                if len(temp_list) == 2:
                    out_dictonary[temp_list[0]] = self._edit_temperature(temp_list[1])
                    temp_list = []

            out_dictonary['time'] = (water_live[len(water_live) - 1][17:19] + '.' +
                                     water_live[len(water_live) - 1][14:16] + '.' +
                                     water_live[len(water_live) - 1][9:13] +
                                     water_live[len(water_live) - 1][19:]
                                     )
        else:
            out_dictonary = {'name': '',
                             'time': '',
                             'Pool': '--°C',
                             'See/Lac': '--°C'}
        return out_dictonary

    def get_aare_bafu(self, data_source, name):
        result_class = self._url_request(data_source)

        if result_class.resp_status == 200:
            result = result_class.result
            date_time = str(datetime.fromtimestamp(result['payload'][2]['timestamp']))
            date_time_temp = date_time[8:10] + '.' + date_time[5:7] + '.' + date_time[:4]
            date_time = date_time_temp + ' ' + date_time[11:19]

            out_dictonary = {'name': name,
                             'time': date_time,
                             'Fluss/See': self._edit_temperature(str(result['payload'][2]['val'])[
                                          :len(str(result['payload'][2]['val'])) - 1] + '°C')}
        else:
            out_dictonary = {'name': name,
                             'time': '',
                             'Fluss/See': '--°C'}
        return out_dictonary

    def _url_request(self, url):
        Clock.start_clock()
        req = UrlRequest(url)
        while not req.is_finished:
            Clock.tick()
        Clock.stop_clock()
        return req

    def _edit_temperature(self, value):
        if '°C' in value:
            value = value.split('°C')[0]
        if '.' not in value:
            return value + '.0°C'
        elif len(value.split('.')) > 1:
            if len(value.split('.')[1]) > 0:
                return value.split('.')[0] + '.' + value.split('.')[1] + '°C'
            else:
                return value.split('.')[0] + '.0°C'
        else:
            return value


class NiDeauGUI(App):

    def build(self):
        nidau = NiDeau()
        self.gray = '#bababa'
        self.black = '#000000'
        self.extra_small = 20   # mobile: 20,   pc: 10
        self.small = 25         # mobile: 25,   pc: 12
        self.mid = 60           # mobile: 60,   pc: 30
        self.big = 200          # mobile: 200,  pc: 100
        Window.clearcolor = (1, 1, 1, 1)

        #   for debug on pc (set text size to the half)
        #Window.size = (540, 1170)

        data_a = nidau.get_water_live(nidau.data_nidau)
        output_c = [data_a['Pool'], data_a['time'], 'Pool']
        output_a = [data_a['See/Lac'], data_a['time'], 'See']

        data_b = nidau.get_water_live(nidau.data_biel_sb)
        output_d = [data_b['See/Lac'], data_b['time'], 'Zihl']

        data_c = nidau.get_aare_bafu(nidau.data_aare_bruegg, 'aare')
        output_b = [data_c['Fluss/See'], data_c['time'], 'Aare']

        box_list = [output_a, output_b, output_c, output_d]

        main_layout = BoxLayout(orientation="vertical", spacing=10)

        main_layout.add_widget(Label(text='',
                                     color=self.gray,
                                     font_size=self.extra_small))

        main_layout.add_widget(Image(source='NiDeau.png'))

        main_layout.add_widget(Label(text='',
                                     color=self.gray,
                                     font_size=self.extra_small))

        for i in box_list:
            self._lable_title(main_layout, i[2])
            self._lable_temperature(main_layout, i[0])
            self._lable_sub(main_layout, i[1])

        main_layout.add_widget(Label(text=nidau.version,
                                     color=self.gray,
                                     font_size=self.extra_small, ))
        return main_layout

    def _lable_title(self, main_layout, text):
        main_layout.add_widget(Label(text=text,
                                     color=self.black,
                                     font_size=self.mid,
                                     pos_hint={'center_x': 0.2}))

    def _lable_temperature(self, main_layout, text):
        main_layout.add_widget(Label(text='[b]' + text + '[/b]',
                                     markup=True,
                                     color=self.black,
                                     font_size=self.big,
                                     pos_hint={'center_x': .6}))

    def _lable_sub(self, main_layout, text):
        main_layout.add_widget(Label(text=text,
                                     color=self.gray,
                                     font_size=self.small,
                                     pos_hint={'center_x': 0.5}))


#   main
#   ----
if __name__ == '__main__':
    app = NiDeauGUI()
    app.run()
