// Copyright 2020 terrier989@gmail.com.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import 'dart:async';

import '_xhr_http_client_request.dart';
import '_exports_in_browser.dart';

/// Browser implementation of _dart:io_ [HttpClient].
final class XhrHttpClient extends BrowserHttpClient {
  @override
  Duration idleTimeout = Duration(seconds: 15);

  @override
  Duration? connectionTimeout;

  @override
  int? maxConnectionsPerHost;

  @override
  bool autoUncompress = true;

  @override
  String? userAgent;

  @override
  Future<bool> Function(Uri url, String scheme, String realm)? authenticate;

  @override
  Future<bool> Function(String host, int port, String scheme, String realm)?
      authenticateProxy;

  @override
  bool Function(X509Certificate cert, String host, int port)?
      badCertificateCallback;

  @override
  String Function(Uri url)? findProxy;

  bool _isClosed = false;

  XhrHttpClient() : super.constructor();

  @override
  set connectionFactory(
    Future<ConnectionTask<Socket>> Function(
      Uri url,
      String? proxyHost,
      int? proxyPort,
    )? f,
  ) {
    // TODO: implement connectionFactory
  }

  @override
  set keyLog(Function(String line)? callback) {
    // TODO: implement keyLog
  }

  @override
  void addCredentials(
    Uri url,
    String realm,
    HttpClientCredentials credentials,
  ) {
    throw UnimplementedError();
  }

  @override
  void addProxyCredentials(
    String host,
    int port,
    String realm,
    HttpClientCredentials credentials,
  ) {}

  @override
  void close({bool force = false}) {
    _isClosed = true;
  }

  @override
  Future<BrowserHttpClientRequest> delete(String host, int? port, String path) {
    return open('DELETE', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> deleteUrl(Uri url) {
    return openUrl('DELETE', url);
  }

  @override
  Future<BrowserHttpClientRequest> get(String host, int? port, String path) {
    return open('GET', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> getUrl(Uri url) {
    return openUrl('GET', url);
  }

  @override
  Future<BrowserHttpClientRequest> head(String host, int? port, String path) {
    return open('HEAD', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> headUrl(Uri url) {
    return openUrl('HEAD', url);
  }

  @override
  Future<BrowserHttpClientRequest> open(
    String method,
    String host,
    int? port,
    String path,
  ) {
    String? query;
    final i = path.indexOf('?');
    if (i >= 0) {
      query = path.substring(i + 1);
      path = path.substring(0, i);
    }
    final uri = Uri(
      scheme: 'http',
      host: host,
      port: port,
      path: path,
      query: query,
      fragment: null,
    );
    return openUrl(method, uri);
  }

  @override
  Future<BrowserHttpClientRequest> openUrl(String method, Uri url) async {
    if (_isClosed) {
      throw StateError('HTTP client is closed');
    }
    var scheme = url.scheme;
    var needsNewUrl = false;
    if (scheme.isEmpty) {
      scheme = 'https';
      needsNewUrl = true;
    } else {
      switch (scheme) {
        case '':
          scheme = 'https';
          needsNewUrl = true;
          break;
        case 'http':
          break;
        case 'https':
          break;
        default:
          throw ArgumentError.value(url, 'url', 'Unsupported scheme');
      }
    }
    if (needsNewUrl) {
      url = Uri(
        scheme: scheme,
        userInfo: url.userInfo,
        host: url.host,
        port: url.port,
        query: url.query,
        fragment: url.fragment,
      );
    }
    return XhrHttpClientRequest(this, method, url);
  }

  @override
  Future<BrowserHttpClientRequest> patch(String host, int? port, String path) {
    return open('PATCH', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> patchUrl(Uri url) {
    return openUrl('PATCH', url);
  }

  @override
  Future<BrowserHttpClientRequest> post(String host, int? port, String path) {
    return open('POST', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> postUrl(Uri url) {
    return openUrl('POST', url);
  }

  @override
  Future<BrowserHttpClientRequest> put(String host, int? port, String path) {
    return open('PUT', host, port, path);
  }

  @override
  Future<BrowserHttpClientRequest> putUrl(Uri url) {
    return openUrl('PUT', url);
  }
}
