import 'package:catcher_2/model/platform_type.dart';
import 'package:catcher_2/model/report.dart';
import 'package:catcher_2/model/report_handler.dart';
import 'package:flutter/material.dart';
import 'package:sentry/sentry.dart';

class SentryHandler extends ReportHandler {
  SentryHandler(
    this.sentryClient, {
    this.userContext,
    this.enableDeviceParameters = true,
    this.enableApplicationParameters = true,
    this.enableCustomParameters = true,
    this.printLogs = true,
    this.customEnvironment,
    this.customRelease,
  });

  /// Sentry Client instance
  final SentryClient sentryClient;

  /// User data
  SentryUser? userContext;

  /// Enable device parameters to be generated by Catcher 2
  final bool enableDeviceParameters;

  /// Enable application parameters to be generated by Catcher 2
  final bool enableApplicationParameters;

  /// Enable custom parameters to be generated by Catcher 2
  final bool enableCustomParameters;

  /// Custom environment, if null, Catcher 2 will generate it
  final String? customEnvironment;

  /// Custom release, if null, Catcher 2 will generate it
  final String? customRelease;

  /// Enable additional logs printing
  final bool printLogs;

  @override
  Future<bool> handle(Report error, BuildContext? context) async {
    try {
      _printLog('Logging to sentry...');

      final tags = <String, dynamic>{};
      if (enableApplicationParameters) {
        tags.addAll(error.applicationParameters);
      }
      if (enableDeviceParameters) {
        tags.addAll(error.deviceParameters);
      }
      if (enableCustomParameters) {
        tags.addAll(error.customParameters);
      }

      final event = buildEvent(error, tags);
      await sentryClient.captureEvent(event, stackTrace: error.stackTrace);

      _printLog('Logged to sentry!');
      return true;
    } catch (exception, stackTrace) {
      _printLog('Failed to send sentry event: $exception $stackTrace');
      return false;
    }
  }

  String _getApplicationVersion(Report report) {
    var applicationVersion = '';
    final applicationParameters = report.applicationParameters;
    if (applicationParameters.containsKey('appName')) {
      applicationVersion += (applicationParameters['appName'] as String?)!;
    }
    if (applicationParameters.containsKey('version')) {
      applicationVersion += " ${applicationParameters["version"]}";
    }
    if (applicationVersion.isEmpty) {
      applicationVersion = '?';
    }
    return applicationVersion;
  }

  SentryEvent buildEvent(Report report, Map<String, dynamic> tags) =>
      SentryEvent(
        logger: 'Catcher 2',
        serverName: 'Catcher 2',
        release: customRelease ?? _getApplicationVersion(report),
        environment: customEnvironment ??
            (report.applicationParameters['environment'] as String?),
        message: const SentryMessage('Error handled by Catcher 2'),
        throwable: report.error,
        level: SentryLevel.error,
        culprit: '',
        tags: changeToSentryMap(tags),
        user: userContext,
      );

  Map<String, String> changeToSentryMap(Map<String, dynamic> map) {
    final sentryMap = <String, String>{};
    map.forEach((key, value) {
      final val = value.toString();
      sentryMap[key] = val.isNotEmpty ? val : 'none';
    });
    return sentryMap;
  }

  void _printLog(String message) {
    if (printLogs) {
      logger.info(message);
    }
  }

  @override
  List<PlatformType> getSupportedPlatforms() => [
        PlatformType.android,
        PlatformType.iOS,
        PlatformType.web,
        PlatformType.linux,
        PlatformType.macOS,
        PlatformType.windows,
      ];
}
