var app = app || {};

app.views = app.views || {};
app.views.utility = app.views.utility || {};

app.views.utility.Form = (function() {

	'use strict';

	return app.abstracts.BaseView.extend({

		// A template is required to display anything.
		template: null,

		events: {
			'change :input': 'process',
			'submit form': 'process',
			'change :input[type=checkbox]': 'onChangeCheckbox',
			'click .form-field-action': 'onFormFieldAction',
		},

		inputs: [],

		initialize: function() {

			_.bindAll(this,
				'process',
				'onKeyboardVisible'
			);
			this.process = _.throttle(this.process, 500, { leading: false });
			this.preparedInputs = this.prepareInputs();
			this.listenTo(app.device, 'keyboard:visible', this.onKeyboardVisible);
		},

		onKeyboardVisible: function() {

			this.scrollToFocusedInput();
		},

		scrollToFocusedInput: function() {

			var $focus = $(document.activeElement);
			if ($focus.length > 0 && $focus.is(':input') && $.contains(this.$el[0], $focus[0])) {
				var $label = $focus.parents('.form-row').first().find('.form-label').first();
				var top = ($label.length > 0 ? $label.offset().top : $focus.offset().top) + 20;
				this.$el.scrollTop(top);
			}
		},

		getInputValueOverride: function(name) {

			return app.settings.get(name);
		},

		getInputDefaultValue: function(name) {

			var input = this.getInputByName(name);
			return input && _.result(input, 'default') || null;
		},

		getInputByName: function(name) {

			return _.findWhere(this.getInputs(), { name: name });
		},

		getPreparedInputByName: function(name) {

			return _.findWhere(this.preparedInputs, { name: name });
		},

		getInputs: function() {

			return _.result(this, 'inputs') || [];
		},

		prepareInputs: function() {

			return _.map(this.getInputs(), this.prepareInput, this);
		},

		prepareInput: function(input) {

			input = _.clone(input);
			input.id = input.id || input.name;
			var value = this.getInputValueOverride(input.name);
			if (!_.isUndefined(value)) {
				input.value = value;
			} else if (input.value) {
				input.value = _.result(input, 'value');
			} else {
				input.value = input.default;
			}
			switch (input.type) {
				case 'select':
					if (input.multiple && _.isString(input.value)) {
						input.value = input.value.split(',');
					}
					input.options = _.result(input, 'options') || [];
					input.options = _.map(input.options, function(option) {
						option = _.clone(option);
						option.selected = false;
						if (input.multiple) {
							option.selected = _.contains(input.value, option.key);
						} else {
							option.selected = input.value === option.key;
						}
						option.label = _.result(option, 'label');
						return option;
					});
					break;
			}
			// This properties might be a function or just a value.
			_.each(['disabled', 'readonly', 'visible'], function(key) {
				input[key] = _.result(input, key);
			});
			// The 'visible' property should be TRUE unless explicitly set to FALSE.
			input.visible = input.visible !== false;
			return input;
		},

		serializeData: function() {

			var data = app.abstracts.BaseView.prototype.serializeData.apply(this, arguments);
			data.inputs = this.preparedInputs;
			return data;
		},

		onChangeCheckbox: function(evt) {

			var $target = $(evt.target);
			var $parentRow = $target.parents('.form-row');
			$parentRow.toggleClass('checked', $target.is(':checked'));
		},

		onFormFieldAction: function(evt) {

			var $target = $(evt.target);
			var name = $target.attr('data-input');
			if (!name) return;
			var input = this.getPreparedInputByName(name);
			if (!input) return;
			var action = _.findWhere(input.actions || [], { name: $target.attr('data-action') });
			if (!action || !action.fn) return;
			var fn = _.bind(action.fn, this);
			var formData = this.getFormData();
			var value = formData[name];
			var $input = this.$(':input[name="' + name + '"]');

			fn(value, function(error, newValue) {

				if (error) {
					return app.mainView.showMessage(error);
				}

				if (newValue) {
					$input.val(newValue).trigger('change');
				}
			});
		},

		showErrors: function(validationErrors) {

			if (!_.isEmpty(validationErrors)) {
				_.each(validationErrors, function(validationError) {
					var $field = this.$(':input[name="' + validationError.field + '"]');
					var $row = $field.parents('.form-row').first();
					var $error = $row.find('.form-error');
					$field.addClass('error');
					$error.append($('<div/>', {
						class: 'form-error-message',
						text: validationError.error,
					}));
				}, this);
			}
		},

		clearErrors: function() {

			this.$('.form-error-message').remove();
			this.$('.form-row.error').removeClass('error');
			this.$(':input.error').removeClass('error');
		},

		getFormData: function() {

			return this.$('form').serializeJSON();
		},

		allRequiredFieldsFilledIn: function() {

			var formData = this.getFormData();
			return _.every(this.preparedInputs, function(input) {
				return input.required !== true || !!formData[input.name];
			});
		},

		process: function(evt) {

			if (evt && evt.preventDefault) {
				evt.preventDefault();
			}

			this.clearErrors();

			var data = this.getFormData();

			// Set defaults.
			_.each(this.preparedInputs, function(input) {
				if (_.isUndefined(data[input.name]) && !_.isUndefined(input.default)) {
					data[input.name] = input.default;
				}
			});

			this.validate(data, _.bind(function(error, validationErrors) {

				if (error) {
					return app.mainView.showMessage(error);
				}

				if (!_.isEmpty(validationErrors)) {
					this.showErrors(validationErrors);
					this.onValidationErrors(validationErrors);
				} else {
					// No validation errors.
					_.each(this.preparedInputs, function(input) {
						switch (input.type) {
							case 'checkbox':
								// Force checkbox fields to have boolean values.
								data[input.name] = !!data[input.name];
								break;
						}
					});
					try {
						// Try saving.
						this.save(data);
					} catch (error) {
						app.log(error);
						return app.mainView.showMessage(error);
					}
				}

			}, this));
		},

		// `data` is an object containing the form data.
		// Execute the callback with an array of errors to indicate failure.
		// Execute the callback with no arguments to indicate success.
		validate: function(data, done) {

			async.map(this.preparedInputs || [], _.bind(function(input, next) {

				var value = data[input.name];
				var errors = [];

				if (input.required && _.isEmpty(value)) {
					errors.push({
						field: input.name,
						error: app.i18n.t('form.field-required', {
							label: _.result(input, 'label')
						}),
					});
					return next(null, errors);
				}

				if (input.validate) {
					var validateFn = _.bind(input.validate, this);
					try {
						validateFn(value, data);
					} catch (error) {
						errors.push({
							field: input.name,
							error: error,
						});
					}
				}

				if (!input.validateAsync) {
					return next(null, errors);
				}

				try {
					var validateAsyncFn = _.bind(input.validateAsync, this);
					validateAsyncFn(value, data, function(error) {
						if (error) {
							errors.push({
								field: input.name,
								error: error,
							});
						}
						next(null, errors);
					});
				} catch (error) {
					next(error);
				}

			}, this), function(error, results) {

				if (error) {
					return done(error);
				}

				// Flatten the errors array.
				var errors = Array.prototype.concat.apply([], results);

				done(null, errors);
			});
		},

		save: function(data) {

			app.settings.set(data);
		},

		onValidationErrors: function(validationErrors) {
			// Left empty intentionally.
			// Override as needed.
		},

	});

})();
