package com.github.muellerma.mute_reminder

import android.content.Context
import android.media.AudioDeviceInfo
import android.media.AudioManager
import android.os.Build
import android.util.Log
import androidx.core.content.getSystemService

class MediaAudioManager(private val context: Context) {
    private val audioManager: AudioManager = context.getSystemService()!!
    private val prefs = Prefs(context)

    fun shouldNotify(): Boolean {
        return (!prefs.notifyOnlyWhenMuted || isRingToneMuted()) && !isMediaMuted() && !usesRemoteOutput()
    }

    fun muteMedia() {
        try {
            audioManager.setStreamVolume(AudioManager.STREAM_MUSIC, 0, AudioManager.FLAG_SHOW_UI)
        } catch (e: SecurityException) {
            // setStreamVolume may fail at don't-disturb Mode
            context.showToast(R.string.mute_media_failed)
        }
    }

    private fun isMediaMuted(): Boolean {
        val mediaVolume = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
        Log.d(TAG, "mediaVolume=$mediaVolume")
        return mediaVolume == 0
    }

    private fun isRingToneMuted(): Boolean {
        val ringVolume = audioManager.getStreamVolume(AudioManager.STREAM_RING)
        Log.d(TAG, "ringVolume=$ringVolume")
        return ringVolume == 0
    }

    private fun usesRemoteOutput(): Boolean {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val ignoreOutputs = mutableListOf(
                AudioDeviceInfo.TYPE_BLUETOOTH_A2DP,
                AudioDeviceInfo.TYPE_WIRED_HEADPHONES,
                AudioDeviceInfo.TYPE_WIRED_HEADSET
            )

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                ignoreOutputs.add(AudioDeviceInfo.TYPE_USB_HEADSET)
            }

            val outputs = audioManager
                .getDevices(AudioManager.GET_DEVICES_OUTPUTS)
                .filter { device -> device.isSink }

            Log.d(TAG, "Connected to sinks: ${outputs.joinToString("|") { "${it.productName}, ${it.type}" }}")

            val remoteOutputs = outputs
                .filter { device -> device.type in ignoreOutputs }

            Log.d(TAG, "Connected to remote sinks: ${remoteOutputs.joinToString("|") { "${it.productName}, ${it.type}" }}")
            remoteOutputs.isNotEmpty()
        } else {
            @Suppress("DEPRECATION")
            audioManager.isWiredHeadsetOn || audioManager.isBluetoothA2dpOn
        }
    }

    companion object {
        private val TAG = MediaAudioManager::class.java.simpleName
    }
}