"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useOverlayPosition = useOverlayPosition;
var _react = _interopRequireDefault(require("react"));
var _reactNative = require("react-native");
var _utils = require("@react-native-aria/utils");
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
//@ts-ignore

const measureOffset = ref => new Promise(resolve => {
  if (ref.current) {
    ref.current.measureInWindow((x, y, width, height) => {
      resolve({
        top: y,
        left: x,
        width,
        height
      });
    });
  }
});
const getArrowPropsWithStatusBarHeight = ({
  top,
  left
}) => {
  let topWithStatusBarHeight = top;
  if (typeof top !== 'undefined' && typeof _utils2.APPROX_STATUSBAR_HEIGHT !== 'undefined') {
    topWithStatusBarHeight = top + _utils2.APPROX_STATUSBAR_HEIGHT;
  } else {
    topWithStatusBarHeight = undefined;
  }
  return {
    style: {
      left: left,
      top: topWithStatusBarHeight
    }
  };
};
function useOverlayPosition(props) {
  var _position$position, _position$position2;
  let {
    targetRef,
    overlayRef,
    placement = 'bottom',
    offset = 0,
    crossOffset = 0,
    isOpen = true,
    shouldFlip = true,
    shouldOverlapWithTrigger = false
  } = props;
  let [position, setPosition] = _react.default.useState({
    position: {},
    arrowOffsetLeft: undefined,
    arrowOffsetTop: undefined,
    maxHeight: undefined,
    placement: undefined
  });

  // Layout measurement happens asynchronously in RN. This causes initial flickr. Using opacity and setting it to 1 post calculation prevents that.
  let [rendered, setRendered] = _react.default.useState(false);
  let updatePosition = async () => {
    const [overlayOffset, triggerOffset] = await Promise.all([measureOffset(overlayRef), measureOffset(targetRef)]);

    // Sometimes measure returns height/width 0. Best solution would be to use onLayout callback, but that might diverege from React Aria's useOverlayPosition API. Decide later, this works for now
    if (!overlayOffset.width || !overlayOffset.height || !triggerOffset.width || !triggerOffset.height) {
      requestAnimationFrame(updatePosition);
      return;
    }
    const {
      height: windowHeight,
      width: windowWidth
    } = _reactNative.Dimensions.get('window');
    const positions = calculatePosition({
      placement: translateRTL(placement),
      targetNode: triggerOffset,
      overlayNode: overlayOffset,
      scrollNode: overlayOffset,
      padding: 0,
      shouldFlip,
      boundaryElement: {
        top: 0,
        left: 0,
        width: windowWidth,
        height: windowHeight
      },
      offset,
      crossOffset,
      shouldOverlapWithTrigger
    });
    setPosition(positions);
    setRendered(true);
  };
  _react.default.useEffect(() => {
    return () => {
      setRendered(false);
    };
  }, []);
  _react.default.useLayoutEffect(() => {
    updatePosition();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [placement, isOpen, offset, shouldFlip, crossOffset, shouldOverlapWithTrigger]);
  const style = {
    ...position.position
  };
  if (position !== null && position !== void 0 && (_position$position = position.position) !== null && _position$position !== void 0 && _position$position.top || (position === null || position === void 0 || (_position$position2 = position.position) === null || _position$position2 === void 0 ? void 0 : _position$position2.top) === 0) {
    var _position$position3;
    style.top = ((position === null || position === void 0 || (_position$position3 = position.position) === null || _position$position3 === void 0 ? void 0 : _position$position3.top) || 0) + (_utils2.APPROX_STATUSBAR_HEIGHT || 0);
  }
  const arrowPropsWithStatusBarHeight = getArrowPropsWithStatusBarHeight({
    left: position === null || position === void 0 ? void 0 : position.arrowOffsetLeft,
    top: position === null || position === void 0 ? void 0 : position.arrowOffsetTop
  });
  const returnProps = {
    rendered,
    overlayProps: {
      style
    },
    placement: position.placement,
    arrowProps: arrowPropsWithStatusBarHeight,
    updatePosition
  };
  if (position.maxHeight !== undefined) {
    //@ts-ignore
    returnProps.overlayProps.style.maxHeight = position.maxHeight;
  }
  return returnProps;
}
function translateRTL(position) {
  if ((0, _utils.isRTL)()) {
    return position.replace('start', 'right').replace('end', 'left');
  }
  return position.replace('start', 'left').replace('end', 'right');
}
const calculatePosition = opts => {
  let {
    placement,
    targetNode,
    overlayNode,
    scrollNode,
    padding,
    shouldFlip,
    boundaryElement,
    offset,
    crossOffset,
    shouldOverlapWithTrigger
  } = opts;
  let childOffset = targetNode;
  let isContainerPositioned = false;
  let overlaySize = overlayNode;
  let margins = {
    top: 0,
    bottom: 0,
    left: 0,
    right: 0
  };
  let scrollSize = scrollNode;
  let boundaryDimensions = boundaryElement;
  let containerOffsetWithBoundary = overlayNode;
  return calculatePositionInternal(placement, childOffset, overlaySize, scrollSize, margins, padding, shouldFlip, boundaryDimensions, containerOffsetWithBoundary, offset, crossOffset, isContainerPositioned, shouldOverlapWithTrigger);
};
function calculatePositionInternal(placementInput, childOffset, overlaySize, scrollSize, margins, padding, flip, boundaryDimensions, containerOffsetWithBoundary, offset, crossOffset, isContainerPositioned, shouldOverlapWithTrigger) {
  let placementInfo = parsePlacement(placementInput);
  let {
    size,
    crossAxis,
    crossSize,
    placement,
    crossPlacement
  } = placementInfo;
  let position = computePosition(childOffset, boundaryDimensions, overlaySize, placementInfo, offset, crossOffset, containerOffsetWithBoundary, isContainerPositioned);
  let normalizedOffset = offset;
  let space = getAvailableSpace(boundaryDimensions, containerOffsetWithBoundary, childOffset, margins, padding + offset, placementInfo);
  if (flip && scrollSize[size] > space) {
    let flippedPlacementInfo = parsePlacement(`${FLIPPED_DIRECTION[placement]} ${crossPlacement}`);
    let flippedPosition = computePosition(childOffset, boundaryDimensions, overlaySize, flippedPlacementInfo, offset, crossOffset, containerOffsetWithBoundary, isContainerPositioned);
    let flippedSpace = getAvailableSpace(boundaryDimensions, containerOffsetWithBoundary, childOffset, margins, padding + offset, flippedPlacementInfo);

    // If the available space for the flipped position is greater than the original available space, flip.
    if (flippedSpace > space) {
      placementInfo = flippedPlacementInfo;
      position = flippedPosition;
      normalizedOffset = offset;
    }
  }
  let delta = getDelta(crossAxis, position[crossAxis], overlaySize[crossSize], boundaryDimensions, padding);
  position[crossAxis] += delta;
  let maxHeight = getMaxHeight(position, boundaryDimensions, containerOffsetWithBoundary, childOffset, margins, padding);
  overlaySize.height = Math.min(overlaySize.height, maxHeight);
  position = computePosition(childOffset, boundaryDimensions, overlaySize, placementInfo, normalizedOffset, crossOffset, containerOffsetWithBoundary, isContainerPositioned);
  delta = getDelta(crossAxis, position[crossAxis], overlaySize[crossSize], boundaryDimensions, padding);
  position[crossAxis] += delta;
  let arrowPosition = {};
  arrowPosition[crossAxis] = childOffset[crossAxis] - position[crossAxis] + childOffset[crossSize] / 2;
  if (shouldOverlapWithTrigger) {
    position[FLIPPED_DIRECTION[placementInfo.placement]] = position[FLIPPED_DIRECTION[placementInfo.placement]] - childOffset[size];
  }
  return {
    position,
    maxHeight,
    arrowOffsetLeft: arrowPosition.left,
    arrowOffsetTop: arrowPosition.top,
    placement: placementInfo.placement
  };
}
function getDelta(axis, offset, size, containerDimensions, padding) {
  //@ts-ignore
  let containerScroll = containerDimensions[axis];
  //@ts-ignore
  let containerHeight = containerDimensions[AXIS_SIZE[axis]];
  let startEdgeOffset = offset - padding - containerScroll;
  let endEdgeOffset = offset + padding - containerScroll + size;
  if (startEdgeOffset < 0) {
    return -startEdgeOffset;
  } else if (endEdgeOffset > containerHeight) {
    return Math.max(containerHeight - endEdgeOffset, -startEdgeOffset);
  } else {
    return 0;
  }
}
function getMaxHeight(position, boundaryDimensions, _containerOffsetWithBoundary, childOffset, _margins, _padding) {
  return position.top != null ?
  // We want the distance between the top of the overlay to the bottom of the boundary
  Math.max(0, boundaryDimensions.height -
  // this is the bottom of the boundary
  position.top // this is the top of the overlay
  ) :
  // We want the distance between the top of the trigger to the top of the boundary
  Math.max(0, childOffset.top -
  // this is the top of the trigger
  0 // this is the top of the boundary
  );
}
function computePosition(childOffset, boundaryDimensions, overlaySize, placementInfo, offset, crossOffset, _containerOffsetWithBoundary, _isContainerPositioned) {
  let {
    placement,
    crossPlacement,
    axis,
    crossAxis,
    size,
    crossSize
  } = placementInfo;
  let position = {};
  //@ts-ignore
  position[crossAxis] = childOffset[crossAxis];
  if (crossPlacement === 'center') {
    position[crossAxis] += (childOffset[crossSize] - overlaySize[crossSize]) / 2;
  } else if (crossPlacement !== crossAxis) {
    position[crossAxis] += childOffset[crossSize] - overlaySize[crossSize];
  }
  position[crossAxis] += crossOffset;

  // this is button center position - the overlay size + half of the button to align bottom of overlay with button center
  let minViablePosition = childOffset[crossAxis] + childOffset[crossSize] / 2 - overlaySize[crossSize];
  // this is button position of center, aligns top of overlay with button center
  let maxViablePosition = childOffset[crossAxis] + childOffset[crossSize] / 2;

  // clamp it into the range of the min/max positions
  position[crossAxis] = Math.min(Math.max(minViablePosition, position[crossAxis]), maxViablePosition);

  // Floor these so the position isn't placed on a partial pixel, only whole pixels. Shouldn't matter if it was floored or ceiled, so chose one.
  if (placement === axis) {
    // If the container is positioned (non-static), then we use the container's actual
    // height, as `bottom` will be relative to this height.  But if the container is static,
    // then it can only be the `document.body`, and `bottom` will be relative to _its_
    // container, which should be as large as boundaryDimensions.
    const containerHeight = boundaryDimensions[size];
    position[FLIPPED_DIRECTION[axis]] = Math.floor(containerHeight - childOffset[axis] + offset);
  } else {
    position[axis] = Math.floor(childOffset[axis] + childOffset[size] + offset);
  }
  return position;
}
function getAvailableSpace(boundaryDimensions, _containerOffsetWithBoundary, childOffset, _margins, padding, placementInfo) {
  let {
    placement,
    axis,
    size
  } = placementInfo;
  if (placement === axis) {
    return Math.max(0, childOffset[axis] - padding);
  }
  return Math.max(0, boundaryDimensions[size] - childOffset[axis] - childOffset[size] - padding);
}
const AXIS = {
  top: 'top',
  bottom: 'top',
  left: 'left',
  right: 'left'
};
const FLIPPED_DIRECTION = {
  top: 'bottom',
  bottom: 'top',
  left: 'right',
  right: 'left'
};
const CROSS_AXIS = {
  top: 'left',
  left: 'top'
};
const AXIS_SIZE = {
  top: 'height',
  left: 'width'
};
const PARSED_PLACEMENT_CACHE = {};
function parsePlacement(input) {
  if (PARSED_PLACEMENT_CACHE[input]) {
    return PARSED_PLACEMENT_CACHE[input];
  }
  let [placement, crossPlacement] = input.split(' ');
  let axis = AXIS[placement] || 'right';
  let crossAxis = CROSS_AXIS[axis];
  if (!AXIS[crossPlacement]) {
    crossPlacement = 'center';
  }
  let size = AXIS_SIZE[axis];
  let crossSize = AXIS_SIZE[crossAxis];
  PARSED_PLACEMENT_CACHE[input] = {
    placement,
    crossPlacement,
    axis,
    crossAxis,
    size,
    crossSize
  };
  return PARSED_PLACEMENT_CACHE[input];
}
//# sourceMappingURL=useOverlayPosition.js.map