"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.hasYarn = hasYarn;
exports.helpMisspelledCommand = helpMisspelledCommand;
exports.joinOrigin = void 0;
exports.makeInstall = makeInstall;
exports.prettyOrigin = prettyOrigin;
exports.removeDirectory = removeDirectory;
exports.splitOrigin = void 0;
exports.writeFileIfChanged = writeFileIfChanged;
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _chalk = _interopRequireDefault(require("chalk"));
var _fuzzaldrin = require("fuzzaldrin");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function removeDirectory(dir, onlyContent = false) {
  if (!_fs.default.existsSync(dir)) return;
  const list = _fs.default.readdirSync(dir);
  for (let i = 0; i < list.length; i++) {
    const filename = _path.default.join(dir, list[i]);
    const stat = _fs.default.statSync(filename);
    if (filename === "." || filename === "..") {
      // pass these files
    } else if (stat.isDirectory()) {
      // rmdir recursively
      removeDirectory(filename);
    } else {
      _fs.default.unlinkSync(filename);
    }
  }
  if (!onlyContent) {
    _fs.default.rmdirSync(dir);
  }
}
function prettyOrigin(origins) {
  try {
    return origins.map(origin => origin.join(":")).join(", ");
  } catch (e) {
    return "";
  }
}

/**
 * .. js:function:: helpMisspelledCommand(command [, availableCommands = []])
 *    :param: command - command passed to CLI
 *    :param: availableCommands - all commands defined in commander.js
 *
 *    If unknown commands is passed to CLI, check it agains all available commands
 *    for possible misspelled letter. Output help with suggestions to console.
 */
function helpMisspelledCommand(command, availableCommands = []) {
  const commandNames = availableCommands.map(command => command.name());

  // if no command is supplied, then commander.js shows help automatically
  if (!command || commandNames.includes(command)) {
    return;
  }
  const suggestions = commandNames.map(name => ({
    name,
    score: (0, _fuzzaldrin.score)(name, command.slice(0, name.length))
  })).filter(nameScore => nameScore.score > 0).slice(0, 3).map(commandScore => _chalk.default.inverse(commandScore.name)).join(", ");
  console.log(`lingui: command ${command} is not a lingui command. ` + `See 'lingui --help' for the list of available commands.`);
  if (suggestions) {
    console.log();
    console.log(`Did you mean: ${suggestions}?`);
  }
}
const splitOrigin = origin => origin.split(":");
exports.splitOrigin = splitOrigin;
const joinOrigin = origin => origin.join(":");
exports.joinOrigin = joinOrigin;
function writeFileIfChanged(filename, newContent) {
  if (_fs.default.existsSync(filename)) {
    const raw = _fs.default.readFileSync(filename).toString();
    if (newContent !== raw) {
      _fs.default.writeFileSync(filename, newContent);
    }
  } else {
    _fs.default.writeFileSync(filename, newContent);
  }
}
function hasYarn() {
  return _fs.default.existsSync(_path.default.resolve("yarn.lock"));
}
function makeInstall() {
  const withYarn = hasYarn();
  return (packageName, dev = false) => withYarn ? `yarn add ${dev ? "--dev " : ""}${packageName}` : `npm install ${dev ? "--save-dev" : "--save"} ${packageName}`;
}