import * as React from 'react';
import { Children } from 'react';
import extractLengthList from './extractLengthList';
import { pickNotNil } from '../util';
import { stringifyPropsForFabric } from './extractProps';
const fontRegExp = /^\s*((?:(?:normal|bold|italic)\s+)*)(?:(\d+(?:\.\d+)?(?:%|px|em|pt|pc|mm|cm|in]))*(?:\s*\/.*?)?\s+)?\s*"?([^"]*)/i;
const fontFamilyPrefix = /^[\s"']*/;
const fontFamilySuffix = /[\s"']*$/;
const commaReg = /\s*,\s*/g;
const cachedFontObjectsFromString = {};
function extractSingleFontFamily(fontFamilyString) {
  // SVG on the web allows for multiple font-families to be specified.
  // For compatibility, we extract the first font-family, hoping
  // we'll get a match.
  return fontFamilyString ? fontFamilyString.split(commaReg)[0].replace(fontFamilyPrefix, '').replace(fontFamilySuffix, '') : null;
}
function parseFontString(font) {
  if (Object.prototype.hasOwnProperty.call(cachedFontObjectsFromString, font)) {
    return cachedFontObjectsFromString[font];
  }
  const match = fontRegExp.exec(font);
  if (!match) {
    cachedFontObjectsFromString[font] = null;
    return null;
  }
  const isBold = /bold/.exec(match[1]);
  const isItalic = /italic/.exec(match[1]);
  cachedFontObjectsFromString[font] = {
    fontSize: match[2] || 12,
    fontWeight: isBold ? 'bold' : 'normal',
    fontStyle: isItalic ? 'italic' : 'normal',
    fontFamily: extractSingleFontFamily(match[3])
  };
  return cachedFontObjectsFromString[font];
}
export function extractFont(props) {
  const {
    fontData,
    fontStyle,
    fontVariant,
    fontWeight,
    fontStretch,
    fontSize,
    fontFamily,
    textAnchor,
    textDecoration,
    letterSpacing,
    wordSpacing,
    kerning,
    fontFeatureSettings,
    fontVariantLigatures,
    fontVariationSettings,
    font
  } = props;
  const ownedFont = pickNotNil({
    fontData,
    fontStyle,
    fontVariant,
    fontWeight,
    fontStretch,
    fontSize,
    fontFamily: extractSingleFontFamily(fontFamily),
    textAnchor,
    textDecoration,
    letterSpacing,
    wordSpacing,
    kerning,
    fontFeatureSettings,
    fontVariantLigatures,
    fontVariationSettings
  });
  const baseFont = typeof font === 'string' ? parseFontString(font) : font;
  const fontProps = {
    ...baseFont,
    ...ownedFont
  };
  const stringifiedFontProps = {};
  Object.keys(fontProps).map(k => stringifiedFontProps[k] = fontProps[k] === null ? null : String(fontProps[k]));
  return stringifiedFontProps;
}
let TSpan;
export function setTSpan(TSpanImplementation) {
  TSpan = TSpanImplementation;
}
function getChild(child) {
  if (typeof child === 'string' || typeof child === 'number') {
    return /*#__PURE__*/React.createElement(TSpan, null, String(child));
  } else {
    return child;
  }
}
export default function extractText(props, container) {
  const {
    x,
    y,
    dx,
    dy,
    rotate,
    children,
    inlineSize,
    baselineShift,
    verticalAlign,
    alignmentBaseline
  } = props;
  const textChildren = typeof children === 'string' || typeof children === 'number' ? container ? /*#__PURE__*/React.createElement(TSpan, null, String(children)) : null : Children.count(children) > 1 || Array.isArray(children) ? Children.map(children, getChild) : children;
  const stringifiedTextProps = stringifyPropsForFabric({
    inlineSize,
    baselineShift,
    verticalAlign
  });
  return {
    content: textChildren === null ? String(children) : null,
    children: textChildren,
    ...stringifiedTextProps,
    alignmentBaseline,
    font: extractFont(props),
    x: extractLengthList(x),
    y: extractLengthList(y),
    dx: extractLengthList(dx),
    dy: extractLengthList(dy),
    rotate: extractLengthList(rotate)
  };
}
//# sourceMappingURL=extractText.js.map