"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSliderThumb = useSliderThumb;

var _utils = require("@react-aria/utils");

var _utils2 = require("./utils");

var _react = require("react");

var _label = require("@react-aria/label");

var _useMove = require("./useMove");

var _utils3 = require("@react-native-aria/utils");

/**
 * Provides behavior and accessibility for a thumb of a slider component.
 *
 * @param opts Options for this Slider thumb.
 * @param state Slider state, created via `useSliderState`.
 */
function useSliderThumb(opts, state, isReversed) {
  var _opts$ariaLabelledby;

  let {
    index,
    isDisabled,
    trackLayout
  } = opts;
  let isVertical = opts.orientation === 'vertical';
  const direction = (0, _utils3.isRTL)() ? 'rtl' : undefined;

  let labelId = _utils2.sliderIds.get(state);

  const {
    labelProps,
    fieldProps
  } = (0, _label.useLabel)({ ...opts,
    'id': (0, _utils2.getSliderThumbId)(state, index),
    'aria-labelledby': `${labelId} ${(_opts$ariaLabelledby = opts['aria-labelledby']) !== null && _opts$ariaLabelledby !== void 0 ? _opts$ariaLabelledby : ''}`.trim()
  });
  const stateRef = (0, _react.useRef)(null);
  stateRef.current = state;
  let reverseX = isReversed || direction === 'rtl';
  let currentPosition = (0, _react.useRef)(null);
  const [startPosition, setStartPosition] = (0, _react.useState)(0);
  let {
    moveProps
  } = (0, _useMove.useMove)({
    onMoveStart() {
      state.setThumbDragging(index, true);
      let size = isVertical ? trackLayout.height : trackLayout.width;
      setStartPosition(stateRef.current.getThumbPercent(index) * size);
    },

    onMove({
      deltaX,
      deltaY
    }) {
      let size = isVertical ? trackLayout.height : trackLayout.width;

      if (currentPosition.current == null) {
        currentPosition.current = stateRef.current.getThumbPercent(index) * size;
      }

      let delta = isVertical ? deltaY : deltaX;

      if (reverseX) {
        if (!isVertical) {
          delta = -delta;
        }
      } else {
        if (isVertical) {
          delta = -delta;
        }
      }

      const position = startPosition + delta;
      stateRef.current.setThumbPercent(index, (0, _utils.clamp)(position / size, 0, 1));
    },

    onMoveEnd() {
      state.setThumbDragging(index, false);
    }

  });
  state.setThumbEditable(index, !isDisabled);

  const onAccessibilityAction = event => {
    var _ref, _ref2;

    const max = state.getThumbMinValue(index);
    const min = state.getThumbMaxValue(index);
    const value = state.getThumbValue(index);
    const incrementValue = Math.min((_ref = value + state.step) !== null && _ref !== void 0 ? _ref : 1, max);
    const decrementValue = Math.max((_ref2 = value - state.step) !== null && _ref2 !== void 0 ? _ref2 : 1, min);

    switch (event.nativeEvent.actionName) {
      case 'increment':
        state.setThumbValue(index, incrementValue);
        break;

      case 'decrement':
        state.setThumbValue(index, decrementValue);
        break;

      default:
        break;
    }
  };

  return {
    inputProps: { ...fieldProps,
      'disabled': isDisabled,
      'role': 'adjustable',
      'aria-value': {
        min: state.getThumbMinValue(index),
        max: state.getThumbMaxValue(index),
        now: state.getThumbValue(index)
      },
      'accessibilityActions': [{
        name: 'increment',
        label: 'Increment'
      }, {
        name: 'decrement',
        label: 'Decrement'
      }],
      onAccessibilityAction
    },
    thumbProps: !isDisabled ? moveProps : {},
    labelProps
  };
}
//# sourceMappingURL=useSliderThumb.js.map