import { useToggleState } from "@react-stately/toggle";
import { useLabel } from "@react-aria/label";
import { filterDOMProps, mergeProps } from "@react-aria/utils";
import { useToggle } from "@react-aria/toggle";
import { useEffect } from "react";
import _babelRuntimeHelpersEsmExtends from "@babel/runtime/helpers/esm/extends";

/**
 * Provides the behavior and accessibility implementation for a checkbox component.
 * Checkboxes allow users to select multiple items from a list of individual items, or
 * to mark one individual item as selected.
 * @param props - Props for the checkbox.
 * @param state - State for the checkbox, as returned by `useToggleState`.
 * @param inputRef - A ref for the HTML input element.
 */
export function useCheckbox(props, state, inputRef) {
  let {
    inputProps
  } = useToggle(props, state, inputRef);
  let {
    isSelected
  } = state;
  let {
    isIndeterminate
  } = props;
  useEffect(() => {
    // indeterminate is a property, but it can only be set via javascript
    // https://css-tricks.com/indeterminate-checkboxes/
    if (inputRef.current) {
      inputRef.current.indeterminate = isIndeterminate;
    }
  });
  return {
    inputProps: _babelRuntimeHelpersEsmExtends({}, inputProps, {
      checked: isSelected,
      'aria-checked': isIndeterminate ? 'mixed' : isSelected
    })
  };
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
const $c1125120fa765e2757aafb57a69ea88$export$checkboxGroupNames = new WeakMap();

/**
 * Provides the behavior and accessibility implementation for a checkbox group component.
 * Checkbox groups allow users to select multiple items from a list of options.
 * @param props - Props for the checkbox group.
 * @param state - State for the checkbox group, as returned by `useCheckboxGroupState`.
 */
export function useCheckboxGroup(props, state) {
  let {
    isDisabled,
    name
  } = props;
  let {
    labelProps,
    fieldProps
  } = useLabel(_babelRuntimeHelpersEsmExtends({}, props, {
    // Checkbox group is not an HTML input element so it
    // shouldn't be labeled by a <label> element.
    labelElementType: 'span'
  }));
  let domProps = filterDOMProps(props, {
    labelable: true
  }); // Pass name prop from group to all items by attaching to the state.

  $c1125120fa765e2757aafb57a69ea88$export$checkboxGroupNames.set(state, name);
  return {
    groupProps: mergeProps(domProps, _babelRuntimeHelpersEsmExtends({
      role: 'group',
      'aria-disabled': isDisabled || undefined
    }, fieldProps)),
    labelProps
  };
}

/**
 * Provides the behavior and accessibility implementation for a checkbox component contained within a checkbox group.
 * Checkbox groups allow users to select multiple items from a list of options.
 * @param props - Props for the checkbox.
 * @param state - State for the checkbox, as returned by `useCheckboxGroupState`.
 * @param inputRef - A ref for the HTML input element.
 */
export function useCheckboxGroupItem(props, state, inputRef) {
  const toggleState = useToggleState({
    isReadOnly: props.isReadOnly || state.isReadOnly,
    isSelected: state.isSelected(props.value),

    onChange(isSelected) {
      if (isSelected) {
        state.addValue(props.value);
      } else {
        state.removeValue(props.value);
      }

      if (props.onChange) {
        props.onChange(isSelected);
      }
    }

  });
  let {
    inputProps
  } = useCheckbox(_babelRuntimeHelpersEsmExtends({}, props, {
    isReadOnly: props.isReadOnly || state.isReadOnly,
    isDisabled: props.isDisabled || state.isDisabled,
    name: props.name || $c1125120fa765e2757aafb57a69ea88$export$checkboxGroupNames.get(state)
  }), toggleState, inputRef);
  return {
    inputProps
  };
}
//# sourceMappingURL=module.js.map
